/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.util.Map;

/**
 * Annotation proxies that support writing to the annotation and other
 * extended features implement this. 
 */
public interface AnnotationEx {
	
	/**
	 * Set the default value for name. This is useful when the actual default
	 * is not the same as the default in the annotation declaration.
	 */
	public void setDefault(String name, Object value);
	
	/**
	 * Set the value name to value. If value is the same as the default or
	 * null then remove it. Note that if name is in the {@link #getErrorMap()}
	 * then this is a NOP.
	 */
	public void set(String name, Object value);
	
	/**
	 * Set the value name to value. If treatEmptyAsNull is true and the
	 * value is an empty String then it is considered null. If value is the 
	 * same as the default or null then remove it. Note that if name is in 
	 * the {@link #getErrorMap()} then this is a NOP.
	 */
	public void set(String name, Object value, boolean treatEmptyAsNull);
	
	/**
	 * Get the size of the array[] annotation value name. If the value has
	 * not been set then 0 is returned.
	 */
	public int getArraySize(String name);
	
	/**
	 * Set the size of the array[] annotation value name. Extra empty
	 * annotations will be created as needed. If size is less than the
	 * current size the extra existing annotations are deleted.  Note that 
	 * if name is in the {@link #getErrorMap()} then nothing is done and
	 * false is returned. Otherwise returns true.
	 */
	public boolean setArraySize(String name, int size);
	
	/**
	 * Create a new element for the array[] annotation value name. This will
	 * be added to the end of the array only wher at least one of its values 
	 * is set to something non-default.
	 */
	public AnnotationEx appendArrayElement(String name);

	/**
	 * Does the annotation have a value for name?
	 */
	public boolean hasValue(String name);
	
	/**
	 * How many values does this annotation have?
	 */
	public int getValueCount();
	
	/**
	 * Get the names of this annotations values or an empty array if none.
	 */
	public String[] getValueNames();
	
	/**
	 * Get the value for name. 
	 */
	public Object get(String name);	
	
	/**
	 * Get the value as a String. This is used for annotation values that
	 * have errors and so on.
	 */
	public String getAsString(String name);
	
	/**
	 * Get a Class value as a String. This returns the name of the class
	 * (no .class on the end or anything).
	 */
	public String getClassValue(String name);
	
	/**
	 * Return a map of value name to error message. This map will be null
	 * if there have been no errors. Note that errors are only discovered
	 * when annotation values are read.
	 */
	public Map<String, String> getErrorMap(); 
	
	/**
	 * Remove this annotation. If it does not actually existing anywhere
	 * this is a NOP.
	 */
	public void delete();
	
	/**
	 * If this is true and the annotation becomes empty then a marker
	 * annotation is created instead of removing the annotation. The default
	 * value of this propery is taken from 
	 * {@link AnnotationRegistry#isMarker(Class)}. If this is initially
	 * false and is changed to true and no annotation exists then a
	 * marker annotation is created immediately.
	 */
	public void setMarker(boolean on);
	
	/**
	 * Get information on the location of name in the source code or XML file
	 * or wherever it is located. If name is null or does not exist then the 
	 * location of the whole annotation is returned. If the annotation does
	 * not exist yet then the location of its potential owner is returned.
	 * Returns null if no location information is available.
	 */
	public Map getLocation(String name);
	
}
