# registry.test --
#
# This file contains a collection of tests for the registry command.
# Sourcing this file into Tcl runs the tests and generates output for
# errors.  No output means no errors were found.
#
# In order for these tests to run, the registry package must be on the
# auto_path or the registry package must have been loaded already.
#
# Copyright (c) 1997 by Sun Microsystems, Inc.  All rights reserved.
# Copyright (c) 1998-1999 by Scriptics Corporation.
#
# RCS: @(#) $Id: registry.tcl,v 1.1 2006/12/29 09:02:53 fourdman Exp $

if {[lsearch [namespace children] ::tcltest] == -1} {
    package require tcltest 2
    namespace import -force ::tcltest::*
}

if {$tcl_platform(platform) == "windows"} {
    if [catch {
	set lib [lindex [glob -directory [file join [pwd] [file dirname \
		[info nameofexecutable]]] tclreg*.dll] 0]
	load $lib registry
    }] {
	puts "Unable to find the registry package. Skipping registry tests."
	return
    }
}

# determine the current locale
testConstraint english [expr {[llength [info commands testlocale]]
	&& [string equal [testlocale all ""] "English_United States.1252"]
}]

set hostname [info hostname]

test registry-1.1 {argument parsing for registry command} {pcOnly} {
    list [catch {registry} msg] $msg
} {1 {wrong # args: should be "registry option ?arg arg ...?"}}
test registry-1.2 {argument parsing for registry command} {pcOnly} {
    list [catch {registry foo} msg] $msg
} {1 {bad option "foo": must be broadcast, delete, get, keys, set, type, or values}}

test registry-1.3 {argument parsing for registry command} {pcOnly} {
    list [catch {registry d} msg] $msg
} {1 {wrong # args: should be "registry delete keyName ?valueName?"}}
test registry-1.4 {argument parsing for registry command} {pcOnly} {
    list [catch {registry delete} msg] $msg
} {1 {wrong # args: should be "registry delete keyName ?valueName?"}}
test registry-1.5 {argument parsing for registry command} {pcOnly} {
    list [catch {registry delete foo bar baz} msg] $msg
} {1 {wrong # args: should be "registry delete keyName ?valueName?"}}

test registry-1.6 {argument parsing for registry command} {pcOnly} {
    list [catch {registry g} msg] $msg
} {1 {wrong # args: should be "registry get keyName valueName"}}
test registry-1.7 {argument parsing for registry command} {pcOnly} {
    list [catch {registry get} msg] $msg
} {1 {wrong # args: should be "registry get keyName valueName"}}
test registry-1.8 {argument parsing for registry command} {pcOnly} {
    list [catch {registry get foo} msg] $msg
} {1 {wrong # args: should be "registry get keyName valueName"}}
test registry-1.9 {argument parsing for registry command} {pcOnly} {
    list [catch {registry get foo bar baz} msg] $msg
} {1 {wrong # args: should be "registry get keyName valueName"}}

test registry-1.10 {argument parsing for registry command} {pcOnly} {
    list [catch {registry k} msg] $msg
} {1 {wrong # args: should be "registry keys keyName ?pattern?"}}
test registry-1.11 {argument parsing for registry command} {pcOnly} {
    list [catch {registry keys} msg] $msg
} {1 {wrong # args: should be "registry keys keyName ?pattern?"}}
test registry-1.12 {argument parsing for registry command} {pcOnly} {
    list [catch {registry keys foo bar baz} msg] $msg
} {1 {wrong # args: should be "registry keys keyName ?pattern?"}}

test registry-1.13 {argument parsing for registry command} {pcOnly} {
    list [catch {registry s} msg] $msg
} {1 {wrong # args: should be "registry set keyName ?valueName data ?type??"}}
test registry-1.14 {argument parsing for registry command} {pcOnly} {
    list [catch {registry set} msg] $msg
} {1 {wrong # args: should be "registry set keyName ?valueName data ?type??"}}
test registry-1.15 {argument parsing for registry command} {pcOnly} {
    list [catch {registry set foo bar} msg] $msg
} {1 {wrong # args: should be "registry set keyName ?valueName data ?type??"}}
test registry-1.16 {argument parsing for registry command} {pcOnly} {
    list [catch {registry set foo bar baz blat gorp} msg] $msg
} {1 {wrong # args: should be "registry set keyName ?valueName data ?type??"}}

test registry-1.17 {argument parsing for registry command} {pcOnly} {
    list [catch {registry t} msg] $msg
} {1 {wrong # args: should be "registry type keyName valueName"}}
test registry-1.18 {argument parsing for registry command} {pcOnly} {
    list [catch {registry type} msg] $msg
} {1 {wrong # args: should be "registry type keyName valueName"}}
test registry-1.19 {argument parsing for registry command} {pcOnly} {
    list [catch {registry type foo} msg] $msg
} {1 {wrong # args: should be "registry type keyName valueName"}}
test registry-1.20 {argument parsing for registry command} {pcOnly} {
    list [catch {registry type foo bar baz} msg] $msg
} {1 {wrong # args: should be "registry type keyName valueName"}}

test registry-1.21 {argument parsing for registry command} {pcOnly} {
    list [catch {registry v} msg] $msg
} {1 {wrong # args: should be "registry values keyName ?pattern?"}}
test registry-1.22 {argument parsing for registry command} {pcOnly} {
    list [catch {registry values} msg] $msg
} {1 {wrong # args: should be "registry values keyName ?pattern?"}}
test registry-1.23 {argument parsing for registry command} {pcOnly} {
    list [catch {registry values foo bar baz} msg] $msg
} {1 {wrong # args: should be "registry values keyName ?pattern?"}}

test registry-2.1 {DeleteKey: bad key} {pcOnly} {
    list [catch {registry delete foo} msg] $msg
} {1 {bad root name "foo": must be HKEY_LOCAL_MACHINE, HKEY_USERS, HKEY_CLASSES_ROOT, HKEY_CURRENT_USER, HKEY_CURRENT_CONFIG, HKEY_PERFORMANCE_DATA, or HKEY_DYN_DATA}}
test registry-2.2 {DeleteKey: bad key} {pcOnly} {
    list [catch {registry delete HKEY_CLASSES_ROOT} msg] $msg
} {1 {bad key: cannot delete root keys}}
test registry-2.3 {DeleteKey: bad key} {pcOnly} {
    list [catch {registry delete HKEY_CLASSES_ROOT\\} msg] $msg
} {1 {bad key: cannot delete root keys}}
test registry-2.4 {DeleteKey: subkey at root level} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry keys HKEY_CLASSES_ROOT TclFoobar
} {}
test registry-2.5 {DeleteKey: subkey below root level} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test
    registry delete HKEY_CLASSES_ROOT\\TclFoobar\\test
    set result [registry keys HKEY_CLASSES_ROOT TclFoobar\\test]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {}
test registry-2.6 {DeleteKey: recursive delete} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test1
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test2\\test3
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result [registry keys HKEY_CLASSES_ROOT TclFoobar]
    set result
} {}
test registry-2.7 {DeleteKey: trailing backslashes} {pcOnly english} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz
    list [catch {registry delete HKEY_CLASSES_ROOT\\TclFoobar\\} msg] $msg
} {1 {unable to delete key: The configuration registry key is invalid.}}
test registry-2.8 {DeleteKey: failure} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
} {}
test registry-2.9 {DeleteKey: unicode} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test\u00c7bar\\a
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test\u00c7bar\\b
    registry delete HKEY_CLASSES_ROOT\\TclFoobar\\test\u00c7bar
    set result [registry keys HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {}

test registry-3.1 {DeleteValue} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz test1 blort
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz test2 blat
    registry delete HKEY_CLASSES_ROOT\\TclFoobar\\baz test1
    set result [registry values HKEY_CLASSES_ROOT\\TclFoobar\\baz]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} test2
test registry-3.2 {DeleteValue: bad key} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry delete HKEY_CLASSES_ROOT\\TclFoobar test} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}
test registry-3.3 {DeleteValue: bad value} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz test2 blort
    set result [list [catch {registry delete HKEY_CLASSES_ROOT\\TclFoobar test1} msg] $msg]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {1 {unable to delete value "test1" from key "HKEY_CLASSES_ROOT\TclFoobar": The system cannot find the file specified.}}
test registry-3.4 {DeleteValue: Unicode} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\\u00c7baz \u00c7test1 blort
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\\u00c7baz test2 blat
    registry delete HKEY_CLASSES_ROOT\\TclFoobar\\\u00c7baz \u00c7test1
    set result [registry values HKEY_CLASSES_ROOT\\TclFoobar\\\u00c7baz]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} test2

test registry-4.1 {GetKeyNames: bad key} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry keys HKEY_CLASSES_ROOT\\TclFoobar} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}
test registry-4.2 {GetKeyNames} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz
    set result [registry keys HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {baz}
test registry-4.3 {GetKeyNames: remote key} {pcOnly nonPortable english} {
    registry set \\\\$hostname\\HKEY_CLASSES_ROOT\\TclFoobar\\baz
    set result [registry keys \\\\gaspode\\HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete \\\\$hostname\\HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {baz}
test registry-4.4 {GetKeyNames: empty key} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar
    set result [registry keys HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {}
test registry-4.5 {GetKeyNames: patterns} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\blat
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\foo
    set result [lsort [registry keys HKEY_CLASSES_ROOT\\TclFoobar b*]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {baz blat}
test registry-4.6 {GetKeyNames: names with spaces} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz\ bar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\blat
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\foo
    set result [lsort [registry keys HKEY_CLASSES_ROOT\\TclFoobar b*]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {{baz bar} blat}
test registry-4.7 {GetKeyNames: Unicode} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz\u00c7bar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\blat
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\foo
    set result [lsort [registry keys HKEY_CLASSES_ROOT\\TclFoobar b*]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} "baz\u00c7bar blat"
test registry-4.8 {GetKeyNames: Unicode} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz\u30b7bar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\blat
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\foo
    set result [lsort [registry keys HKEY_CLASSES_ROOT\\TclFoobar b*]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} "baz\u30b7bar blat"

test registry-5.1 {GetType} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry type HKEY_CLASSES_ROOT\\TclFoobar val1} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}
test registry-5.2 {GetType} {pcOnly english} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry type HKEY_CLASSES_ROOT\\TclFoobar val1} msg] $msg
} {1 {unable to get type of value "val1" from key "HKEY_CLASSES_ROOT\TclFoobar": The system cannot find the file specified.}}
test registry-5.3 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar none
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} none
test registry-5.4 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} sz
test registry-5.5 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar sz
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} sz
test registry-5.6 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar expand_sz
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} expand_sz
test registry-5.7 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 binary
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} binary
test registry-5.8 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 dword
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} dword
test registry-5.9 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 dword_big_endian
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} dword_big_endian
test registry-5.10 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 link
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} link
test registry-5.11 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar multi_sz
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} multi_sz
test registry-5.12 {GetType} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 resource_list
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} resource_list
test registry-5.13 {GetType: unknown types} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 24
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 24
test registry-5.14 {GetType: Unicode} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar va\u00c7l1 1 24
    set result [registry type HKEY_CLASSES_ROOT\\TclFoobar va\u00c7l1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 24

test registry-6.1 {GetValue} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry get HKEY_CLASSES_ROOT\\TclFoobar val1} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}
test registry-6.2 {GetValue} {pcOnly english} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry get HKEY_CLASSES_ROOT\\TclFoobar val1} msg] $msg
} {1 {unable to get value "val1" from key "HKEY_CLASSES_ROOT\TclFoobar": The system cannot find the file specified.}}
test registry-6.3 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar none
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} foobar
test registry-6.4 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} foobar
test registry-6.5 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} foobar
test registry-6.6 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar expand_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} foobar
test registry-6.7 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 binary
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 1
test registry-6.8 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 0x20 dword
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 32
test registry-6.9 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 0x20 dword_big_endian
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 32
test registry-6.10 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 link
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 1
test registry-6.11 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 foobar multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} foobar
test registry-6.12 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 {foo\ bar baz} multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {{foo bar} baz}
test registry-6.13 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 {} multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {}
test registry-6.14 {GetValue: truncation of multivalues with null elements} \
	{pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 {a {} b} multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} a
test registry-6.15 {GetValue} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 resource_list
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 1
test registry-6.16 {GetValue: unknown types} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 1 24
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} 1
test registry-6.17 {GetValue: Unicode value names} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val\u00c71 foobar multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val\u00c71]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} foobar
test registry-6.18 {GetValue: values with Unicode strings} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 {foo ba\u30b7r baz} multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} "foo ba\u30b7r baz"
test registry-6.19 {GetValue: values with Unicode strings} {pcOnly english} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 {foo ba\u00c7r baz} multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} "foo ba\u00c7r baz"
test registry-6.20 {GetValue: values with Unicode strings with embedded nulls} {pcOnly} {
    registry set HKEY_CLASSES_ROOT\\TclFoobar val1 {foo ba\u0000r baz} multi_sz
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar val1]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} "foo ba r baz"

test registry-7.1 {GetValueNames: bad key} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry values HKEY_CLASSES_ROOT\\TclFoobar} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}
test registry-7.2 {GetValueNames} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar baz foobar
    set result [registry values HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} baz
test registry-7.3 {GetValueNames} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar baz foobar1
    registry set HKEY_CLASSES_ROOT\\TclFoobar blat foobar2
    registry set HKEY_CLASSES_ROOT\\TclFoobar {} foobar3
    set result [lsort [registry values HKEY_CLASSES_ROOT\\TclFoobar]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {{} baz blat}
test registry-7.4 {GetValueNames: remote key} {pcOnly nonPortable english} {
    registry set \\\\$hostname\\HKEY_CLASSES_ROOT\\TclFoobar baz blat
    set result [registry values \\\\$hostname\\HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete \\\\$hostname\\HKEY_CLASSES_ROOT\\TclFoobar
    set result
} baz
test registry-7.5 {GetValueNames: empty key} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar
    set result [registry values HKEY_CLASSES_ROOT\\TclFoobar]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {}
test registry-7.6 {GetValueNames: patterns} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar baz foobar1
    registry set HKEY_CLASSES_ROOT\\TclFoobar blat foobar2
    registry set HKEY_CLASSES_ROOT\\TclFoobar foo foobar3
    set result [lsort [registry values HKEY_CLASSES_ROOT\\TclFoobar b*]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {baz blat}
test registry-7.7 {GetValueNames: names with spaces} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar baz\ bar foobar1
    registry set HKEY_CLASSES_ROOT\\TclFoobar blat foobar2
    registry set HKEY_CLASSES_ROOT\\TclFoobar foo foobar3
    set result [lsort [registry values HKEY_CLASSES_ROOT\\TclFoobar b*]]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {{baz bar} blat}

test registry-8.1 {OpenSubKey} {pcOnly nonPortable english} {
    # This test will only succeed if the current user does not have registry
    # access on the specified machine.
    list [catch {registry keys {\\mom\HKEY_LOCAL_MACHINE}} msg] $msg
} {1 {unable to open key: Access is denied.}}
test registry-8.2 {OpenSubKey} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar
    set result [registry keys HKEY_CLASSES_ROOT TclFoobar]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} TclFoobar
test registry-8.3 {OpenSubKey} {pcOnly english} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    list [catch {registry keys HKEY_CLASSES_ROOT\\TclFoobar} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}

test registry-9.1 {ParseKeyName: bad keys} {pcOnly} {
    list [catch {registry values \\} msg] $msg
} "1 {bad key \"\\\": must start with a valid root}"
test registry-9.2 {ParseKeyName: bad keys} {pcOnly} {
    list [catch {registry values \\foobar} msg] $msg
} {1 {bad key "\foobar": must start with a valid root}}
test registry-9.3 {ParseKeyName: bad keys} {pcOnly} {
    list [catch {registry values \\\\} msg] $msg
} {1 {bad root name "": must be HKEY_LOCAL_MACHINE, HKEY_USERS, HKEY_CLASSES_ROOT, HKEY_CURRENT_USER, HKEY_CURRENT_CONFIG, HKEY_PERFORMANCE_DATA, or HKEY_DYN_DATA}}
test registry-9.4 {ParseKeyName: bad keys} {pcOnly} {
    list [catch {registry values \\\\\\} msg] $msg
} {1 {bad root name "": must be HKEY_LOCAL_MACHINE, HKEY_USERS, HKEY_CLASSES_ROOT, HKEY_CURRENT_USER, HKEY_CURRENT_CONFIG, HKEY_PERFORMANCE_DATA, or HKEY_DYN_DATA}}
test registry-9.5 {ParseKeyName: bad keys} {pcOnly english} {
    list [catch {registry values \\\\\\HKEY_CLASSES_ROOT} msg] $msg
} {1 {unable to open key: The network address is invalid.}}
test registry-9.6 {ParseKeyName: bad keys} {pcOnly} {
    list [catch {registry values \\\\gaspode} msg] $msg
} {1 {bad root name "": must be HKEY_LOCAL_MACHINE, HKEY_USERS, HKEY_CLASSES_ROOT, HKEY_CURRENT_USER, HKEY_CURRENT_CONFIG, HKEY_PERFORMANCE_DATA, or HKEY_DYN_DATA}}
test registry-9.7 {ParseKeyName: bad keys} {pcOnly} {
    list [catch {registry values foobar} msg] $msg
} {1 {bad root name "foobar": must be HKEY_LOCAL_MACHINE, HKEY_USERS, HKEY_CLASSES_ROOT, HKEY_CURRENT_USER, HKEY_CURRENT_CONFIG, HKEY_PERFORMANCE_DATA, or HKEY_DYN_DATA}}
test registry-9.8 {ParseKeyName: null keys} {pcOnly} {
    list [catch {registry delete HKEY_CLASSES_ROOT\\} msg] $msg
} {1 {bad key: cannot delete root keys}}
test registry-9.9 {ParseKeyName: null keys} {pcOnly english} {
    list [catch {registry keys HKEY_CLASSES_ROOT\\TclFoobar\\baz} msg] $msg
} {1 {unable to open key: The system cannot find the file specified.}}

test registry-10.1 {RecursiveDeleteKey} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test1
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test2\\test3
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result [registry keys HKEY_CLASSES_ROOT TclFoobar]
    set result
} {}
test registry-10.2 {RecursiveDeleteKey} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test1
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\test2\\test3
    set result [registry delete HKEY_CLASSES_ROOT\\TclFoobar\\test2\\test4]
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    set result
} {}

test registry-11.1 {SetValue: recursive creation} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz blat foobar
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar\\baz blat]
} foobar
test registry-11.2 {SetValue: modification} {pcOnly} {
    registry delete HKEY_CLASSES_ROOT\\TclFoobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz blat foobar
    registry set HKEY_CLASSES_ROOT\\TclFoobar\\baz blat frob
    set result [registry get HKEY_CLASSES_ROOT\\TclFoobar\\baz blat]
} frob
test registry-11.3 {SetValue: failure} {pcOnly nonPortable english} {
    # This test will only succeed if the current user does not have registry
    # access on the specified machine.
    list [catch {registry set {\\mom\HKEY_CLASSES_ROOT\TclFoobar} bar foobar} msg] $msg
} {1 {unable to open key: Access is denied.}}

test registry-12.1 {BroadcastValue} {pcOnly} {
    list [catch {registry broadcast} msg] $msg
} {1 {wrong # args: should be "registry broadcast keyName ?-timeout millisecs?"}}
test registry-12.2 {BroadcastValue} {pcOnly} {
    list [catch {registry broadcast "" -time} msg] $msg
} {1 {wrong # args: should be "registry broadcast keyName ?-timeout millisecs?"}}
test registry-12.3 {BroadcastValue} {pcOnly} {
    list [catch {registry broadcast "" - 500} msg] $msg
} {1 {wrong # args: should be "registry broadcast keyName ?-timeout millisecs?"}}
test registry-12.4 {BroadcastValue} {pcOnly} {
    list [catch {registry broadcast {Environment}} msg] $msg
} {0 {1 0}}
test registry-12.5 {BroadcastValue} {pcOnly} {
    list [catch {registry b {}} msg] $msg
} {0 {1 0}}

# cleanup
unset hostname
::tcltest::cleanupTests
return
