/*******************************************************************************
 * Copyright (c) 2009, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.utility.model.listener;

import org.eclipse.persistence.tools.utility.ObjectTools;
import org.eclipse.persistence.tools.utility.model.event.ChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionAddEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionClearEvent;
import org.eclipse.persistence.tools.utility.model.event.CollectionRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListAddEvent;
import org.eclipse.persistence.tools.utility.model.event.ListChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.ListClearEvent;
import org.eclipse.persistence.tools.utility.model.event.ListMoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListRemoveEvent;
import org.eclipse.persistence.tools.utility.model.event.ListReplaceEvent;
import org.eclipse.persistence.tools.utility.model.event.PropertyChangeEvent;
import org.eclipse.persistence.tools.utility.model.event.StateChangeEvent;

/**
 * Convenience abstract implementation of {@link ChangeListener}.
 * All change notifications are funneled through a single method.
 * This class can be used by
 * subclassing it and overriding either {@link #modelChanged(ChangeEvent)}
 * (if access to the event is required) or {@link #modelChanged()} (if access
 * to the event is not required).
 * <p>
 * Provisional API: This class is part of an interim API that is still
 * under development and expected to change significantly before reaching
 * stability. It is available at this early stage to solicit feedback from
 * pioneering adopters on the understanding that any code that uses this API
 * will almost certainly be broken (repeatedly) as the API evolves.
 */
@SuppressWarnings("nls")
public abstract class AbstractChangeListener
	implements ChangeListener
{
	public AbstractChangeListener() {
		super();
	}

	@Override
	public void stateChanged(StateChangeEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void propertyChanged(PropertyChangeEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void collectionChanged(CollectionChangeEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void collectionCleared(CollectionClearEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void itemsAdded(CollectionAddEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void itemsRemoved(CollectionRemoveEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void itemsAdded(ListAddEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void itemsMoved(ListMoveEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void itemsRemoved(ListRemoveEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void itemsReplaced(ListReplaceEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void listChanged(ListChangeEvent event) {
		this.modelChanged(event);
	}

	@Override
	public void listCleared(ListClearEvent event) {
		this.modelChanged(event);
	}

	/**
	 * The model has notified the listener of the change described by the
	 * specified change event. By default the listener executes {@link #modelChanged()}.
	 */
	protected void modelChanged(@SuppressWarnings("unused") ChangeEvent event) {
		this.modelChanged();
	}

	/**
	 * The model has notified the listener of a change.
	 * By default the listener throws an exception.
	 */
	protected void modelChanged() {
		throw new RuntimeException("This method was not overridden.");
	}

	@Override
	public String toString() {
		return ObjectTools.toString(this);
	}
}