/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.persistence;

import java.util.List;
import java.util.ListIterator;
import javax.persistence.ValidationMode;
import org.eclipse.persistence.tools.mapping.ExternalPropertyHolder;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * Interface defining one persistence unit in the persistence.xml file
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalPersistenceUnit extends ExternalPropertyHolder {

	/**
	 * The element name of the property for the default cache type.
	 */
	String CACHE_TYPE = "cache-type";

	/**
	 * The element name of the child text nodes for the listed entity class names.
	 */
	String CLASS = "class";

	/**
	 * The element name of the child text node for the description.
	 */
	String DESCRIPTION = "description";

	/**
	 * The element name of the child boolean node for the exclude unlisted classes flag.
	 */
	String EXCLUDE_UNLISTED_CLASSES = "exclude-unlisted-classes";

	/**
	 * The element name of the child text nodes for the Java archive files.
	 */
	String JAR_FILE = "jar-file";

	/**
	 * The element name of the child text node for the JTA data source.
	 */
	String JTA_DATA_SOURCE = "jta-data-source";

	/**
	 * The element name of the child text nodes for the JPA mapping files.
	 */
	String MAPPING_FILE = "mapping-file";

	/**
	 * The element name of the property for the persistence unit name.
	 */
	String NAME = "name";

	/**
	 * The element name of the child text node for the non-JTA data source.
	 */
	String NON_JTA_DATA_SOURCE = "non-jta-data-source";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String PERSISTENCE_UNIT = "persistence-unit";

	/**
	 * The element name of the child text node for the provider.
	 */
	String PROVIDER = "provider";

	/**
	 * The element name of the property for the default transaction type.
	 */
	String TRANSACTION_TYPE = "transaction-type";

	/**
	 * The element name of the property for the default validation mode type.
	 */
	String VALIDATION_MODE = "validation-mode";

	/**
	 * Returns the cache type for this persistence unit.
	 */
	CacheType getCachType();

	/**
	 * Returns the {@link TextRange} for the cache type value.
	 */
	TextRange getCachTypeTextRange();

	/**
	 * Returns a text description of the persistence unit.
	 */
	String getDescription();

	/**
	 * Returns the {@link TextRange} for the given included entity class.
	 */
	TextRange getIncludedEntityClassTextRange(String className);

	@Deprecated
	int getIndex();

	/**
	 * Returns the {@link TextRange} for the given jar file path.
	 */
	TextRange getJarFilePathTextRange(int index, String path);

	/**
	 * Returns the class name of the JDBC driver used, fully qualified by its package name.
	 */
	String getJdbcDriver();

	/**
	 * Returns the {@link TextRange} for the JDBC class name value.
	 */
	TextRange getJdbcDriverTextRange();

	/**
	 * Returns the password used to connect to the data source via a JDBC connection.
	 */
	String getJdbcPassword();

	/**
	 * Returns the {@link TextRange} for the password.
	 */
	TextRange getJdbcPasswordTextRange();

	/**
	 * Returns the JDBC connection URL required by the JDBC driver, which is used to connect to the
	 * data source.
	 */
	String getJdbcUrl();

	/**
	 * Returns the {@link TextRange} for the JDBC connection URL.
	 */
	TextRange getJdbcUrlTextRange();

	/**
	 * Returns the user name used to connect to the data source via a JDBC connection.
	 */
	String getJdbcUser();

	/**
	 * Returns the {@link TextRange} for the user name value.
	 */
	TextRange getJdbcUserTextRange();

	/**
	 * Returns the JTA data source name for the persistence unit.
	 */
	String getJtaDatasource();

	/**
	 * Returns the {@link TextRange} for the JTA data source name.
	 */
	TextRange getJtaDatasourceTextRange();

	/**
	 * Returns the {@link TextRange} of the given class name entry.
	 */
	TextRange getMappedClassTextRange(String name);

	/**
	 * Returns the {@link TextRange} for the given relative path (which is the ORM configuration
	 * relative path entry).
	 */
	TextRange getMappingFilePathTextRange(int index, String path);

	/**
	 * Returns the name of the persistence unit.
	 */
	String getName();

	/**
	 * Returns the {@link TextRange} of the name property.
	 */
	TextRange getNameTextRange();

	/**
	 * Returns the non-JTA data source name for the persistence unit.
	 */
	String getNonJtaDatasource();

	/**
	 * Returns the {@link TextRange} for the non-JTA data source name value.
	 */
	TextRange getNonJtaDatasourceTextRange();

	/**
	 * Returns the pessimistic lock timeout in milliseconds for the persistence unit.
	 */
	Integer getPessimisticLockTimeout();

	/**
	 * Returns the {@link TextRange} for the pessimistic lock timeout value.
	 */
	TextRange getPessimisticLockTimeoutTextRange();

	/**
	 * Returns the class name of the persistence provider for this persistence unit.
	 */
	String getProvider();

	/**
	 * Returns the {@link TextRange} for the class name of the persistence provider.
	 */
	TextRange getProviderTextRange();

	/**
	 * Returns the query timeout in milliseconds for this persistence unit.
	 */
	Integer getQueryTimeout();

	/**
	 * Returns the {@link TextRange} for the query timeout.
	 */
	TextRange getQueryTimeoutTextRange();

	/**
	 * Returns the transaction type for this persistence unit.
	 */
	TransactionType getTransactionType();

	/**
	 * Returns the {@link TextRange} for the transaction type.
	 */
	TextRange getTransactionTypeTextRange();

	/**
	 * Returns the validation mode for this persistence unit.
	 */
	ValidationMode getValidationMode();

	/**
	 * Returns the {@link TextRange} for the validation mode.
	 */
	TextRange getValidationModeTextRange();

	/**
	 * Returns a collection of class names that should be included in the persistence unit.
	 */
	List<String> includedClasses();

	/**
	 * Returns the count of all the class names that should be included in the persistence unit.
	 */
	int includedClassesSize();

	/**
	 * Returns a collection of jar files that should be included in the persistence unit.
	 */
	List<String> jarFiles();

	/**
	 * Returns the count of all the Jar file entries.
	 */
	int jarFilesSize();

	/**
	 * Returns a collection of mapping files, these are orm.xml files that should be included in the persistence unit.
	 */
	List<String> mappingFiles();

	/**
	 * Returns the count of all the JPA Mapping file entries.
	 */
	int mappingFilesSize();

	/**
	 * Sets the cache type for this persistence unit.
	 */
	void setCacheType(CacheType type);

	/**
	 * Sets the description for the persistence unit.
	 */
	void setDescription(String description);

	/**
	 * Sets whether the persistence unit should exclude annotated classes in root of this persistence unit.
	 */
	void setExcludeUnlistedClasses(boolean excludeUnlistedClasses);

	/**
	 * Adds a included class name.
	 */
	void setIncludedClasses(ListIterator<String> classes);

	/**
	 * Updates the list of Jar file.
	 */
	void setJarFiles(ListIterator<String> jarFiles);

	/**
	 * Sets the JTA data source name for the persistence unit.
	 */
	void setJtaDatasource(String name);

	/**
	 * Adds a mapping file.
	 */
	void setMappingFiles(ListIterator<String> mappingFiles);

	/**
	 * Sets the name of the persistence unit.
	 */
	void setName(String name);

	/**
	 * Sets the non-JTA data source name for the persistence unit.
	 */
	void setNonJtaDatasource(String name);

	/**
	 * Sets the pessimistic lock timeout in milliseconds for this persistence unit.
	 */
	void setPessimisticLockTimeout(Integer length);

	/**
	 * Sets the class name of the persistence provider for this persistence unit.
	 */
	void setProvider(String name);

	/**
	 * Sets the query timeout in milliseconds for this persistence unit.
	 */
	void setQueryTimeout(Integer length);

	/**
	 * Sets the transaction type for this persistence unit.
	 */
	void setTransactionType(TransactionType type);

	/**
	 * Sets the validation mode type for this persistence unit.
	 */
	void setValidationMode(ValidationMode type);

	/**
	 * Returns whether the persistence unit should exclude annotated classes in
	 * root of the persistence unit
	 */
	boolean shouldExcludeUnlistedClasses();

	/**
	 * Enumeration of shared cache types for a persistence unit.
	 */
	public enum CacheType {
		ALL,
		DISABLE_SELECTIVE,
		ENABLE_SELECTIVE,
		NONE,
		UNSPECIFIED
	}

	/**
	 * Enumeration of the possible persistence types for a persistence unit.
	 */
	public enum TransactionType {
		JTA,
		RESOURCE_LOCAL
	}
}