/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalTableGenerator;
import org.eclipse.persistence.tools.mapping.orm.ExternalUniqueConstraint;
import org.eclipse.persistence.tools.utility.TextRange;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * The external form of a table generator.
 *
 * @see BasicMapping
 * @see Entity
 * @see IdMapping
 * @see ORMConfiguration
 *
 * @version 2.6
 */
final class TableGenerator extends PrimaryKeyGenerator
                           implements ExternalTableGenerator {

	/**
	 * Creates a new <code>TableGenerator</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	TableGenerator(AbstractExternalForm parent, int index) {
		super(parent, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalUniqueConstraint addUniqueConstraint(ListIterable<String> columnNames) {
		UniqueConstraint uniqueConstraint = buildUniqueConstraint(-1);
		uniqueConstraint.addSelf();
		uniqueConstraint.setColumnNames(columnNames);
		return uniqueConstraint;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(NAME);
		names.add(TABLE);
		names.add(CATALOG);
		names.add(SCHEMA);
		names.add(PK_COLUMN_NAME);
		names.add(VALUE_COLUMN_NAME);
		names.add(PK_COLUMN_VALUE);
		names.add(INITIAL_VALUE);
		names.add(ALLOCATION_SIZE);
		return names;
	}

	private UniqueConstraint buildUniqueConstraint(int index) {
		return new UniqueConstraint(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return TABLE_GENERATOR;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPKColumnName() {
		return getAttribute(PK_COLUMN_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPKColumnNameTextRange() {
		return getAttributeTextRange(PK_COLUMN_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPKColumnValue() {
		return getAttribute(PK_COLUMN_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPKColumnValueTextRange() {
		return getAttributeTextRange(PK_COLUMN_VALUE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getTableName() {
		return getAttribute(TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTableNameTextRange() {
		return getAttributeTextRange(TABLE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalUniqueConstraint getUniqueConstraint(int index) {

		if (hasChild(UniqueConstraint.UNIQUE_CONSTRAINT, index)) {
			return buildUniqueConstraint(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getValueColumnName() {
		return getAttribute(VALUE_COLUMN_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getValueColumnNameTextRange() {
		return getAttributeTextRange(VALUE_COLUMN_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeUniqueConstraint(int index) {

		UniqueConstraint uniqueConstraint = buildUniqueConstraint(index);
		uniqueConstraint.removeSelf();

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPKColumnName(String name) {

		setAttribute(PK_COLUMN_NAME, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPKColumnValue(String value) {

		setAttribute(PK_COLUMN_VALUE, value);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setTableName(String name) {

		setAttribute(TABLE, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setValueColumnName(String name) {

		setAttribute(VALUE_COLUMN_NAME, name);

		if (shouldBeRemoved()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalUniqueConstraint> uniqueConstraints() {

		int count = uniqueConstraintsSize();
		List<ExternalUniqueConstraint> uniqueConstraints = new ArrayList<ExternalUniqueConstraint>(count);

		for (int index = 0; index < count; index++) {
			uniqueConstraints.add(buildUniqueConstraint(index));
		}

		return uniqueConstraints;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int uniqueConstraintsSize() {
		return getChildrenSize(UniqueConstraint.UNIQUE_CONSTRAINT);
	}
}