/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.AccessType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAccessType;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.eclipse.persistence.tools.mapping.orm.ExternalPersistenceUnit;
import org.eclipse.persistence.tools.mapping.orm.ExternalTenantDiscriminatorColumn;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of a persistence unit.
 *
 * @see ORMConfiguration
 *
 * @version 2.6
 */
final class PersistenceUnit extends AbstractExternalForm
                            implements ExternalPersistenceUnit {

	/**
	 * Creates a new <code>PersistenceUnit</code>.
	 *
	 * @param parent The parent of this external form
	 */
	PersistenceUnit(ORMConfiguration parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantDiscriminatorColumn addDiscriminatorColumn() {
		PersistenceUnitTenantDiscriminatorColumn column = buildDiscriminatorColumn(-1);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityListener addEntityListener(String className) {
		PersistenceUnitEntityListener entityListener = buildEntityListener(-1);
		entityListener.addSelf();
		entityListener.setClassName(className);
		return entityListener;
	}

	private PersistenceUnitTenantDiscriminatorColumn buildDiscriminatorColumn(int index) {
		return new PersistenceUnitTenantDiscriminatorColumn(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {

		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(XML_MAPPING_METADATA_COMPLETE);
		names.add(EXCLUDE_DEFAULT_MAPPINGS);
		names.add(PERSISTENCE_UNIT_DEFAULTS);
		return names;
	}

	private PersistenceUnitEntityListener buildEntityListener(int index) {
		return new PersistenceUnitEntityListener(this, index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalTenantDiscriminatorColumn> discriminatorColumns() {

		int count = discriminatorColumnSize();
		List<ExternalTenantDiscriminatorColumn> columns = new ArrayList<ExternalTenantDiscriminatorColumn>(count);

		for (int index = 0; index < count; index++) {
			columns.add(buildDiscriminatorColumn(index));
		}

		return columns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int discriminatorColumnSize() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildrenSize(element, AbstractTenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN);
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalEntityListener> entityListeners() {

		int count = entityListenersSize();
		List<ExternalEntityListener> entityListeners = new ArrayList<ExternalEntityListener>(count);

		for (int index = 0; index < count; index++) {
			entityListeners.add(buildEntityListener(index));
		}

		return entityListeners;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int entityListenersSize() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return 0;
		}

		element = getChild(element, AbstractEntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return 0;
		}

		return getChildrenSize(element, AbstractEntityListener.ENTITY_LISTENER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getCascadePersistByDefaultTextRange() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return null;
		}

		element = getChild(element, CASCADE_PERSIST);

		if (element == null) {
			return null;
		}

		return getTextRange(element);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public AccessType getDefaultAccessType() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildEnumNode(element, ExternalAccessType.ACCESS, AccessType.class);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDefaultAccessTypeTextRange() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNodeTextRange(element, ExternalAccessType.ACCESS);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultCatalogName() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNode(element, CATALOG);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDefaultCatalogNameTextRange() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNodeTextRange(element, CATALOG);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultDescription() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNode(element, DESCRIPTION);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDefaultSchemaName() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNode(element, SCHEMA);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDefaultSchemaNameTextRange() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return getChildTextNodeTextRange(element, SCHEMA);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTenantDiscriminatorColumn getDiscriminatorColumn(int index) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return null;
		}

		element = getChild(element, AbstractTenantDiscriminatorColumn.TENANT_DISCRIMINATOR_COLUMN, index);

		if (element == null) {
			return null;
		}

		return buildDiscriminatorColumn(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return PERSISTENCE_UNIT_METADATA;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityListener getEntityListener(int index) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element == null) {
			return null;
		}

		element = getChild(element, AbstractEntityListener.ENTITY_LISTENERS);

		if (element == null) {
			return null;
		}

		element = getChild(element, AbstractEntityListener.ENTITY_LISTENER, index);

		if (element == null) {
			return null;
		}

		return buildEntityListener(index);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDefault() {
		return getAllChildrenSize() == 0;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isDelimitedIdentifers() {
		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);
		return (element != null) && hasChild(element, DELIMITED_IDENTIFIERS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isExcludeDefaultMappings() {
		return hasChild(EXCLUDE_DEFAULT_MAPPINGS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isMappingMetaDataComplete() {
		return hasChild(XML_MAPPING_METADATA_COMPLETE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeDiscriminatorColumn(int index) {

		AbstractTenantDiscriminatorColumn entityListener = buildDiscriminatorColumn(index);
		entityListener.removeSelf();

		Element parentElement = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((parentElement != null) && !hasAnyChildren(parentElement)) {
			remove(parentElement);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeEntityListener(int index) {

		AbstractEntityListener entityListener = buildEntityListener(index);
		entityListener.removeSelf();

		Element parentElement = getChild(PERSISTENCE_UNIT_DEFAULTS);
		Element element = getChild(parentElement, AbstractEntityListener.ENTITY_LISTENERS);

		if ((parentElement != null) && (element != null) && !hasAnyChildren(element)) {
			remove(parentElement, element);
		}

		if ((parentElement != null) && !hasAnyChildren(parentElement)) {
			remove(parentElement);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultAccessType(AccessType type) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (type == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, ExternalAccessType.ACCESS, type);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultCatalogName(String catalogName) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (catalogName == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, CATALOG, catalogName);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultDescription(String description) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (description == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, DESCRIPTION, description);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDefaultSchemaName(String schemaName) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && (schemaName == null)) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		updateTextNode(element, SCHEMA, schemaName);

		if (!hasAnyChildren(element)) {
			remove(element);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDelimitedIdentifers(boolean delimitIdentifiers) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (delimitIdentifiers) {
			if (element == null) {
				element = addChild(PERSISTENCE_UNIT_DEFAULTS);
			}
			addChild(element, DELIMITED_IDENTIFIERS);
		}
		else {
			if (element != null) {
				removeChild(element, DELIMITED_IDENTIFIERS);

				if (!hasAnyChildren(element)) {
					remove(element);

					if (!hasAnyChildren()) {
						removeSelf();
					}
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {

		if (description == null) {
			removeChild(DESCRIPTION);

			if (!hasAnyChildren()) {
				removeSelf();
			}
		}
		else {
			Element element = getChild(DESCRIPTION);

			if (element == null) {
				addChild(DESCRIPTION);
			}

			updateChildTextNode(DESCRIPTION, description);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExcludeDefaultMappings(boolean excludeDefaultMappings) {

		if (excludeDefaultMappings) {
			addChild(EXCLUDE_DEFAULT_MAPPINGS);
		}
		else {
			removeChild(EXCLUDE_DEFAULT_MAPPINGS);

			if (!hasAnyChildren()) {
				removeSelf();

				if (!hasAnyChildren()) {
					removeSelf();
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMappingMetaDataComplete(boolean metaDataComplete) {

		if (metaDataComplete) {
			addChild(XML_MAPPING_METADATA_COMPLETE);
		}
		else {
			removeChild(XML_MAPPING_METADATA_COMPLETE);

			if (!hasAnyChildren()) {
				removeSelf();

				if (!hasAnyChildren()) {
					removeSelf();
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setUsesCascadePersistByDefault(boolean cascadePersistByDefault) {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if ((element == null) && !cascadePersistByDefault) {
			return;
		}

		if (element == null) {
			element = addChild(PERSISTENCE_UNIT_DEFAULTS);
		}

		if (cascadePersistByDefault) {
			addChild(element, CASCADE_PERSIST);
		}
		else {
			removeChild(element, CASCADE_PERSIST);

			if (!hasAnyChildren(element)) {
				remove(element);

				if (!hasAnyChildren()) {
					removeSelf();
				}
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean usesCascadePersistByDefault() {

		Element element = getChild(PERSISTENCE_UNIT_DEFAULTS);

		if (element != null) {
			return hasChild(element, CASCADE_PERSIST);
		}

		return false;
	}
}