/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import javax.persistence.EnumType;
import javax.persistence.TemporalType;
import org.eclipse.persistence.tools.mapping.orm.ExternalAssociationOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalAttributeOverride;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalJoinColumn;
import org.eclipse.persistence.tools.mapping.orm.ExternalObjectCollectionMapping;
import org.eclipse.persistence.tools.mapping.orm.ExternalOrderColumn;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form for a convertible mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.6
 */
abstract class ObjectCollectionMapping extends RelationshipMapping
                                       implements ExternalObjectCollectionMapping {

	/**
	 * Creates a new <code>ObjectCollectionMapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	ObjectCollectionMapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride addMapKeyAssociationOverride() {
		AssociationOverride associationOverride = buildMapKeyAssociationOverride(-1);
		associationOverride.addSelf();
		return associationOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride addMapKeyAttributeOverride() {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(-1);
		attributeOverride.addSelf();
		return attributeOverride;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn addMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.addSelf();
		return column;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn addMapKeyJoinColumn(String name) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, -1);
		joinColumn.addSelf();
		joinColumn.setName(name);
		return joinColumn;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalOrderColumn addOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.addSelf();
		return orderColumn;
	}

	private EntityColumn buildColumn(String elementName) {
		return new EntityColumn(this, elementName, -1);
	}

	final JoinColumn buildJoinColumn(String elementName, int index) {
		return new JoinColumn(this, elementName, index);
	}

	private AssociationOverride buildMapKeyAssociationOverride(int index) {
		return new AssociationOverride(this, index);
	}

	private AttributeOverride buildMapKeyAttributeOverride(int index) {
		return new AttributeOverride(this, index);
	}

	private OrderColumn buildOrderColumn() {
		return new OrderColumn(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKey() {
		return getChildAttribute(MAP_KEY, NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAssociationOverride getMapKeyAssociationOverride(int index) {

		if (hasChild(MAP_KEY_ASSOCIATION_OVERRIDE, index)) {
			return buildMapKeyAssociationOverride(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalAttributeOverride getMapKeyAttributeOverride(int index) {

		if (hasChild(MAP_KEY_ATTRIBUTE_OVERRIDE, index)) {
			return buildMapKeyAttributeOverride(index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyClassName() {
		return getChildAttribute(MAP_KEY_CLASS, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyClassNameTextRange() {
		return getChildAttributeTextRange(MAP_KEY_CLASS, CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalEntityColumn getMapKeyColumn() {

		if (hasChild(MAP_KEY_COLUMN)) {
			return buildColumn(MAP_KEY_COLUMN);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getMapKeyConverterName() {
		return getChildTextNode(MAP_KEY_CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyConverterNameTextRange() {
		return getChildTextNodeTextRange(MAP_KEY_CONVERT);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EnumType getMapKeyEnumeratedType() {
		return getChildEnumNode(MAP_KEY_ENUMERATED, EnumType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyEnumeratedTypeTextRange() {
		return getChildTextNodeTextRange(MAP_KEY_ENUMERATED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalJoinColumn getMapKeyJoinColumn(int index) {

		if (hasChild(MAP_KEY_JOIN_COLUMN, index)) {
			return buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TemporalType getMapKeyTemporalType() {
		return getChildEnumNode(MAP_KEY_TEMPORAL, TemporalType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyTemporalTypeTextRange() {
		return getChildTextNodeTextRange(MAP_KEY_TEMPORAL);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getMapKeyTextRange() {
		return getChildAttributeTextRange(MAP_KEY, NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getOrderByFieldName() {
		return getChildTextNode(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getOrderByFieldNameTextRange() {
		return getChildTextNodeTextRange(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalOrderColumn getOrderColumn() {

		if (hasChild(OrderColumn.ORDER_COLUMN)) {
			return buildOrderColumn();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasOrderByClause() {
		return hasChild(ORDER_BY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalAssociationOverride> mapKeyAssociationOverrides() {

		int count = mapKeyAssociationOverridesSize();
		List<ExternalAssociationOverride> associationOverrides = new ArrayList<ExternalAssociationOverride>(count);

		for (int index = 0; index < count; index++) {
			associationOverrides.add(buildMapKeyAssociationOverride(index));
		}

		return associationOverrides;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAssociationOverridesSize() {
		return getChildrenSize(MAP_KEY_ASSOCIATION_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalAttributeOverride> mapKeyAttributeOverrides() {

		int count = mapKeyAttributeOverridesSize();
		List<ExternalAttributeOverride> attributeOverrides = new ArrayList<ExternalAttributeOverride>(count);

		for (int index = 0; index < count; index++) {
			attributeOverrides.add(buildMapKeyAttributeOverride(index));
		}

		return attributeOverrides;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyAttributeOverridesSize() {
		return getChildrenSize(MAP_KEY_ATTRIBUTE_OVERRIDE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public List<ExternalJoinColumn> mapKeyJoinColumns() {

		int count = mapKeyJoinColumnsSize();
		List<ExternalJoinColumn> joinColumns = new ArrayList<ExternalJoinColumn>(count);

		for (int index = 0; index < count; index++) {
			joinColumns.add(buildJoinColumn(MAP_KEY_JOIN_COLUMN, index));
		}

		return joinColumns;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int mapKeyJoinColumnsSize() {
		return getChildrenSize(MAP_KEY_JOIN_COLUMN);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAssociationOverride(int index) {
		AssociationOverride joinColumn = buildMapKeyAssociationOverride(index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyAttributeOverride(int index) {
		AttributeOverride attributeOverride = buildMapKeyAttributeOverride(index);
		attributeOverride.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyColumn() {
		EntityColumn column = buildColumn(MAP_KEY_COLUMN);
		column.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeMapKeyJoinColumn(int index) {
		JoinColumn joinColumn = buildJoinColumn(MAP_KEY_JOIN_COLUMN, index);
		joinColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeOrderColumn() {
		OrderColumn orderColumn = buildOrderColumn();
		orderColumn.removeSelf();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKey(String mapKey) {

		if (mapKey == null) {
			removeChild(MAP_KEY);
		}
		else {
			Element element = getChild(MAP_KEY);

			if (element == null) {
				element = addChild(MAP_KEY);
			}

			setAttribute(element, NAME, mapKey);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyClassName(String className) {

		if (className == null) {
			removeChild(MAP_KEY_CLASS);
		}
		else {
			Element element = getChild(MAP_KEY_CLASS);

			if (element == null) {
				element = addChild(MAP_KEY_CLASS);
			}

			setAttribute(element, CLASS, className);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyConverterName(String converterName) {
		updateChildTextNode(MAP_KEY_CONVERT, converterName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyEnumeratedType(EnumType enumType) {
		updateChildTextNode(MAP_KEY_ENUMERATED, enumType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMapKeyTemporalType(TemporalType temporalType) {
		updateChildTextNode(MAP_KEY_TEMPORAL, temporalType);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setOrderByFieldName(String name) {
		updateChildTextNode(ORDER_BY, name);
	}
}