/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalMapping;
import org.eclipse.persistence.tools.utility.ClassNameTools;
import org.eclipse.persistence.tools.utility.StringTools;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form for a mapping, which is a child of an entity.
 *
 * @see EmbeddableEntity
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
abstract class Mapping extends AbstractExternalForm
                       implements ExternalMapping {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>Mapping</code>.
	 *
	 * @param parent The parent of this external form
	 */
	Mapping(Embeddable parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected Element addChild(Element element, String elementName) {
		return addChild(element, elementName, getParent().getMappingElementNamesOrder());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element addSelf(String elementName, List<String> elementNamesOrder) {

		Element element = getChild(getParent(), ATTRIBUTES);

		if (element == null) {
			element = addChild(getParent(), ATTRIBUTES, getParent().getElementNamesOrder());
		}

		return addChild(element, elementName);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getAttributeType() {
		return getAttribute(ATTRIBUTE_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getAttributeTypeShortName() {
		return ClassNameTools.simpleName(getAttributeType());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getAttributeTypeTextRange() {
		return getAttributeTextRange(ATTRIBUTE_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final Element getElement() {

		Element element = getChild(getParent(), ATTRIBUTES);

		if (element != null) {
			return getChild(element, getElementName(), index);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getGetMethodName() {
		return javabeanAccessorName("get", getName(), StringTools.EMPTY_STRING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getName() {
		return getAttribute(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final TextRange getNameTextRange() {
		return getAttributeTextRange(NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Embeddable getParent() {
		return (Embeddable) super.getParent();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final String getSetMethodName() {
		return javabeanAccessorName("set", getName(), StringTools.EMPTY_STRING);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isBasicMapping() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isElementCollectionMapping() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isEmbeddedMapping() {
		return false;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isIdMapping() {
		return false;
	}

	/**
	 * Generates a JavaBean compliant method name based on the provided property name. Does not alter
	 * the case of the property name if the second character is capitalized, e.g.:
	 * <ul>
	 *    <li>javabeanMethodName("get", "URL", "") produces "getURL"
	 *    <li>javabeanMethodName("get", "fName", "") produces "getfName"
	 *    <li>javabeanMethodName("get", "fname", "") produces "getFname"
	 *    <li>javabeanMethodName("get", "firstName", "") produces "getFirstName"
	 * </ul>
	 *
	 * The "fName" example in particular may seem strange, but is required by JDev ADF interpretation
	 * of the JavaBean naming conventions.<p>
	 * See bug #4572393.
	 *
	 * @param prefix <code>String</code> representing the prefix for the method name (for example,
	 * "get" to produce a getter method)
	 * @param propertyName <code>String</code> representing the name of the property for which an
	 * accessor name is being generated
	 * @param suffix <code>String</code> representing the suffix for the accessor being generated
	 * (for example, "Holder" for ValueHolder accessors)
	 */
	private String javabeanAccessorName(String prefix, String propertyName, String suffix) {

		StringBuilder sb = new StringBuilder(propertyName.length() + 10);
		sb.append(prefix);

		if (propertyName.length() >=  2 && Character.isUpperCase(propertyName.charAt(1))) {
			sb.append(propertyName);
		}
		else {
			sb.append(StringTools.capitalize(propertyName));
		}

		sb.append(suffix);

		return sb.toString();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void removeSelf() {

		Element element = getChild(getParent(), ATTRIBUTES);

		if (element != null) {
			removeChild(element, getElementName(), index);

			if (!hasAnyChildren(element)) {
				remove(getParentElement(), element);
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setAttributeType(String attributeType) {
		setAttribute(ATTRIBUTE_TYPE, attributeType);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public final void setName(String name) {
		setAttribute(NAME, name);
	}
}