/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.annotations.CacheCoordinationType;
import org.eclipse.persistence.annotations.CacheType;
import org.eclipse.persistence.annotations.DatabaseChangeNotificationType;
import org.eclipse.persistence.config.CacheIsolationType;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalCache;
import org.eclipse.persistence.tools.mapping.orm.ExternalTimeOfDay;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * The external form of a cache, which is a child of a mapped superclass entity.
 *
 * @see MappedSuperClassEntity
 *
 * @version 2.6
 */
final class Cache extends AbstractExternalForm
                  implements ExternalCache {

	/**
	 * Creates a new <code>Cache</code>.
	 *
	 * @param parent The parent of this external form
	 */
	Cache(MappedSuperclass parent) {
		super(parent);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTimeOfDay addExpiryTimeOfDay() {
		TimeOfDay timeOfDay = buildTimeOfDay();
		timeOfDay.addSelf();
		return timeOfDay;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildAttributeNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(SIZE);
		names.add(SHARED);
		names.add(TYPE);
		names.add(ALWAYS_REFRESH);
		names.add(REFRESH_ONLY_IF_NEWER);
		names.add(DISABLE_HITS);
		names.add(COORDINATION_TYPE);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(EXPIRY);
		names.add(TimeOfDay.EXPIRY_TIME_OF_DAY);
		return names;
	}

	private TimeOfDay buildTimeOfDay() {
		return new TimeOfDay(this);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getAlwaysRefreshTextRange() {
		return getAttributeTextRange(ALWAYS_REFRESH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheCoordinationType getCoordinationType() {
		return getEnumAttribute(COORDINATION_TYPE, CacheCoordinationType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getCoordinationTypeTextRange() {
		return getAttributeTextRange(COORDINATION_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public DatabaseChangeNotificationType getDatabaseChangeNotificationType() {
		return getEnumAttribute(DATABASE_CHANGE_NOTIFICATION_TYPE, DatabaseChangeNotificationType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDatabaseChangeNotificationTypeTextRange() {
		return getAttributeTextRange(DATABASE_CHANGE_NOTIFICATION_TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getDisableHitsTextRange() {
		return getAttributeTextRange(DISABLE_HITS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return CACHE;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ExternalTimeOfDay getExpiryTimeOfDay() {

		if (hasExpiryTimeOfDay()) {
			return buildTimeOfDay();
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getExpiryTimeOfDayTextRange() {
		return getChildTextRange(TimeOfDay.EXPIRY_TIME_OF_DAY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getExpiryTimeToLive() {
		return getChildIntegerNode(EXPIRY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getExpiryTimeToLiveTextRange() {
		return getChildTextNodeTextRange(EXPIRY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheIsolationType getIsolationType() {
		return getEnumAttribute(ISOLATION, CacheIsolationType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getIsolationTypeTextRange() {
		return getAttributeTextRange(ISOLATION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getRefreshOnlyIfNewerTextRange() {
		return getAttributeTextRange(REFRESH_ONLY_IF_NEWER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getSharedTextRange() {
		return getAttributeTextRange(SHARED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Integer getSize() {
		return getIntegerAttribute(SIZE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getSizeTextRange() {
		return getAttributeTextRange(SIZE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public CacheType getType() {
		return getEnumAttribute(TYPE, CacheType.class);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getTypeTextRange() {
		return getAttributeTextRange(TYPE);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean hasExpiryTimeOfDay() {
		return hasChild(TimeOfDay.EXPIRY_TIME_OF_DAY);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isAlwaysRefresh() {
		return getBooleanAttribute(ALWAYS_REFRESH);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isDisableHits() {
		return getBooleanAttribute(DISABLE_HITS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isRefreshOnlyIfNewer() {
		return getBooleanAttribute(REFRESH_ONLY_IF_NEWER);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Boolean isShared() {
		return getBooleanAttribute(SHARED);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeExpiryTimeOfDay() {

		TimeOfDay timeOfDay = buildTimeOfDay();
		timeOfDay.removeSelf();

		if (!hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setAlwaysRefresh(Boolean alwaysRefresh) {

		setAttribute(ALWAYS_REFRESH, alwaysRefresh);

		if ((alwaysRefresh == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setCoordinationType(CacheCoordinationType type) {

		setAttribute(COORDINATION_TYPE, type);

		if ((type == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDatabaseChangeNotificationType(DatabaseChangeNotificationType type) {

		setAttribute(DATABASE_CHANGE_NOTIFICATION_TYPE, type);

		if ((type == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDisableHits(Boolean disableHits) {

		setAttribute(DISABLE_HITS, disableHits);

		if ((disableHits == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setExpiryTimeToLive(Integer timeToLive) {

		updateChildTextNode(EXPIRY, timeToLive);

		if ((timeToLive == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setIsolationType(CacheIsolationType isolationType) {

		setAttribute(ISOLATION, isolationType);

		if ((isolationType == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setRefreshOnlyIfNewer(Boolean refreshOnlyIfNewer) {

		setAttribute(REFRESH_ONLY_IF_NEWER, refreshOnlyIfNewer);

		if ((refreshOnlyIfNewer == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setShared(Boolean shared) {

		setAttribute(SHARED, shared);

		if ((shared == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSize(Integer size) {

		setAttribute(SIZE, size);

		if ((size == null) && !hasAnything()) {
			removeSelf();
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setType(CacheType type) {

		setAttribute(TYPE, type);

		if ((type == null) && !hasAnything()) {
			removeSelf();
		}
	}
}