/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm.dom;

import java.util.ArrayList;
import java.util.List;
import org.eclipse.persistence.tools.mapping.AbstractExternalForm;
import org.eclipse.persistence.tools.mapping.orm.ExternalEntityListener;
import org.eclipse.persistence.tools.utility.TextRange;
import org.w3c.dom.Element;

/**
 * The external form of an entity listener.
 *
 * @see PersistenceUnit
 *
 * @version 2.6
 */
abstract class AbstractEntityListener extends AbstractExternalForm
                                      implements ExternalEntityListener {

	/**
	 * The position of the element within the list of children with the same type owned by the parent.
	 */
	private int index;

	/**
	 * Creates a new <code>EntityListener</code>.
	 *
	 * @param parent The parent of this external form
	 * @param index The position of the element within the list of children with the same type owned by the parent
	 */
	AbstractEntityListener(AbstractExternalForm parent, int index) {
		super(parent);
		this.index  = index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected List<String> buildElementNamesOrder() {
		List<String> names = new ArrayList<String>();
		names.add(DESCRIPTION);
		names.add(PRE_PERSIST);
		names.add(POST_PERSIST);
		names.add(PRE_REMOVE);
		names.add(POST_REMOVE);
		names.add(PRE_UPDATE);
		names.add(POST_UPDATE);
		names.add(POST_LOAD);
		return names;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void calculateInsertionIndex(Element parent, Element child, String elementName) {
		if (elementName == getElementName()) {
			index = index(parent, child, elementName);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getClassName() {
		return getAttribute(CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getClassNameTextRange() {
		return getAttributeTextRange(CLASS);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getDescription() {
		return getChildTextNode(DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected String getElementName() {
		return ENTITY_LISTENER;
	}

	@Override
	@Deprecated
	// Note: Once the model does not require this index,
	// change the visibility to default and remove @Deprecated
	public int getIndex() {
		return index;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostLoadMethod() {
		return getChildAttribute(POST_LOAD, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostLoadMethodDescription() {
		return getChildAttribute(POST_LOAD, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPostLoadMethodTextRange() {
		return getChildAttributeTextRange(POST_LOAD, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostPersistMethod() {
		return getChildAttribute(POST_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostPersistMethodDescription() {
		return getChildAttribute(POST_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPostPersistMethodTextRange() {
		return getChildAttributeTextRange(POST_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostRemoveMethod() {
		return getChildAttribute(POST_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostRemoveMethodDescription() {
		return getChildAttribute(POST_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPostRemoveMethodTextRange() {
		return getChildAttributeTextRange(POST_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostUpdateMethod() {
		return getChildAttribute(POST_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPostUpdateMethodDescription() {
		return getChildAttribute(POST_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPostUpdateMethodTextRange() {
		return getChildAttributeTextRange(POST_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPrePersistMethod() {
		return getChildAttribute(PRE_PERSIST, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPrePersistMethodDescription() {
		return getChildAttribute(PRE_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPrePersistMethodTextRange() {
		return getChildAttributeTextRange(PRE_PERSIST, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreRemoveMethod() {
		return getChildAttribute(PRE_REMOVE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreRemoveMethodDescription() {
		return getChildAttribute(PRE_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPreRemoveMethodTextRange() {
		return getChildAttributeTextRange(PRE_REMOVE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreUpdateMethod() {
		return getChildAttribute(PRE_UPDATE, METHOD_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String getPreUpdateMethodDescription() {
		return getChildAttribute(PRE_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public TextRange getPreUpdateMethodTextRange() {
		return getChildAttributeTextRange(PRE_UPDATE, DESCRIPTION);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setClassName(String name) {
		setAttribute(CLASS, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDescription(String description) {
		updateChildTextNode(DESCRIPTION, description);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostLoadMethod(String value) {
		updateChildAttribute(POST_LOAD, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostLoadMethodDescription(String name) {
		updateChildChildTextNode(POST_LOAD, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostPersistMethod(String value) {
		updateChildAttribute(POST_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostPersistMethodDescription(String name) {
		updateChildChildTextNode(POST_PERSIST, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostRemoveMethod(String value) {
		updateChildAttribute(POST_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostRemoveMethodDescription(String name) {
		updateChildChildTextNode(POST_REMOVE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostUpdateMethod(String value) {
		updateChildAttribute(POST_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPostUpdateMethodDescription(String name) {
		updateChildChildTextNode(POST_UPDATE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrePersistMethod(String value) {
		updateChildAttribute(PRE_PERSIST, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPrePersistMethodDescription(String name) {
		updateChildChildTextNode(PRE_PERSIST, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreRemoveMethod(String value) {
		updateChildAttribute(PRE_REMOVE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreRemoveMethodDescription(String name) {
		updateChildChildTextNode(PRE_REMOVE, DESCRIPTION, name);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreUpdateMethod(String value) {
		updateChildAttribute(PRE_UPDATE, METHOD_NAME, value);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setPreUpdateMethodDescription(String name) {
		updateChildChildTextNode(PRE_UPDATE, DESCRIPTION, name);
	}
}