/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import java.util.List;
import org.eclipse.persistence.tools.mapping.ExternalForm;
import org.eclipse.persistence.tools.utility.TextRange;
import org.eclipse.persistence.tools.utility.iterable.ListIterable;

/**
 * This interface and inherited behavior describe the configurable properties
 * defined for a table definition in the ORM xml. There are a variety of places
 * where a table may be used or declared in the ORM xml.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.Table
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalEntityTable extends ExternalForm {

	/**
	 * The attribute name used to store and retrieve the catalog property.
	 */
	String CATALOG = "catalog";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the schema property.
	 */
	String SCHEMA = "schema";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String TABLE = "table";

	/**
	 * Adds unique constraint with the provided column names.
	 */
	ExternalUniqueConstraint addUniqueConstraint(ListIterable<String> columnNames);

	/**
	 * Returns the name of this table's catalog.
	 */
	String getCatalogName();

	/**
	 * Returns the {@link TextRange} of the catalog.
	 */
	TextRange getCatalogTextRange();

	/**
	 * Returns the name of the table.
	 */
	String getName();

	/**
	 * Returns the {@link TextRange} of the table name.
	 */
	TextRange getNameTextRange();

	/**
	 * Returns the name of this table's schema.
	 */
	String getSchemaName();

	/**
	 * Returns the {@link TextRange} of the schema name.
	 */
	TextRange getSchemaTextRange();

	/**
	 * Returns the unique constraints at the specified index.
	 */
	ExternalUniqueConstraint getUniqueConstraint(int index);

	/**
	 * Removes all the unique constraints owned by this entity table.
	 */
	void removeAllUniqueConstraints();

	/**
	 * Remove the unique constraint at the given position.
	 */
	void removeUniqueConstraint(int index);

	/**
	 * Set the name of this tables catalog.
	 */
	void setCatalogName(String catalog);

	/**
	 * Set the name of the table.
	 */
	void setName(String name);

	/**
	 * Set the schema name of this table.
	 */
	void setSchemaName(String name);

	/**
	 * Returns a list of all unique constraints defined for this table definition.
	 */
	List<ExternalUniqueConstraint> uniqueConstraints();

	/**
	 * Returns the count of all unique constraints defined for this table definition.
	 */
	int uniqueConstraintsSize();
}