/*
 * Copyright (c) 2006, 2012, Oracle and/or its affiliates. All rights reserved.
 *
 * This software is the proprietary information of Oracle Corporation.
 * Use is subject to license terms.
 */
package org.eclipse.persistence.tools.mapping.orm;

import java.util.List;
import javax.persistence.InheritanceType;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * This interface and inherited behavior describes the configurable properties for an entity. One or
 * more entities can be declared as part of an ORM configuration.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @see javax.persistence.Entity
 * @see http://jcp.org/aboutJava/communityprocess/final/jsr220/index.html, persistence specificaton
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalEntity extends ExternalMappedSuperClass {

	/**
	 * The element name used to store and retrieve the class-extractor child node.
	 */
	String CLASS_EXTRACTOR = "class-extractor";

	/**
	 * The element name used to store and retrieve the column child text node.
	 */
	String DISCRIMINATOR_VALUE = "discriminator-value";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String ENTITY = "entity";

	/**
	 * The element name used to store and retrieve the inheritance child node.
	 */
	String INHERITANCE = "inheritance";

	/**
	 * The attribute name used to store and retrieve the name property.
	 */
	String NAME = "name";

	/**
	 * The attribute name used to store and retrieve the strategy property of the inheritance child node.
	 */
	String STRATEGY = "strategy";

	/**
	 * Creates a new discriminator column definition replacing the old one.
	 */
	ExternalDiscriminatorColumn addDiscriminatorColumn(String columnName);

	/**
	 * Adds a primary key join column with the given name.
	 */
	ExternalPrimaryKeyJoinColumn addPrimaryKeyJoinColumn(String name);

	/**
	 * Creates new primary table for this entity replacing the old one.
	 */
	ExternalEntityTable addPrimaryTable(String tableName);

	/**
	 * Adds a secondary table with the given name.
	 */
	ExternalSecondaryTable addSecondaryTable(String name, String catalog, String schema);

	/**
	 * Adds a sql result set mapping with the given name.
	 */
	ExternalSQLResultSetMapping addSqlResultSetMapping(String name);

	/**
	 * Returns the class extractor class name defined for this entity.
	 */
	String getClassExtractorClassName();

	/**
	 * Returns the {@link TextRange} for the class extractor class name defined for this entity.
	 */
	TextRange getClassExtractorClassNameTextRange();

	/**
	 * Returns the discriminator column definition.
	 */
	ExternalDiscriminatorColumn getDiscriminatorColumn();

	/**
	 * Returns the discriminator value.
	 */
	String getDiscriminatorValue();

	/**
	 * Returns the {@link TextRange} for the discriminator value.
	 */
	TextRange getDiscriminatorValueTextRange();

	/**
	 * Returns the inheritance strategy type if one has been declared.
	 */
	InheritanceType getInheritenceStrategy();

	/**
	 * Returns the {@link TextRange} for the inheritance strategy type if one has been declared.
	 */
	TextRange getInheritenceStrategyTextRange();

	/**
	 * Returns the name of the entity.
	 */
	String getName();

	/**
	 * Returns the {@link TextRange} for the name of the entity.
	 */
	TextRange getNameTextRange();

	/**
	 * Returns a primary key join column with the given name.
	 */
	ExternalPrimaryKeyJoinColumn getPrimaryKeyJoinColumn(int index);

	/**
	 * Returns the primary table definition for this entity.
	 */
	ExternalEntityTable getPrimaryTable();

	/**
	 * Returns a the secondary table definition for the given catalog, schema and table name.
	 */
	ExternalSecondaryTable getSecondaryTable(int index);

	/**
	 * Returns the SQL Result Set Mappings at the given position.
	 */
	ExternalSQLResultSetMapping getSqlResultSetMapping(int index);

	/**
	 * Determines whether the primary table definition for this entity is defined or not.
	 */
	boolean hasPrimaryTable();

	/**
	 * Returns a list of the primary key join columns.
	 */
	List<ExternalPrimaryKeyJoinColumn> primaryKeyJoinColumns();

	/**
	 * Returns the count of primary key join columns.
	 */
	int primaryKeyJoinColumnsSize();

	/**
	 * Removes the primary key join column named.
	 */
	void removePrimaryKeyJoinColumn(int index);

	/**
	 * Removes the primary table definition from this entity if it exists.
	 */
	void removePrimaryTable();

	/**
	 * Removes the secondary table named.
	 */
	void removeSecondaryTable(int index);

	/**
	 * Removes the sql result set mapping named.
	 */
	void removeSqlResultSetMapping(int index);

	/**
	 * Returns a list of the secondary table definitions for this entity.
	 */
	List<ExternalSecondaryTable> secondaryTables();

	/**
	 * Returns the count of secondary table definitions for this entity.
	 */
	int secondaryTablesSize();

	/**
	 * Sets the name of the class extractor class declared for this entity.
	 */
	void setClassExtractorClassName(String className);

	/**
	 * set the discriminator value.
	 */
	void setDiscriminatorValue(String discriminatorValue);

	/**
	 * Set the inheritance strategy.
	 */
	void setInheritanceStrategy(InheritanceType type);

	/**
	 * Set the entity name.
	 */
	void setName(String name);

	/**
	 * Returns a list of sql result set mappings defined on this entity.
	 */
	List<ExternalSQLResultSetMapping> sqlResultSetMappings();

	/**
	 * Returns the count of SQL result set mappings defined on this entity.
	 */
	int sqlResultSetMappingsSize();
}