/*******************************************************************************
 * Copyright (c) 2006, 2013 Oracle and/or its affiliates. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.mapping.orm;

import javax.persistence.EnumType;
import org.eclipse.persistence.annotations.JoinFetchType;
import org.eclipse.persistence.tools.utility.TextRange;

/**
 * This interface represents the basis for all element collection mapping types defined by the
 * EclipseLink JPA spec that are available for configuration via the EclipseLink ORM xml.
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.<p>
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ExternalElementCollectionMapping extends ExternalColumnMapping,
                                                          ExternalObjectCollectionMapping,
                                                          ExternalFetchableMapping {

	/**
	 * The attribute name used to store and retrieve the class property.
	 */
	String CLASS = "class";

	/**
	 * The element name used to store and retrieve the convert-key child text node.
	 */
	String CONVERT_KEY = "convert-key";

	/**
	 * The node name used to store and retrieve the element encapsulated by this external form.
	 */
	String ELEMENT_COLLECTION = "element-collection";

	/**
	 * The element name used to store and retrieve the expiry child text node.
	 */
	String ENUMERATED = "enumerated";

	/**
	 * The element name used to store and retrieve the join-fetch child text node.
	 */
	String JOIN_FETCH = "join-fetch";

	/**
	 * The element name used to store and retrieve the lob child text node.
	 */
	String LOB = "lob";

	/**
	 * The element name used to store and retrieve the map-key-class child node.
	 */
	String MAP_KEY_CLASS = "map-key-class";

	/**
	 * The element name used to store and retrieve the map-key-convert child node.
	 */
	String MAP_KEY_CONVERT = "map-key-convert";

	/**
	 * The element name used to store and retrieve the map-key-enumerated child node.
	 */
	String MAP_KEY_ENUMERATED = "map-key-enumerated";

	/**
	 * The element name used to store and retrieve the map-key-temporal child node.
	 */
	String MAP_KEY_TEMPORAL = "map-key-temporal";

	/**
	 * The element name used to store and retrieve the order-by child text node.
	 */
	String ORDER_BY = "order-by";

	/**
	 * The attribute name used to store and retrieve the target-class property.
	 */
	String TARGET_CLASS = "target-class";

	/**
	 * The element name used to store and retrieve the temporal child text node.
	 */
	String TEMPORAL = "temporal";

	/**
	 * Visits this {@link ExternalElementCollectionMapping} by the given {@link ExternalMappingVisitor visitor}.
	 *
	 * @param visitor The {@link ExternalMappingVisitor} to visit this object
	 */
	void accept(ExternalMappingVisitor visitor);

	/**
	 * Adds a override with the given name.
	 */
	ExternalAssociationOverride addAssociationOverride(String name);

	/**
	 * Adds a override with the given name.
	 */
	ExternalAttributeOverride addAttributeOverride(String name);

	/**
	 * Adds a batch fetch to use.
	 */
	ExternalBatchFetch addBatchFetch();

	/**
	 * Adds a collection table to this mapping.
	 */
	ExternalCollectionTable addCollectionTable();

	/**
	 * Returns the batch fetch definition for this mapping.
	 */
	ExternalBatchFetch getBatchFetch();

	/**
	 * Returns the collection table for this mapping.
	 */
	ExternalCollectionTable getCollectionTable();

	/**
	 * Returns the convert key for this mapping.
	 */
	String getConvertKey();

	/**
	 * Returns the {@link TextRange} for the convert key for this mapping.
	 */
	TextRange getConvertKeyTextRange();

	/**
	 * Returns the enumerated type of this mapping if one exists.
	 */
	EnumType getEnumeratedType();

	/**
	 * Returns the {@link TextRange} for the enumerated type of this mapping if one exists.
	 */
	TextRange getEnumeratedTypeTextRange();

	/**
	 * Returns the join fetch type for this mapping.
	 */
	JoinFetchType getJoinFetchType();

	/**
	 * Returns the {@link TextRange} for the join fetch type for this mapping.
	 */
	TextRange getJoinFetchTypeTextRange();

	/**
	 * Returns the key converter for this mapping.
	 */
	ExternalClassConverter getKeyConverter();

	/**
	 * Returns the key object type converter for this mapping.
	 */
	ExternalObjectTypeConverter getKeyObjectTypeConverter();

	/**
	 * Returns the key struct converter for this mapping.
	 */
	ExternalStructConverter getKeyStructConverter();

	/**
	 * Returns the key type converter for this mapping.
	 */
	ExternalTypeConverter getKeyTypeConverter();

	/**
	 * Returns the {@link TextRange} for the LOB value.
	 */
	TextRange getLobTextRange();

	/**
	 * Returns the text range for this mapping's element declaration.
	 */
	TextRange getMappingTextRange();

	/**
	 * Returns the target class name for this mapping.
	 */
	String getTargetClassName();

	/**
	 * Returns the target class name text range for this mapping.
	 */
	TextRange getTargetClassNameTextRange();

	/**
	 * Returns the target class short name for this mapping.
	 */
	String getTargetClassShortName();

	/**
	 * Indicates whether this basic mapping's column is a LOB.
	 */
	Boolean isLob();

	/**
	 * Removes the batch fetch.
	 */
	void removeBatchFetch();

	/**
	 * Removes the collection table from this mapping.
	 */
	void removeCollectionTable();

	/**
	 * Removes the key converter from mapping.
	 */
	void removeKeyConverter();

	/**
	 * Removes the key object type converter from mapping.
	 */
	void removeKeyObjectTypeConverter();

	/**
	 * Removes the key struct converter from mapping.
	 */
	void removeKeyStructConverter();

	/**
	 * Removes the key type converter from mapping.
	 */
	void removeKeyTypeConverter();

	/**
	 * Sets the mapping to use this converter for its key converter.
	 */
	void setAsKeyConverter(String name, String className);

	/**
	 * Sets the mapping to use this object type converter for its key converter.
	 */
	void setAsKeyObjectTypeConverter(String name);

	/**
	 * Sets the mapping to use this struct converter for its key converter.
	 */
	void setAsKeyStructConverter(String name, String converter);

	/**
	 * Sets the mapping to use this type converter for its key converter.
	 */
	void setAsKeyTypeConverter(String name);

	/**
	 * Sets the convert key.
	 */
	void setConvertKey(String value);

	/**
	 * Sets enumerated type.
	 */
	void setEnumeratedType(EnumType type);

	/**
	 * sets a join fetch type to this mapping.
	 */
	void setJoinFetchType(JoinFetchType type);

	/**
	 * Sets mapping as lob.
	 */
	void setLob(Boolean lob);

	/**
	 * Sets the target class name.
	 */
	void setTargetClassName(String className);
}