/*******************************************************************************
 * Copyright (c) 2008, 2012 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *     Oracle - initial API and implementation
 *
 ******************************************************************************/
package org.eclipse.persistence.tools.gen.db;

import java.sql.Connection;
import org.eclipse.persistence.tools.utility.status.IStatus;

/**
 * Database connection profile
 * <p>
 * Provisional API: This interface is part of an interim API that is still under development and
 * expected to change significantly before reaching stability. It is available at this early stage
 * to solicit feedback from pioneering adopters on the understanding that any code that uses this
 * API will almost certainly be broken (repeatedly) as the API evolves.
 *
 * @version 2.6
 */
@SuppressWarnings("nls")
public interface ConnectionProfile extends DatabaseObject {
	// ********** properties **********

	/**
	 * Returns the connection profile's database.
	 * Returns <code>null</code> if the connection profile is inactive.
	 */
	@Override
	Database getDatabase();

	/**
	 * Returns ID of the provider managing the DTP profile.
	 */
	String getProviderID();

	/**
	 * Returns the connection profile's static ID.
	 */
	String getInstanceID();

	/**
	 * Returns the default database name.
	 */
	String getDatabaseName();

	/**
	 * Returns the database product name.
	 */
	String getDatabaseProduct();

	/**
	 * Returns the database vendor.
	 */
	String getDatabaseVendor();

	/**
	 * Returns the database version.
	 */
	String getDatabaseVersion();

	/**
	 * Returns the driver class name.
	 */
	String getDriverClassName();

	/**
	 * Returns the default connection URL.
	 */
	String getURL();

	/**
	 * Returns the default user name.
	 */
	String getUserName();

	/**
	 * Returns the default user password.
	 */
	String getUserPassword();

	/**
	 * Returns the ID of the associated Driver definition.
	 */
	String getDriverDefinitionID();

	/**
	 * Returns the jar list for the associated Driver as a
	 * comma-delimited string.
	 */
	String getDriverJarList();

	/**
	 * Returns the name of the associated Driver definition.
	 */
	String getDriverName();


	// ********** identifiers **********

	/**
	 * Returns whether all identifiers are to be treated as though they were
	 * delimited. This is determined by the client-supplied database
	 * identifier adapter.
	 */
	boolean treatIdentifiersAsDelimited();


	// ********** connection **********

	/**
	 * Returns whether the profile is either connected to a live database
	 * session or working off-line (i.e. it has access to meta-data).
	 * @see #isConnected()
	 * @see #isWorkingOffline()
	 */
	boolean isActive();

	/**
	 * Returns whether the profile is neither connected to a live database
	 * session nor working off-line (i.e. it has access to meta-data).
	 * @see #isActive()
	 */
	boolean isInactive();

	/**
	 * Returns whether the profile is connected to a live database session
	 * (i.e. the meta-data comes from the database), as opposed to working
	 * off-line.
	 * @see #isActive()
	 */
	boolean isConnected();

	/**
	 * Returns whether the profile is not connected to a live database session
	 * (i.e. the meta-data comes from the database), as opposed to working
	 * off-line.
	 * @see #isConnected()
	 */
	boolean isDisconnected();

	/**
	 * Connect to the database.
	 * @see #disconnect()
	 */
	void connect();

	/**
	 * Disconnect from the database.
	 * @see #connect()
	 */
	void disconnect();

	/**
	 * Returns the JDBC connection.
	 */
	Connection getJDBCConnection();


	// ********** off-line support **********

	/**
	 * Returns whether the profile is working off-line (i.e. the meta-data
	 * comes from a local cache), as opposed to connected to a live
	 * database session.
	 * @see #isActive()
	 */
	boolean isWorkingOffline();

	/**
	 * Returns whether the connection factories associated with the
	 * connection profile's provider support working offline.
	 */
	boolean supportsWorkOfflineMode();

	/**
	 * Save the state of the connection profile for working in an offline mode.
	 * If the connection profile does not support working in an offline mode, no
	 * Returns immediately.
	 */
	IStatus saveWorkOfflineData();

	/**
	 * Returns whether the connection profile supports working offline and data
	 * has been saved for working offline.
	 */
	boolean canWorkOffline();

	/**
	 * Begin working off-line.
	 */
	IStatus workOffline();


	// ********** listeners **********

	/**
	 * Add the specified connection listener to the connection profile.
	 */
	void addConnectionListener(ConnectionListener listener);

	/**
	 * Remove the specified connection listener from the connection profile.
	 */
	void removeConnectionListener(ConnectionListener listener);


	// ********** constants **********

	String CONNECTION_PROFILE_TYPE = "org.eclipse.datatools.connectivity.db.generic.connectionProfile";
	String DRIVER_DEFINITION_PROP_ID = "org.eclipse.datatools.connectivity.driverDefinitionID";
	String DRIVER_DEFINITION_TYPE_PROP_ID = "org.eclipse.datatools.connectivity.drivers.defnType";
	String DRIVER_JAR_LIST_PROP_ID = "jarList";
//	String DATABASE_SAVE_PWD_PROP_ID = IJDBCDriverDefinitionConstants.PROP_PREFIX + "savePWD";
}