/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libe-book project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef LRFCOLLECTOR_H_INCLUDED
#define LRFCOLLECTOR_H_INCLUDED

#include <string>
#include <stack>

#include "LRFTypes.h"


namespace libebook
{

class LRFCollector
{
  // -Weffc++
  LRFCollector(const LRFCollector &other);
  LRFCollector &operator=(const LRFCollector &other);

  struct ImageData
  {
    librevenge::RVNGInputStream *image;
    ImageType type;
  };

  typedef std::map<unsigned, ImageData> ImageMap_t;

public:
  explicit LRFCollector(librevenge::RVNGTextInterface *document);
  ~LRFCollector();

  void startDocument();
  void endDocument();

  void openPage(unsigned pageAtrID, const LRFAttributes &attributes);
  void closePage();

  void openBlock(unsigned atrID, const LRFAttributes &attributes);
  void closeBlock();

  void openTextBlock(unsigned atrID, const LRFAttributes &attributes);
  void closeTextBlock();

  void openParagraph(unsigned atrID, const LRFAttributes &attributes);
  void closeParagraph();

  void collectText(const std::string &text, const LRFAttributes &attributes);
  void insertLineBreak();

  void collectMetadata(const LRFMetadata &metadata);

  void collectBookAttributes(const LRFAttributes &attributes);
  void collectTextAttributes(unsigned id, const LRFAttributes &attributes);
  void collectBlockAttributes(unsigned id, const LRFAttributes &attributes);
  void collectPageAttributes(unsigned id, const LRFAttributes &attributes);
  void collectParagraphAttributes(unsigned id, const LRFAttributes &attributes);

  void collectImage(unsigned id);
  void collectImageData(unsigned id, ImageType type, librevenge::RVNGInputStream *image);

private:
  void collectAttributes(unsigned id, const LRFAttributes &attributes, LRFAttributeMap_t &attributeMap);
  void openBlock(unsigned atrID, const LRFAttributes &attributes, const LRFAttributeMap_t *attributeMap);

private:
  LRFAttributes m_bookAttributes;
  LRFAttributeMap_t m_textAttributeMap;
  LRFAttributeMap_t m_blockAttributeMap;
  LRFAttributeMap_t m_pageAttributeMap;
  LRFAttributeMap_t m_paragraphAttributeMap;
  librevenge::RVNGTextInterface *const m_document;
  std::stack<LRFAttributes> m_currentAttributes;
  ImageMap_t m_imageMap;
};

}

#endif // LRFCOLLECTOR_H_INCLUDED

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
