# Copyright (c) Streamlit Inc. (2018-2022) Snowflake Inc. (2022-2026)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from __future__ import annotations

import re
from collections.abc import Sequence
from dataclasses import dataclass
from datetime import date, datetime, time, timedelta
from textwrap import dedent
from typing import (
    TYPE_CHECKING,
    Any,
    Final,
    Literal,
    TypeAlias,
    cast,
    overload,
)

from streamlit.elements.lib.form_utils import current_form_id
from streamlit.elements.lib.layout_utils import (
    LayoutConfig,
    WidthWithoutContent,
    validate_width,
)
from streamlit.elements.lib.policies import (
    check_widget_policies,
    maybe_raise_label_warnings,
)
from streamlit.elements.lib.utils import (
    Key,
    LabelVisibility,
    compute_and_register_element_id,
    get_label_visibility_proto_value,
    to_key,
)
from streamlit.errors import StreamlitAPIException
from streamlit.proto.DateInput_pb2 import DateInput as DateInputProto
from streamlit.proto.DateTimeInput_pb2 import DateTimeInput as DateTimeInputProto
from streamlit.proto.TimeInput_pb2 import TimeInput as TimeInputProto
from streamlit.runtime.metrics_util import gather_metrics
from streamlit.runtime.scriptrunner import ScriptRunContext, get_script_run_ctx
from streamlit.runtime.state import (
    BindOption,
    WidgetArgs,
    WidgetCallback,
    WidgetKwargs,
    get_session_state,
    register_widget,
)
from streamlit.time_util import adjust_years

if TYPE_CHECKING:
    from streamlit.delta_generator import DeltaGenerator

# Type for things that point to a specific time (even if a default time, though not None).
TimeValue: TypeAlias = time | datetime | str | Literal["now"]
DateTimeScalarValue: TypeAlias = datetime | date | time | str | Literal["now"]
DateTimeValue: TypeAlias = DateTimeScalarValue | None

# Type for things that point to a specific date (even if a default date, including None).
NullableScalarDateValue: TypeAlias = date | datetime | str | Literal["today"] | None

# The accepted input value for st.date_input. Can be a date scalar or a date range.
DateValue: TypeAlias = NullableScalarDateValue | Sequence[NullableScalarDateValue]

# The return value of st.date_input.
DateWidgetRangeReturn: TypeAlias = tuple[()] | tuple[date] | tuple[date, date]
DateWidgetReturn: TypeAlias = date | DateWidgetRangeReturn | None


DEFAULT_STEP_MINUTES: Final = 15
ALLOWED_DATE_FORMATS: Final = re.compile(
    r"^(YYYY[/.\-]MM[/.\-]DD|DD[/.\-]MM[/.\-]YYYY|MM[/.\-]DD[/.\-]YYYY)$"
)
_DATETIME_LEGACY_FORMAT: Final = "%Y/%m/%d, %H:%M"
_DATETIME_ISO_FORMAT: Final = "%Y-%m-%dT%H:%M"
_DEFAULT_MIN_BOUND_TIME: Final = time(hour=0, minute=0)
_DEFAULT_MAX_BOUND_TIME: Final = time(hour=23, minute=59)


def _convert_timelike_to_time(value: TimeValue) -> time:
    if value == "now":
        # Set value default.
        return datetime.now().time().replace(second=0, microsecond=0)

    if isinstance(value, str):
        try:
            return time.fromisoformat(value)
        except ValueError:
            try:
                return (
                    datetime.fromisoformat(value)
                    .time()
                    .replace(second=0, microsecond=0)
                )
            except ValueError:
                # We throw an error below.
                pass

    if isinstance(value, datetime):
        return value.time().replace(second=0, microsecond=0)

    if isinstance(value, time):
        return value

    raise StreamlitAPIException(
        "The type of value should be one of datetime, time, ISO string or None"
    )


def _convert_datelike_to_date(
    value: NullableScalarDateValue,
) -> date:
    if isinstance(value, datetime):
        return value.date()

    if isinstance(value, date):
        return value

    if value == "today":
        return datetime.now().date()

    if isinstance(value, str):
        try:
            return date.fromisoformat(value)
        except ValueError:
            try:
                return datetime.fromisoformat(value).date()
            except ValueError:
                # We throw an error below.
                pass

    raise StreamlitAPIException(
        'Date value should either be an date/datetime or an ISO string or "today"'
    )


def _parse_date_value(value: DateValue) -> tuple[list[date] | None, bool]:
    if value is None:
        return None, False

    value_tuple: Sequence[NullableScalarDateValue]

    if isinstance(value, Sequence) and not isinstance(value, str):
        is_range = True
        value_tuple = value  # ty: ignore[invalid-assignment]
    else:
        is_range = False
        value_tuple = [cast("NullableScalarDateValue", value)]

    if len(value_tuple) not in {0, 1, 2}:
        raise StreamlitAPIException(
            "DateInput value should either be an date/datetime or a list/tuple of "
            "0 - 2 date/datetime values"
        )

    parsed_dates = [_convert_datelike_to_date(v) for v in value_tuple]

    return parsed_dates, is_range


def _parse_min_date(
    min_value: NullableScalarDateValue,
    parsed_dates: Sequence[date] | None,
) -> date:
    parsed_min_date: date
    if isinstance(min_value, (datetime, date, str)):
        parsed_min_date = _convert_datelike_to_date(min_value)
    elif min_value is None:
        if parsed_dates:
            parsed_min_date = adjust_years(parsed_dates[0], years=-10)
        else:
            parsed_min_date = adjust_years(date.today(), years=-10)
    else:
        raise StreamlitAPIException(
            "DateInput min should either be a date/datetime or None"
        )
    return parsed_min_date


def _parse_max_date(
    max_value: NullableScalarDateValue,
    parsed_dates: Sequence[date] | None,
) -> date:
    parsed_max_date: date
    if isinstance(max_value, (datetime, date, str)):
        parsed_max_date = _convert_datelike_to_date(max_value)
    elif max_value is None:
        if parsed_dates:
            parsed_max_date = adjust_years(parsed_dates[-1], years=10)
        else:
            parsed_max_date = adjust_years(date.today(), years=10)
    else:
        raise StreamlitAPIException(
            "DateInput max should either be a date/datetime or None"
        )
    return parsed_max_date


def _normalize_time(value: time) -> time:
    """Return a time without seconds, microseconds, or timezone info."""
    return value.replace(second=0, microsecond=0, tzinfo=None)


def _normalize_datetime_value(value: datetime) -> datetime:
    """Return a datetime without seconds, microseconds, or timezone info."""
    if value.tzinfo is not None:
        value = value.replace(tzinfo=None)
    return value.replace(second=0, microsecond=0)


def _combine_date_time(component_date: date, component_time: time) -> datetime:
    """Combine a date and time into a normalized datetime."""
    return datetime.combine(component_date, _normalize_time(component_time))


def _try_parse_datetime_with_format(value: str, fmt: str) -> datetime | None:
    """Try to parse a datetime string with a specific format."""
    try:
        return datetime.strptime(value, fmt)
    except ValueError:
        return None


def _convert_datetimelike_to_datetime(
    value: DateTimeScalarValue,
    *,
    fallback_date: date,
    fallback_time: time,
) -> datetime:
    """Convert supported datetime inputs into a normalized datetime."""
    fallback_time = _normalize_time(fallback_time)

    if value == "now":
        return _normalize_datetime_value(datetime.now())

    if isinstance(value, datetime):
        return _normalize_datetime_value(value)

    if isinstance(value, date) and not isinstance(value, datetime):
        return _combine_date_time(value, fallback_time)

    if isinstance(value, time):
        return _combine_date_time(fallback_date, value)

    if isinstance(value, str):
        stripped_value = value.strip()

        try:
            parsed_dt = datetime.fromisoformat(stripped_value)
            return _normalize_datetime_value(parsed_dt)
        except ValueError:
            pass

        for fmt in (
            "%Y/%m/%d %H:%M",
            "%Y/%m/%d %H:%M:%S",
            "%Y-%m-%d %H:%M",
            "%Y-%m-%d %H:%M:%S",
        ):
            maybe_parsed_dt = _try_parse_datetime_with_format(stripped_value, fmt)
            if maybe_parsed_dt is not None:
                return _normalize_datetime_value(maybe_parsed_dt)

        try:
            parsed_date = date.fromisoformat(stripped_value)
            return _combine_date_time(parsed_date, fallback_time)
        except ValueError:
            pass

        try:
            parsed_time = time.fromisoformat(stripped_value)
            return _combine_date_time(fallback_date, parsed_time)
        except ValueError:
            pass

    raise StreamlitAPIException(
        "The type of value should be one of datetime, date, time, ISO string, or 'now'."
    )


def _default_min_datetime(base_date: date) -> datetime:
    return _combine_date_time(
        adjust_years(base_date, years=-10), _DEFAULT_MIN_BOUND_TIME
    )


def _default_max_datetime(base_date: date) -> datetime:
    return _combine_date_time(
        adjust_years(base_date, years=10), _DEFAULT_MAX_BOUND_TIME
    )


def _datetime_to_proto_string(value: datetime) -> str:
    return _normalize_datetime_value(value).strftime(_DATETIME_ISO_FORMAT)


@dataclass(frozen=True)
class _DateTimeInputValues:
    value: datetime | None
    min: datetime
    max: datetime

    @classmethod
    def from_raw_values(
        cls,
        value: DateTimeValue,
        min_value: DateTimeValue,
        max_value: DateTimeValue,
    ) -> _DateTimeInputValues:
        parsed_value = (
            None
            if value is None
            else _convert_datetimelike_to_datetime(
                value,
                fallback_date=date.today(),
                fallback_time=_DEFAULT_MIN_BOUND_TIME,
            )
        )

        base_date_for_bounds = (
            parsed_value.date() if parsed_value is not None else date.today()
        )

        parsed_min = (
            _default_min_datetime(base_date_for_bounds)
            if min_value is None
            else _convert_datetimelike_to_datetime(
                min_value,
                fallback_date=base_date_for_bounds,
                fallback_time=_DEFAULT_MIN_BOUND_TIME,
            )
        )

        parsed_max = (
            _default_max_datetime(base_date_for_bounds)
            if max_value is None
            else _convert_datetimelike_to_datetime(
                max_value,
                fallback_date=base_date_for_bounds,
                fallback_time=_DEFAULT_MAX_BOUND_TIME,
            )
        )

        return cls(
            value=parsed_value,
            min=parsed_min,
            max=parsed_max,
        )

    def __post_init__(self) -> None:
        if self.min > self.max:
            raise StreamlitAPIException(
                f"The `min_value`, set to {self.min}, shouldn't be larger "
                f"than the `max_value`, set to {self.max}."
            )

        if self.value is not None and (self.value < self.min or self.value > self.max):
            raise StreamlitAPIException(
                f"The default `value` of {self.value} must lie between the `min_value` "
                f"of {self.min} and the `max_value` of {self.max}, inclusively."
            )


@dataclass(frozen=True)
class _DateInputValues:
    value: Sequence[date] | None
    is_range: bool
    max: date
    min: date

    @classmethod
    def from_raw_values(
        cls,
        value: DateValue,
        min_value: NullableScalarDateValue,
        max_value: NullableScalarDateValue,
    ) -> _DateInputValues:
        parsed_value, is_range = _parse_date_value(value=value)
        parsed_min = _parse_min_date(
            min_value=min_value,
            parsed_dates=parsed_value,
        )
        parsed_max = _parse_max_date(
            max_value=max_value,
            parsed_dates=parsed_value,
        )

        if value == "today":
            v = cast("list[date]", parsed_value)[0]
            if v < parsed_min:
                parsed_value = [parsed_min]
            if v > parsed_max:
                parsed_value = [parsed_max]

        return cls(
            value=parsed_value,
            is_range=is_range,
            min=parsed_min,
            max=parsed_max,
        )

    def __post_init__(self) -> None:
        if self.min > self.max:
            raise StreamlitAPIException(
                f"The `min_value`, set to {self.min}, shouldn't be larger "
                f"than the `max_value`, set to {self.max}."
            )

        if self.value:
            start_value = self.value[0]
            end_value = self.value[-1]

            if (start_value < self.min) or (end_value > self.max):
                raise StreamlitAPIException(
                    f"The default `value` of {self.value} "
                    f"must lie between the `min_value` of {self.min} "
                    f"and the `max_value` of {self.max}, inclusively."
                )


@dataclass
class DateTimeInputSerde:
    value: datetime | None
    min: datetime
    max: datetime

    def deserialize(self, ui_value: list[str] | None) -> datetime | None:
        if ui_value is not None and len(ui_value) > 0:
            for fmt in (_DATETIME_ISO_FORMAT, _DATETIME_LEGACY_FORMAT):
                try:
                    deserialized = _normalize_datetime_value(
                        datetime.strptime(ui_value[0], fmt)
                    )
                    break
                except ValueError:
                    continue
            else:
                # Unparseable URL query param value — revert to default.
                return self.value
            if deserialized < self.min or deserialized > self.max:
                return self.value
            return deserialized
        return self.value

    def serialize(self, v: datetime | None) -> list[str]:
        if v is None:
            return []
        return [_datetime_to_proto_string(v)]


@dataclass
class TimeInputSerde:
    value: time | None
    step: int = 900

    def deserialize(self, ui_value: str | None) -> time | None:
        if ui_value is None:
            return self.value
        try:
            # TODO(query-params): URL values that don't align to the step
            # (e.g., ?time=14:37 with step=900) are accepted as-is.
            # Consider snapping to the nearest valid step for consistency
            # with the UI. See also SliderSerde.deserialize.
            return datetime.strptime(ui_value, "%H:%M").time()
        except ValueError:
            # Unparseable URL query param value — revert to default.
            return self.value

    def serialize(self, v: datetime | time | None) -> str | None:
        if v is None:
            return None
        if isinstance(v, datetime):
            v = v.time()
        return time.strftime(v, "%H:%M")


def _to_date(v: date) -> date:
    """Convert datetime to date for comparison.

    st.date_input can receive datetime values from session_state. Since datetime
    is a subclass of date, isinstance(v, date) returns True, but datetime and date
    objects cannot be directly compared with < or >. This helper normalizes the
    value for safe comparison with date bounds.
    """
    return v.date() if isinstance(v, datetime) else v


def _validate_date_value(
    current_value: DateWidgetReturn,
    parsed_values: _DateInputValues,
    has_explicit_bounds: bool,
) -> tuple[DateWidgetReturn, bool]:
    """Validate current value against min/max bounds and reset if needed.

    Only validates when has_explicit_bounds is True (user provided min_value or max_value).
    This avoids incorrectly resetting values against computed default bounds.

    Parameters
    ----------
    current_value : DateWidgetReturn
        The current value of the date input widget. Can be a single date, a tuple of
        dates (for range mode), or None.
    parsed_values : _DateInputValues
        Parsed configuration containing min, max, default value, and whether the widget
        is in range mode.
    has_explicit_bounds : bool
        Whether the user explicitly provided min_value or max_value. If False, validation
        is skipped to avoid resetting against computed default bounds.

    Returns
    -------
    tuple[DateWidgetReturn, bool]
        A tuple of (validated_value, was_reset) where validated_value is either the
        original value (if valid) or the default value (if reset was needed), and
        was_reset indicates whether a reset occurred.
    """
    value_needs_reset = False

    if current_value is None or not has_explicit_bounds:
        return current_value, value_needs_reset

    # For range inputs, current_value is a tuple; for single inputs, it's a date
    if (
        parsed_values.is_range
        and isinstance(current_value, tuple)
        and len(current_value) > 0
    ):
        # For range mode, check if any date in the tuple is outside bounds.
        # Cast to tuple[date, ...] to satisfy the type checker after the length check.
        non_empty_value = cast("tuple[date, ...]", current_value)
        start_date = non_empty_value[0]
        end_date = non_empty_value[-1] if len(non_empty_value) > 1 else start_date
        if (
            _to_date(start_date) < parsed_values.min
            or _to_date(end_date) > parsed_values.max
        ):
            value_needs_reset = True
    elif not parsed_values.is_range and isinstance(current_value, date):
        # For single date mode. Use _to_date to handle datetime values from session_state.
        if (
            _to_date(current_value) < parsed_values.min
            or _to_date(current_value) > parsed_values.max
        ):
            value_needs_reset = True
    else:
        # Type mismatch: widget mode doesn't match current value type (e.g., range mode
        # with a single date value or single mode with a tuple). Reset to match the mode.
        value_needs_reset = True

    if not value_needs_reset:
        return current_value, value_needs_reset

    # Reset to the default value from parsed_values
    if parsed_values.value is None or len(parsed_values.value) == 0:
        return (() if parsed_values.is_range else None), True
    if not parsed_values.is_range:
        return parsed_values.value[0], True
    return cast("DateWidgetReturn", tuple(parsed_values.value)), True


def _validate_datetime_value(
    current_value: datetime | None,
    parsed_values: _DateTimeInputValues,
    has_explicit_bounds: bool,
) -> tuple[datetime | None, bool]:
    """Validate current datetime value against min/max bounds and determine if reset is needed.

    Only validates when has_explicit_bounds is True (user provided min_value or max_value).
    This avoids incorrectly determining if reset is needed against computed default bounds.

    Parameters
    ----------
    current_value : datetime | None
        The current value of the datetime input widget.
    parsed_values : _DateTimeInputValues
        Parsed configuration containing min, max, and default value.
    has_explicit_bounds : bool
        Whether the user explicitly provided min_value or max_value. If False, validation
        is skipped to avoid resetting against computed default bounds.

    Returns
    -------
    tuple[datetime | None, bool]
        A tuple of (validated_value, was_reset) where validated_value is either the
        original value (if valid) or the default value (if reset was needed), and
        was_reset indicates whether a reset occurred.
    """
    if current_value is None or not has_explicit_bounds:
        return current_value, False

    if current_value < parsed_values.min or current_value > parsed_values.max:
        return parsed_values.value, True

    return current_value, False


@dataclass
class DateInputSerde:
    value: _DateInputValues

    @staticmethod
    def _parse_date(value: str) -> date:
        """Parse a date string in ISO (YYYY-MM-DD) or legacy (YYYY/MM/DD) format."""
        for fmt in ("%Y-%m-%d", "%Y/%m/%d"):
            try:
                return datetime.strptime(value, fmt).date()
            except ValueError:  # noqa: PERF203
                continue
        raise ValueError(f"Unable to parse date: {value}")

    def deserialize(self, ui_value: Any) -> DateWidgetReturn:
        return_value: Sequence[date] | None
        if ui_value is not None:
            try:
                return_value = tuple(self._parse_date(v) for v in ui_value)
            except ValueError:
                # Invalid URL query param value (e.g. "not-a-date") — revert to default.
                return_value = self.value.value
            else:
                # Reject out-of-range dates from URL query params — revert to default.
                # Matches SliderSerde.deserialize which validates bounds in the
                # deserializer so _seed_widget_from_url can detect default equality
                # and clear the URL param via _clear_url_param.
                if return_value and any(
                    d < self.value.min or d > self.value.max for d in return_value
                ):
                    return_value = self.value.value
        else:
            return_value = self.value.value

        if return_value is None or len(return_value) == 0:
            return () if self.value.is_range else None

        if not self.value.is_range:
            return return_value[0]
        return cast("DateWidgetReturn", tuple(return_value))

    def serialize(self, v: DateWidgetReturn) -> list[str]:
        if v is None:
            return []

        to_serialize = list(v) if isinstance(v, Sequence) else [v]
        return [date.strftime(v, "%Y-%m-%d") for v in to_serialize]


class TimeWidgetsMixin:
    @overload
    def time_input(
        self,
        label: str,
        value: TimeValue = "now",
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> time:
        pass

    @overload
    def time_input(
        self,
        label: str,
        value: None = None,
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> time | None:
        pass

    @gather_metrics("time_input")
    def time_input(
        self,
        label: str,
        value: TimeValue | None = "now",
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> time | None:
        r"""Display a time input widget.

        Parameters
        ----------
        label : str
            A short label explaining to the user what this time input is for.
            The label can optionally contain GitHub-flavored Markdown of the
            following types: Bold, Italics, Strikethroughs, Inline Code, Links,
            and Images. Images display like icons, with a max height equal to
            the font height.

            Unsupported Markdown elements are unwrapped so only their children
            (text contents) render. Common block-level Markdown (headings,
            lists, blockquotes) is automatically escaped and displays as
            literal text in labels.

            See the ``body`` parameter of |st.markdown|_ for additional,
            supported Markdown directives.

            For accessibility reasons, you should never set an empty label, but
            you can hide it with ``label_visibility`` if needed. In the future,
            we may disallow empty labels by raising an exception.

            .. |st.markdown| replace:: ``st.markdown``
            .. _st.markdown: https://docs.streamlit.io/develop/api-reference/text/st.markdown

        value : "now", datetime.time, datetime.datetime, str, or None
            The value of this widget when it first renders. This can be one of
            the following:

            - ``"now"`` (default): The widget initializes with the current time.
            - A ``datetime.time`` or ``datetime.datetime`` object: The widget
              initializes with the given time, ignoring any date if included.
            - An ISO-formatted time (hh:mm[:ss.sss]) or datetime
              (YYYY-MM-DD hh:mm[:ss]) string: The widget initializes with the
              given time, ignoring any date if included.
            - ``None``: The widget initializes with no time and returns
              ``None`` until the user selects a time.

        key : str, int, or None
            An optional string or integer to use as the unique key for
            the widget. If this is ``None`` (default), a key will be
            generated for the widget based on the values of the other
            parameters. No two widgets may have the same key. Assigning
            a key stabilizes the widget's identity and preserves its
            state across reruns even when other parameters change.

            .. note::
               Changing ``step`` resets the widget even when a key is
               provided, because it constrains valid values.

            A key lets you read or update the widget's value via
            ``st.session_state[key]``. For more details, see `Widget
            behavior <https://docs.streamlit.io/develop/concepts/architecture/widget-behavior>`_.

            Additionally, if ``key`` is provided, it will be used as a
            CSS class name prefixed with ``st-key-``.

        help : str or None
            A tooltip that gets displayed next to the widget label. Streamlit
            only displays the tooltip when ``label_visibility="visible"``. If
            this is ``None`` (default), no tooltip is displayed.

            The tooltip can optionally contain GitHub-flavored Markdown,
            including the Markdown directives described in the ``body``
            parameter of ``st.markdown``.

        on_change : callable
            An optional callback invoked when this time_input's value changes.

        args : list or tuple
            An optional list or tuple of args to pass to the callback.

        kwargs : dict
            An optional dict of kwargs to pass to the callback.

        disabled : bool
            An optional boolean that disables the time input if set to
            ``True``. The default is ``False``.

        label_visibility : "visible", "hidden", or "collapsed"
            The visibility of the label. The default is ``"visible"``. If this
            is ``"hidden"``, Streamlit displays an empty spacer instead of the
            label, which can help keep the widget aligned with other widgets.
            If this is ``"collapsed"``, Streamlit displays no label or spacer.

        step : int or timedelta
            The stepping interval in seconds. This defaults to ``900`` (15
            minutes). You can also pass a ``datetime.timedelta`` object. The
            value must be between 60 seconds and 23 hours.

        width : "stretch" or int
            The width of the time input widget. This can be one of the following:

            - ``"stretch"`` (default): The width of the widget matches the
              width of the parent container.
            - An integer specifying the width in pixels: The widget has a
              fixed width. If the specified width is greater than the width of
              the parent container, the width of the widget matches the width
              of the parent container.

        bind : "query-params" or None
            Binding mode for syncing the widget's value with a URL query
            parameter. If this is ``None`` (default), the widget's value
            is not synced to the URL. When this is set to
            ``"query-params"``, changes to the widget update the URL, and
            the widget can be initialized or updated through a query
            parameter in the URL. This requires ``key`` to be set. The
            key is used as the query parameter name.

            When the widget's value equals its default, the query
            parameter is removed from the URL to keep it clean. A bound
            query parameter can't be set or deleted through
            ``st.query_params``; it can only be programmatically changed
            through ``st.session_state``.

            Times use HH:MM format in the URL. Invalid query parameter
            values are ignored and removed from the URL. If ``value``
            is ``None``, an empty query parameter (e.g., ``?my_key=``)
            clears the widget.

        Returns
        -------
        datetime.time or None
            The current value of the time input widget or ``None`` if no time has been
            selected.

        Example
        -------
        **Example 1: Basic usage**

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> t = st.time_input("Set an alarm for", datetime.time(8, 45))
        >>> st.write("Alarm is set for", t)

        .. output::
           https://doc-time-input.streamlit.app/
           height: 260px

        **Example 2: Empty initial value**

        To initialize an empty time input, use ``None`` as the value:

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> t = st.time_input("Set an alarm for", value=None)
        >>> st.write("Alarm is set for", t)

        .. output::
           https://doc-time-input-empty.streamlit.app/
           height: 260px

        """
        ctx = get_script_run_ctx()
        return self._time_input(
            label=label,
            value=value,
            key=key,
            help=help,
            on_change=on_change,
            args=args,
            kwargs=kwargs,
            disabled=disabled,
            label_visibility=label_visibility,
            step=step,
            width=width,
            bind=bind,
            ctx=ctx,
        )

    def _time_input(
        self,
        label: str,
        value: TimeValue | None = "now",
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
        ctx: ScriptRunContext | None = None,
    ) -> time | None:
        key = to_key(key)

        check_widget_policies(
            self.dg,
            key,
            on_change,
            default_value=value if value != "now" else None,
        )
        maybe_raise_label_warnings(label, label_visibility)

        parsed_time: time | None
        parsed_time = None if value is None else _convert_timelike_to_time(value)

        element_id = compute_and_register_element_id(
            "time_input",
            user_key=key,
            # Ensure stable ID when key is provided; only whitelist step since it
            # affects the selection granularity and available options.
            key_as_main_identity={"step"},
            dg=self.dg,
            label=label,
            value=parsed_time if isinstance(value, (datetime, time)) else value,
            help=help,
            step=step,
            width=width,
        )
        del value

        session_state = get_session_state().filtered_state
        if key is not None and key in session_state and session_state[key] is None:
            parsed_time = None

        time_input_proto = TimeInputProto()
        time_input_proto.id = element_id
        time_input_proto.label = label
        if parsed_time is not None:
            time_input_proto.default = time.strftime(parsed_time, "%H:%M")
        time_input_proto.form_id = current_form_id(self.dg)
        if not isinstance(step, (int, timedelta)):
            raise StreamlitAPIException(
                f"`step` can only be `int` or `timedelta` but {type(step)} is provided."
            )
        if isinstance(step, timedelta):
            step = step.seconds
        if step < 60 or step > timedelta(hours=23).seconds:
            raise StreamlitAPIException(
                f"`step` must be between 60 seconds and 23 hours but is currently set to {step} seconds."
            )
        time_input_proto.step = step
        time_input_proto.disabled = disabled
        time_input_proto.label_visibility.value = get_label_visibility_proto_value(
            label_visibility
        )

        if help is not None:
            time_input_proto.help = dedent(help)

        if bind == "query-params" and key is not None:
            time_input_proto.query_param_key = str(key)

        serde = TimeInputSerde(parsed_time, step=step)
        widget_state = register_widget(
            time_input_proto.id,
            on_change_handler=on_change,
            args=args,
            kwargs=kwargs,
            deserializer=serde.deserialize,
            serializer=serde.serialize,
            ctx=ctx,
            value_type="string_value",
            bind=bind,
            clearable=(parsed_time is None),
        )

        if widget_state.value_changed:
            if (serialized_value := serde.serialize(widget_state.value)) is not None:
                time_input_proto.value = serialized_value
            time_input_proto.set_value = True

        validate_width(width)
        layout_config = LayoutConfig(width=width)

        self.dg._enqueue("time_input", time_input_proto, layout_config=layout_config)
        return widget_state.value

    @overload
    def datetime_input(
        self,
        label: str,
        value: None,
        min_value: DateTimeValue = None,
        max_value: DateTimeValue = None,
        *,  # keyword-only arguments:
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        format: str = "YYYY/MM/DD",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> datetime | None: ...

    @overload
    def datetime_input(
        self,
        label: str,
        value: DateTimeScalarValue = "now",
        min_value: DateTimeValue = None,
        max_value: DateTimeValue = None,
        *,  # keyword-only arguments:
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        format: str = "YYYY/MM/DD",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> datetime: ...

    @gather_metrics("datetime_input")
    def datetime_input(
        self,
        label: str,
        value: DateTimeValue = "now",
        min_value: DateTimeValue = None,
        max_value: DateTimeValue = None,
        *,  # keyword-only arguments:
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        format: str = "YYYY/MM/DD",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> datetime | None:
        r"""Display a date and time input widget.

        Parameters
        ----------
        label : str
            A short label explaining to the user what this datetime input is for.
            The label can optionally contain GitHub-flavored Markdown of the
            following types: Bold, Italics, Strikethroughs, Inline Code, Links,
            and Images. Images display like icons, with a max height equal to
            the font height.

            Unsupported Markdown elements are unwrapped so only their children
            (text contents) render. Common block-level Markdown (headings,
            lists, blockquotes) is automatically escaped and displays as
            literal text in labels.

            See the ``body`` parameter of |st.markdown|_ for additional,
            supported Markdown directives.

            For accessibility reasons, you should never set an empty label, but
            you can hide it with ``label_visibility`` if needed. In the future,
            we may disallow empty labels by raising an exception.

            .. |st.markdown| replace:: ``st.markdown``
            .. _st.markdown: https://docs.streamlit.io/develop/api-reference/text/st.markdown

        value : "now", datetime.datetime, datetime.date, datetime.time, str, or None
            The value of this widget when it first renders. This can be one of
            the following:

            - ``"now"`` (default): The widget initializes with the current date and time.
            - A ``datetime.datetime`` object: The widget initializes with the given
              datetime, stripping any timezone information.
            - A ``datetime.date`` object: The widget initializes with the given date
              at 00:00.
            - A ``datetime.time`` object: The widget initializes with today's date
              and the provided time.
            - An ISO-formatted datetime (YYYY-MM-DD hh:mm[:ss]) or date/time
              string: The widget initializes with the parsed value.
            - ``None``: The widget initializes with no value and returns ``None``
              until the user selects a datetime.

        min_value : "now", datetime.datetime, datetime.date, datetime.time, str, or None
            The minimum selectable datetime. This can be any of the datetime
            types accepted by ``value``.

            If this is ``None`` (default), the minimum selectable datetime is
            ten years before the initial value. If no initial value is set, the
            minimum selectable datetime is ten years before today at 00:00.

        max_value : "now", datetime.datetime, datetime.date, datetime.time, str, or None
            The maximum selectable datetime. This can be any of the datetime
            types accepted by ``value``.

            If this is ``None`` (default), the maximum selectable datetime is
            ten years after the initial value. If no initial value is set, the
            maximum selectable datetime is ten years after today at 23:59.

        key : str, int, or None
            An optional string or integer to use as the unique key for
            the widget. If this is ``None`` (default), a key will be
            generated for the widget based on the values of the other
            parameters. No two widgets may have the same key. Assigning
            a key stabilizes the widget's identity and preserves its
            state across reruns even when other parameters change.

            .. note::
               Changing ``format`` or ``step`` resets the widget even
               when a key is provided.

            A key lets you read or update the widget's value via
            ``st.session_state[key]``. For more details, see `Widget
            behavior <https://docs.streamlit.io/develop/concepts/architecture/widget-behavior>`_.

            Additionally, if ``key`` is provided, it will be used as a
            CSS class name prefixed with ``st-key-``.

        help : str or None
            A tooltip that gets displayed next to the widget label. Streamlit
            only displays the tooltip when ``label_visibility="visible"``. If
            this is ``None`` (default), no tooltip is displayed.

            The tooltip can optionally contain GitHub-flavored Markdown,
            including the Markdown directives described in the ``body``
            parameter of ``st.markdown``.

        on_change : callable
            An optional callback invoked when this datetime_input's value changes.

        args : list or tuple
            An optional list or tuple of args to pass to the callback.

        kwargs : dict
            An optional dict of kwargs to pass to the callback.

        format : str
            A format string controlling how the interface displays dates.
            Supports ``"YYYY/MM/DD"`` (default), ``"DD/MM/YYYY"``, or ``"MM/DD/YYYY"``.
            You may also use a period (.) or hyphen (-) as separators. This
            doesn't affect the time format.

        step : int or timedelta
            The stepping interval in seconds. This defaults to ``900`` (15
            minutes). You can also pass a ``datetime.timedelta`` object. The
            value must be between 60 seconds and 23 hours.

        disabled : bool
            An optional boolean that disables the widget if set to ``True``.
            The default is ``False``.

        label_visibility : "visible", "hidden", or "collapsed"
            The visibility of the label. The default is ``"visible"``. If this
            is ``"hidden"``, Streamlit displays an empty spacer instead of the
            label, which can help keep the widget aligned with other widgets.
            If this is ``"collapsed"``, Streamlit displays no label or spacer.

        width : "stretch" or int
            The width of the widget. This can be one of the following:

            - ``"stretch"`` (default): The width of the widget matches the width
              of the parent container.
            - An integer specifying the width in pixels: The widget has a fixed
              width. If the specified width is greater than the width of the
              parent container, the widget matches the container width.

        bind : "query-params" or None
            Binding mode for syncing the widget's value with a URL query
            parameter. If this is ``None`` (default), the widget's value
            is not synced to the URL. When this is set to
            ``"query-params"``, changes to the widget update the URL, and
            the widget can be initialized or updated through a query
            parameter in the URL. This requires ``key`` to be set. The
            key is used as the query parameter name.

            When the widget's value equals its default, the query
            parameter is removed from the URL to keep it clean. A bound
            query parameter can't be set or deleted through
            ``st.query_params``; it can only be programmatically changed
            through ``st.session_state``.

            Datetimes use ISO 8601 format (YYYY-MM-DDThh:mm) in the URL.
            Invalid query parameter values are ignored and removed from
            the URL. If ``value`` is ``None``, an empty query parameter
            (e.g., ``?my_key=``) clears the widget.

        Returns
        -------
        datetime.datetime or None
            The current value of the datetime input widget (without timezone)
            or ``None`` if no value has been selected.

        Examples
        --------
        **Example 1: Basic usage**

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> event_time = st.datetime_input(
        ...     "Schedule your event",
        ...     datetime.datetime(2025, 11, 19, 16, 45),
        ... )
        >>> st.write("Event scheduled for", event_time)

        .. output::
           https://doc-datetime-input.streamlit.app/
           height: 500px

        **Example 2: Empty initial value**

        To initialize an empty datetime input, use ``None`` as the value:

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> event_time = st.datetime_input("Schedule your event", value=None)
        >>> st.write("Event scheduled for", event_time)

        .. output::
           https://doc-datetime-input-empty.streamlit.app/
           height: 500px

        """
        ctx = get_script_run_ctx()
        return self._datetime_input(
            label=label,
            value=value,
            min_value=min_value,
            max_value=max_value,
            key=key,
            help=help,
            on_change=on_change,
            args=args,
            kwargs=kwargs,
            format=format,
            step=step,
            disabled=disabled,
            label_visibility=label_visibility,
            width=width,
            bind=bind,
            ctx=ctx,
        )

    def _datetime_input(
        self,
        label: str,
        value: DateTimeValue = "now",
        min_value: DateTimeValue = None,
        max_value: DateTimeValue = None,
        *,  # keyword-only arguments:
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        format: str = "YYYY/MM/DD",
        step: int | timedelta = timedelta(minutes=DEFAULT_STEP_MINUTES),
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
        ctx: ScriptRunContext | None = None,
    ) -> datetime | None:
        key = to_key(key)

        check_widget_policies(
            self.dg,
            key,
            on_change,
            default_value=value if value != "now" else None,
        )
        maybe_raise_label_warnings(label, label_visibility)

        datetime_values = _DateTimeInputValues.from_raw_values(
            value=value,
            min_value=min_value,
            max_value=max_value,
        )

        default_value = datetime_values.value
        min_value_proto = _datetime_to_proto_string(datetime_values.min)
        max_value_proto = _datetime_to_proto_string(datetime_values.max)

        if isinstance(value, (datetime, date, time)):
            value_for_id: Any = (
                None
                if default_value is None
                else _datetime_to_proto_string(default_value)
            )
        else:
            value_for_id = value

        element_id = compute_and_register_element_id(
            "date_time_input",
            user_key=key,
            # Format is whitelisted because of a bug in the BaseWeb date input component.
            # Step is whitelisted because it invalidates the current selection.
            # We might be able to unlock this as a follow-up.
            key_as_main_identity={"format", "step"},
            dg=self.dg,
            label=label,
            value=value_for_id,
            min_value=min_value_proto,
            max_value=max_value_proto,
            help=help,
            format=format,
            step=step,
            width=width,
        )
        # Track if user explicitly set bounds (before del)
        has_explicit_bounds = min_value is not None or max_value is not None
        del value, min_value, max_value

        if not bool(ALLOWED_DATE_FORMATS.match(format)):
            raise StreamlitAPIException(
                f"The provided format (`{format}`) is not valid. DateTimeInput format "
                "should be one of `YYYY/MM/DD`, `DD/MM/YYYY`, or `MM/DD/YYYY` "
                "and can also use a period (.) or hyphen (-) as separators."
            )

        if not isinstance(step, (int, timedelta)):
            raise StreamlitAPIException(
                f"`step` can only be `int` or `timedelta` but {type(step)} is provided."
            )
        step_seconds = (
            int(step.total_seconds()) if isinstance(step, timedelta) else step
        )
        if step_seconds < 60 or step_seconds > timedelta(hours=23).seconds:
            raise StreamlitAPIException(
                f"`step` must be between 60 seconds and 23 hours but is currently set to {step_seconds} seconds."
            )

        session_state = get_session_state().filtered_state
        default_value_for_proto = default_value
        if key is not None and key in session_state and session_state[key] is None:
            default_value_for_proto = None

        date_time_input_proto = DateTimeInputProto()
        date_time_input_proto.id = element_id
        date_time_input_proto.label = label
        if default_value_for_proto is not None:
            date_time_input_proto.default[:] = [
                _datetime_to_proto_string(default_value_for_proto)
            ]
        date_time_input_proto.min = min_value_proto
        date_time_input_proto.max = max_value_proto
        date_time_input_proto.form_id = current_form_id(self.dg)
        date_time_input_proto.step = step_seconds
        date_time_input_proto.disabled = disabled
        date_time_input_proto.label_visibility.value = get_label_visibility_proto_value(
            label_visibility
        )
        date_time_input_proto.format = format
        date_time_input_proto.is_range = False

        if help is not None:
            date_time_input_proto.help = dedent(help)

        if bind == "query-params" and key is not None:
            date_time_input_proto.query_param_key = str(key)

        serde = DateTimeInputSerde(
            value=default_value_for_proto,
            min=datetime_values.min,
            max=datetime_values.max,
        )
        widget_state = register_widget(
            date_time_input_proto.id,
            on_change_handler=on_change,
            args=args,
            kwargs=kwargs,
            deserializer=serde.deserialize,
            serializer=serde.serialize,
            ctx=ctx,
            value_type="string_array_value",
            bind=bind,
            clearable=(default_value is None),
        )

        # Validate the current value against the new min/max bounds.
        # Only validate when user explicitly provided min_value or max_value.
        current_value, value_needs_reset = _validate_datetime_value(
            widget_state.value, datetime_values, has_explicit_bounds
        )

        if value_needs_reset and key is not None:
            # Update session_state so subsequent accesses in this run
            # return the corrected value. Use reset_state_value to avoid
            # the "cannot be modified after widget instantiated" error.
            get_session_state().reset_state_value(key, current_value)

            # Clear stale URL param when an out-of-bounds URL value was reset.
            if bind == "query-params":
                with get_session_state().query_params() as qp:
                    qp.remove_param(str(key))

        if value_needs_reset or widget_state.value_changed:
            date_time_input_proto.value[:] = serde.serialize(current_value)
            date_time_input_proto.set_value = True

        validate_width(width)
        layout_config = LayoutConfig(width=width)

        self.dg._enqueue(
            "date_time_input", date_time_input_proto, layout_config=layout_config
        )
        return current_value

    @overload
    def date_input(
        self,
        label: str,
        value: date | datetime | str | Literal["today"] = "today",
        min_value: NullableScalarDateValue = None,
        max_value: NullableScalarDateValue = None,
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        format: str = "YYYY/MM/DD",
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> date: ...

    @overload
    def date_input(
        self,
        label: str,
        value: None,
        min_value: NullableScalarDateValue = None,
        max_value: NullableScalarDateValue = None,
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        format: str = "YYYY/MM/DD",
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> date | None: ...

    @overload
    def date_input(
        self,
        label: str,
        value: tuple[NullableScalarDateValue]
        | tuple[NullableScalarDateValue, NullableScalarDateValue]
        | list[NullableScalarDateValue],
        min_value: NullableScalarDateValue = None,
        max_value: NullableScalarDateValue = None,
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        format: str = "YYYY/MM/DD",
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> DateWidgetRangeReturn: ...

    @gather_metrics("date_input")
    def date_input(
        self,
        label: str,
        value: DateValue = "today",
        min_value: NullableScalarDateValue = None,
        max_value: NullableScalarDateValue = None,
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        format: str = "YYYY/MM/DD",
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
    ) -> DateWidgetReturn:
        r"""Display a date input widget.

        The date input widget can be configured to accept a single date or a
        date range. The first day of the week is determined from the user's
        locale in their browser.

        Parameters
        ----------
        label : str
            A short label explaining to the user what this date input is for.
            The label can optionally contain GitHub-flavored Markdown of the
            following types: Bold, Italics, Strikethroughs, Inline Code, Links,
            and Images. Images display like icons, with a max height equal to
            the font height.

            Unsupported Markdown elements are unwrapped so only their children
            (text contents) render. Common block-level Markdown (headings,
            lists, blockquotes) is automatically escaped and displays as
            literal text in labels.

            See the ``body`` parameter of |st.markdown|_ for additional,
            supported Markdown directives.

            For accessibility reasons, you should never set an empty label, but
            you can hide it with ``label_visibility`` if needed. In the future,
            we may disallow empty labels by raising an exception.

            .. |st.markdown| replace:: ``st.markdown``
            .. _st.markdown: https://docs.streamlit.io/develop/api-reference/text/st.markdown

        value : "today", datetime.date, datetime.datetime, str, list/tuple of these, or None
            The value of this widget when it first renders. This can be one of
            the following:

            - ``"today"`` (default): The widget initializes with the current date.
            - A ``datetime.date`` or ``datetime.datetime`` object: The widget
              initializes with the given date, ignoring any time if included.
            - An ISO-formatted date (YYYY-MM-DD) or datetime
              (YYYY-MM-DD hh:mm:ss) string: The widget initializes with the
              given date, ignoring any time if included.
            - A list or tuple with up to two of the above: The widget will
              initialize with the given date interval and return a tuple of the
              selected interval. You can pass an empty list to initialize the
              widget with an empty interval or a list with one value to
              initialize only the beginning date of the iterval.
            - ``None``: The widget initializes with no date and returns
              ``None`` until the user selects a date.

        min_value : "today", datetime.date, datetime.datetime, str, or None
            The minimum selectable date. This can be any of the date types
            accepted by ``value``, except list or tuple.

            If this is ``None`` (default), the minimum selectable date is ten
            years before the initial value. If the initial value is an
            interval, the minimum selectable date is ten years before the start
            date of the interval. If no initial value is set, the minimum
            selectable date is ten years before today.

        max_value : "today", datetime.date, datetime.datetime, str, or None
            The maximum selectable date. This can be any of the date types
            accepted by ``value``, except list or tuple.

            If this is ``None`` (default), the maximum selectable date is ten
            years after the initial value. If the initial value is an interval,
            the maximum selectable date is ten years after the end date of the
            interval. If no initial value is set, the maximum selectable date
            is ten years after today.

        key : str, int, or None
            An optional string or integer to use as the unique key for
            the widget. If this is ``None`` (default), a key will be
            generated for the widget based on the values of the other
            parameters. No two widgets may have the same key. Assigning
            a key stabilizes the widget's identity and preserves its
            state across reruns even when other parameters change.

            .. note::
               Changing ``format`` resets the widget even when a key is
               provided.

            A key lets you read or update the widget's value via
            ``st.session_state[key]``. For more details, see `Widget
            behavior <https://docs.streamlit.io/develop/concepts/architecture/widget-behavior>`_.

            Additionally, if ``key`` is provided, it will be used as a
            CSS class name prefixed with ``st-key-``.

        help : str or None
            A tooltip that gets displayed next to the widget label. Streamlit
            only displays the tooltip when ``label_visibility="visible"``. If
            this is ``None`` (default), no tooltip is displayed.

            The tooltip can optionally contain GitHub-flavored Markdown,
            including the Markdown directives described in the ``body``
            parameter of ``st.markdown``.

        on_change : callable
            An optional callback invoked when this date_input's value changes.

        args : list or tuple
            An optional list or tuple of args to pass to the callback.

        kwargs : dict
            An optional dict of kwargs to pass to the callback.

        format : str
            A format string controlling how the interface should display dates.
            Supports ``"YYYY/MM/DD"`` (default), ``"DD/MM/YYYY"``, or ``"MM/DD/YYYY"``.
            You may also use a period (.) or hyphen (-) as separators.

        disabled : bool
            An optional boolean that disables the date input if set to
            ``True``. The default is ``False``.

        label_visibility : "visible", "hidden", or "collapsed"
            The visibility of the label. The default is ``"visible"``. If this
            is ``"hidden"``, Streamlit displays an empty spacer instead of the
            label, which can help keep the widget aligned with other widgets.
            If this is ``"collapsed"``, Streamlit displays no label or spacer.

        width : "stretch" or int
            The width of the date input widget. This can be one of the following:

            - ``"stretch"`` (default): The width of the widget matches the
              width of the parent container.
            - An integer specifying the width in pixels: The widget has a
              fixed width. If the specified width is greater than the width of
              the parent container, the width of the widget matches the width
              of the parent container.

        bind : "query-params" or None
            Binding mode for syncing the widget's value with a URL query
            parameter. If this is ``None`` (default), the widget's value
            is not synced to the URL. When this is set to
            ``"query-params"``, changes to the widget update the URL, and
            the widget can be initialized or updated through a query
            parameter in the URL. This requires ``key`` to be set. The
            key is used as the query parameter name.

            When the widget's value equals its default, the query
            parameter is removed from the URL to keep it clean. A bound
            query parameter can't be set or deleted through
            ``st.query_params``; it can only be programmatically changed
            through ``st.session_state``.

            Dates use ISO 8601 format (YYYY-MM-DD) in the URL. Invalid
            query parameter values are ignored and removed from the URL.
            If ``value`` is ``None``, an empty query parameter (e.g.,
            ``?vacation=``) clears the widget. Date ranges use repeated
            parameters (e.g.,
            ``?vacation=2025-01-01&vacation=2025-01-31``).

        Returns
        -------
        datetime.date or a tuple with 0-2 dates or None
            The current value of the date input widget or ``None`` if no date has been
            selected.

        Examples
        --------
        **Example 1: Basic usage**

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> d = st.date_input("When's your birthday", datetime.date(2019, 7, 6))
        >>> st.write("Your birthday is:", d)

        .. output::
           https://doc-date-input.streamlit.app/
           height: 380px

        **Example 2: Date range**

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> today = datetime.datetime.now()
        >>> next_year = today.year + 1
        >>> jan_1 = datetime.date(next_year, 1, 1)
        >>> dec_31 = datetime.date(next_year, 12, 31)
        >>>
        >>> d = st.date_input(
        ...     "Select your vacation for next year",
        ...     (jan_1, datetime.date(next_year, 1, 7)),
        ...     jan_1,
        ...     dec_31,
        ...     format="MM.DD.YYYY",
        ... )
        >>> d

        .. output::
           https://doc-date-input1.streamlit.app/
           height: 380px

        **Example 3: Empty initial value**

        To initialize an empty date input, use ``None`` as the value:

        >>> import datetime
        >>> import streamlit as st
        >>>
        >>> d = st.date_input("When's your birthday", value=None)
        >>> st.write("Your birthday is:", d)

        .. output::
           https://doc-date-input-empty.streamlit.app/
           height: 380px

        """
        ctx = get_script_run_ctx()
        return self._date_input(
            label=label,
            value=value,
            min_value=min_value,
            max_value=max_value,
            key=key,
            help=help,
            on_change=on_change,
            args=args,
            kwargs=kwargs,
            disabled=disabled,
            label_visibility=label_visibility,
            format=format,
            width=width,
            bind=bind,
            ctx=ctx,
        )

    def _date_input(
        self,
        label: str,
        value: DateValue = "today",
        min_value: NullableScalarDateValue = None,
        max_value: NullableScalarDateValue = None,
        key: Key | None = None,
        help: str | None = None,
        on_change: WidgetCallback | None = None,
        args: WidgetArgs | None = None,
        kwargs: WidgetKwargs | None = None,
        *,  # keyword-only arguments:
        format: str = "YYYY/MM/DD",
        disabled: bool = False,
        label_visibility: LabelVisibility = "visible",
        width: WidthWithoutContent = "stretch",
        bind: BindOption = None,
        ctx: ScriptRunContext | None = None,
    ) -> DateWidgetReturn:
        key = to_key(key)

        check_widget_policies(
            self.dg,
            key,
            on_change,
            default_value=value if value != "today" else None,
        )
        maybe_raise_label_warnings(label, label_visibility)

        def parse_date_deterministic_for_id(v: NullableScalarDateValue) -> str | None:
            if v == "today":
                # For ID purposes, no need to parse the input string.
                return None
            if isinstance(v, str):
                # For ID purposes, no need to parse the input string.
                return v
            if isinstance(v, datetime):
                return date.strftime(v.date(), "%Y-%m-%d")
            if isinstance(v, date):
                return date.strftime(v, "%Y-%m-%d")

            return None

        parsed_min_date = parse_date_deterministic_for_id(min_value)
        parsed_max_date = parse_date_deterministic_for_id(max_value)

        parsed: str | list[str | None] | None
        if value == "today":
            parsed = None
        elif isinstance(value, Sequence):
            parsed = [parse_date_deterministic_for_id(v) for v in value]  # ty: ignore[invalid-argument-type]
        else:
            parsed = parse_date_deterministic_for_id(value)

        # TODO: this is missing the error path, integrate with the dateinputvalues parsing

        element_id = compute_and_register_element_id(
            "date_input",
            user_key=key,
            # Ensure stable ID when key is provided. Only format is whitelisted because
            # there is a bug in baseweb where changing the format dynamically leads to
            # a wrongly formatted date. min_value and max_value support dynamic changes.
            key_as_main_identity={"format"},
            dg=self.dg,
            label=label,
            value=parsed,
            min_value=parsed_min_date,
            max_value=parsed_max_date,
            help=help,
            format=format,
            width=width,
        )
        if not bool(ALLOWED_DATE_FORMATS.match(format)):
            raise StreamlitAPIException(
                f"The provided format (`{format}`) is not valid. DateInput format "
                "should be one of `YYYY/MM/DD`, `DD/MM/YYYY`, or `MM/DD/YYYY` "
                "and can also use a period (.) or hyphen (-) as separators."
            )

        parsed_values = _DateInputValues.from_raw_values(
            value=value,
            min_value=min_value,
            max_value=max_value,
        )

        # Track if user explicitly set bounds (before del)
        has_explicit_bounds = min_value is not None or max_value is not None

        if value == "today":
            # We need to know if this is a single or range date_input, but don't have
            # a default value, so we check if session_state can tell us.
            # We already calculated the id, so there is no risk of this causing
            # the id to change.

            session_state = get_session_state().filtered_state

            if key is not None and key in session_state:
                state_value = session_state[key]
                parsed_values = _DateInputValues.from_raw_values(
                    value=state_value,
                    min_value=min_value,
                    max_value=max_value,
                )

        del value, min_value, max_value

        date_input_proto = DateInputProto()
        date_input_proto.id = element_id
        date_input_proto.is_range = parsed_values.is_range
        date_input_proto.disabled = disabled
        date_input_proto.label_visibility.value = get_label_visibility_proto_value(
            label_visibility
        )
        date_input_proto.format = format
        date_input_proto.label = label
        if parsed_values.value is None:
            # An empty array represents the empty state. The reason for using an empty
            # array here is that we cannot optional keyword for repeated fields
            # in protobuf.
            date_input_proto.default[:] = []
        else:
            date_input_proto.default[:] = [
                date.strftime(v, "%Y-%m-%d") for v in parsed_values.value
            ]
        date_input_proto.min = date.strftime(parsed_values.min, "%Y-%m-%d")
        date_input_proto.max = date.strftime(parsed_values.max, "%Y-%m-%d")
        date_input_proto.form_id = current_form_id(self.dg)

        if help is not None:
            date_input_proto.help = dedent(help)

        if bind == "query-params" and key is not None:
            date_input_proto.query_param_key = str(key)

        serde = DateInputSerde(parsed_values)

        widget_state = register_widget(
            date_input_proto.id,
            on_change_handler=on_change,
            args=args,
            kwargs=kwargs,
            deserializer=serde.deserialize,
            serializer=serde.serialize,
            ctx=ctx,
            value_type="string_array_value",
            bind=bind,
            clearable=(parsed_values.value is None),
        )

        # Validate the current value against the new min/max bounds.
        # Only validate when user explicitly provided min_value or max_value.
        current_value, value_needs_reset = _validate_date_value(
            widget_state.value, parsed_values, has_explicit_bounds
        )

        # Reset if needed.
        if value_needs_reset and key is not None:
            # Update session_state so subsequent accesses in this run
            # return the corrected value. Use reset_state_value to avoid
            # the "cannot be modified after widget instantiated" error.
            get_session_state().reset_state_value(key, current_value)

            # Clear stale URL param when an out-of-bounds URL value was reset.
            if bind == "query-params":
                with get_session_state().query_params() as qp:
                    qp.remove_param(str(key))

        if value_needs_reset or widget_state.value_changed:
            date_input_proto.value[:] = serde.serialize(current_value)
            date_input_proto.set_value = True

        validate_width(width)
        layout_config = LayoutConfig(width=width)

        self.dg._enqueue("date_input", date_input_proto, layout_config=layout_config)
        return current_value

    @property
    def dg(self) -> DeltaGenerator:
        """Get our DeltaGenerator."""
        return cast("DeltaGenerator", self)
