/*
 * Copyright (c) 2004-2005 The Trustees of Indiana University and Indiana
 *                         University Research and Technology
 *                         Corporation.  All rights reserved.
 * Copyright (c) 2004-2005 The University of Tennessee and The University
 *                         of Tennessee Research Foundation.  All rights
 *                         reserved.
 * Copyright (c) 2004-2005 High Performance Computing Center Stuttgart, 
 *                         University of Stuttgart.  All rights reserved.
 * Copyright (c) 2004-2005 The Regents of the University of California.
 *                         All rights reserved.
 * Copyright (c) 2011-2013 UT-Battelle, LLC. All rights reserved.
 * Copyright (C) 2013      Mellanox Technologies Ltd. All rights reserved.
 * $COPYRIGHT$
 * 
 * Additional copyrights may follow
 * 
 * $HEADER$
 *
 */

/** @file
 *
 *     A red black tree
 */

#ifndef OMPI_RB_TREE_H
#define OMPI_RB_TREE_H

#include "ocoms/platform/ocoms_config.h"
#include <stdlib.h>
#include "ocoms/platform/ocoms_constants.h"
#include "ocoms/util/ocoms_object.h"
#include "ocoms/util/ocoms_free_list.h"

BEGIN_C_DECLS
/*
 * Data structures and datatypes
 */

/**
  * red and black enum
  */
typedef enum {RED, BLACK} ocoms_rb_tree_nodecolor_t;

/**
  * node data structure
  */
struct ocoms_rb_tree_node_t
{
    ocoms_free_list_item_t super;             /**< the parent class */
    ocoms_rb_tree_nodecolor_t color;     /**< the node color */
    struct ocoms_rb_tree_node_t * parent;/**< the parent node, can be NULL */
    struct ocoms_rb_tree_node_t * left;  /**< the left child - can be nill */
    struct ocoms_rb_tree_node_t * right; /**< the right child - can be nill */
    void *key;                          /**< a pointer to the key */
    void *value;                        /**< a pointer to the value */
};
typedef struct ocoms_rb_tree_node_t ocoms_rb_tree_node_t;

/**
  * the compare function typedef. This function is used to compare 2 nodes.
  */
typedef int (*ocoms_rb_tree_comp_fn_t)(void *key1, void *key2);

/**
  * the data structure that holds all the needed information about the tree.
  */
struct ocoms_rb_tree_t {
    ocoms_object_t parent;           /**< the parent class */
    /* this root pointer doesn't actually point to the root of the tree.
     * rather, it points to a sentinal node who's left branch is the real
     * root of the tree. This is done to eliminate special cases */
    ocoms_rb_tree_node_t * root_ptr;/**< a pointer to the root of the tree */
    ocoms_rb_tree_node_t * nill;     /**< the nill sentinal node */
    ocoms_rb_tree_comp_fn_t comp;    /**< the compare function */
    ocoms_free_list_t free_list;   /**< the free list to get the memory from */
    size_t tree_size;                  /**< the size of the tree */
};
typedef struct ocoms_rb_tree_t ocoms_rb_tree_t;

/** declare the tree node as a class */
OCOMS_DECLSPEC OBJ_CLASS_DECLARATION(ocoms_rb_tree_node_t);
/** declare the tree as a class */
OCOMS_DECLSPEC OBJ_CLASS_DECLARATION(ocoms_rb_tree_t);

/* Function pointers for map traversal function */
/**
  * this function is used for the ocoms_rb_tree_traverse function.
  * it is passed a pointer to the value for each node and, if it returns
  * a one, the action function is called on that node. Otherwise, the node is ignored.
  */
typedef int (*ocoms_rb_tree_condition_fn_t)(void *);
/**
  * this function is uused for the user to perform any action on the passed
  * values. The first argument is the key and the second is the value.
  * note that this function SHOULD NOT modify the keys, as that would
  * mess up the tree.
  */
typedef void (*ocoms_rb_tree_action_fn_t)(void *, void *);

/*
 * Public function protoypes
 */

/**
  * the construct function. creates the free list to get the nodes from
  *
  * @param object the tree that is to be used
  *
  * @retval NONE
  */
OCOMS_DECLSPEC void ocoms_rb_tree_construct(ocoms_object_t * object);

/**
  * the destruct function. tries to free the tree and destroys the free list
  *
  * @param object the tree object
  */
OCOMS_DECLSPEC void ocoms_rb_tree_destruct(ocoms_object_t * object);

/**
  * the function creates a new tree
  *
  * @param tree a pointer to an allocated area of memory for the main
  *  tree data structure.
  * @param comp a pointer to the function to use for comaparing 2 nodes
  *
  * @retval OCOMS_SUCCESS if it is successful
  * @retval OCOMS_ERR_TEMP_OUT_OF_RESOURCE if unsuccessful
  */
OCOMS_DECLSPEC int ocoms_rb_tree_init(ocoms_rb_tree_t * tree, ocoms_rb_tree_comp_fn_t comp);


/**
  * inserts a node into the tree
  *
  * @param tree a pointer to the tree data structure
  * @param key the key for the node
  * @param value the value for the node
  *
  * @retval OCOMS_SUCCESS
  * @retval OCOMS_ERR_TEMP_OUT_OF_RESOURCE if unsuccessful
  */
OCOMS_DECLSPEC int ocoms_rb_tree_insert(ocoms_rb_tree_t *tree, void * key, void * value);

/**
  * finds a value in the tree based on the passed key using passed
  * compare function
  *
  * @param tree a pointer to the tree data structure
  * @param key a pointer to the key
  * @param compare function
  *
  * @retval pointer to the value if found
  * @retval NULL if not found
  */
OCOMS_DECLSPEC void * ocoms_rb_tree_find_with(ocoms_rb_tree_t *tree, void *key, ocoms_rb_tree_comp_fn_t compfn);

/**
  * finds a value in the tree based on the passed key
  *
  * @param tree a pointer to the tree data structure
  * @param key a pointer to the key
  *
  * @retval pointer to the value if found
  * @retval NULL if not found
  */
static inline void * ocoms_rb_tree_find(ocoms_rb_tree_t *tree, void *key)
{
        return ocoms_rb_tree_find_with(tree, key, tree->comp);
}

/**
  * deletes a node based on its key
  *
  * @param tree a pointer to the tree data structure
  * @param key a pointer to the key
  *
  * @retval OCOMS_SUCCESS if the node is found and deleted
  * @retval OCOMS_ERR_NOT_FOUND if the node is not found
  */
OCOMS_DECLSPEC int ocoms_rb_tree_delete(ocoms_rb_tree_t *tree, void *key);

/**
  * frees all the nodes on the tree
  *
  * @param tree a pointer to the tree data structure
  *
  * @retval OCOMS_SUCCESS
  */
OCOMS_DECLSPEC int ocoms_rb_tree_destroy(ocoms_rb_tree_t *tree);

/**
  * traverses the entire tree, performing the cond function on each of the
  * values and if it returns one it performs the action function on the values
  *
  * @param tree a pointer to the tree
  * @param cond a pointer to the condition function
  * @param action a pointer to the action function
  *
  * @retval OCOMS_SUCCESS
  * @retval OCOMS_ERROR if there is an error
  */
OCOMS_DECLSPEC int ocoms_rb_tree_traverse(ocoms_rb_tree_t *tree,
                          ocoms_rb_tree_condition_fn_t cond,
                          ocoms_rb_tree_action_fn_t action);

/**
  * returns the size of the tree
  *
  * @param tree a pointer to the tree data structure
  *
  * @retval int the nuber of items on the tree
  */
OCOMS_DECLSPEC int ocoms_rb_tree_size(ocoms_rb_tree_t *tree);

#define CONSTRUCT_RB_TREE(object, progress) do {\
    OBJ_CONSTRUCT(object, ocoms_rb_tree_t);\
    ((ocoms_rb_tree_t *)object)->free_list.fl_condition.ocoms_progress_fn = progress;\
    }\
    while(0);

static inline int ocoms_rb_tree_new(ocoms_rb_tree_t **rb_tree, ocoms_progress_fn_t progress)
{
    ocoms_rb_tree_t *rbt = OBJ_NEW(ocoms_rb_tree_t);
    rbt->free_list.fl_condition.ocoms_progress_fn = progress;
    *rb_tree = rbt;
    return OCOMS_SUCCESS;
}
END_C_DECLS
#endif /* OMPI_RB_TREE_H */

