"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergeFiles = exports.decodeEncodedValues = exports.encodeFormData = exports.mergeKeyValueEntries = void 0;
const mona_dish_1 = require("mona-dish");
const ExtDomQuery_1 = require("./ExtDomQuery");
const Const_1 = require("../core/Const");
/*
 * various routines for encoding and decoding url parameters
 * into configs and vice versa
 */
/**
 * merges a list of key value entries into a target config
 * @param target the target receiving the key value entries
 * @param keyValueEntries a list of key value entries divided by =
 * @param paramsMapper a key value remapper
 */
function mergeKeyValueEntries(target, keyValueEntries, paramsMapper = (key, value) => [key, value]) {
    function splitToKeyVal(line) {
        return line.split(/=(.*)/gi);
    }
    function fixKeyWithoutVal(keyVal) {
        var _a, _b;
        return keyVal.length < 3 ? [(_a = keyVal === null || keyVal === void 0 ? void 0 : keyVal[0]) !== null && _a !== void 0 ? _a : [], (_b = keyVal === null || keyVal === void 0 ? void 0 : keyVal[1]) !== null && _b !== void 0 ? _b : []] : keyVal;
    }
    let toMerge = new ExtDomQuery_1.ExtConfig({});
    mona_dish_1.Stream.of(...keyValueEntries)
        .map(line => splitToKeyVal(line))
        //special case of having keys without values
        .map(keyVal => fixKeyWithoutVal(keyVal))
        .map(keyVal => paramsMapper(keyVal[0], keyVal[1]))
        .each(keyVal => {
        var _a, _b;
        toMerge.append(keyVal[0]).value = (_b = (_a = keyVal === null || keyVal === void 0 ? void 0 : keyVal.splice(1)) === null || _a === void 0 ? void 0 : _a.join("")) !== null && _b !== void 0 ? _b : "";
    });
    target.shallowMerge(toMerge);
}
exports.mergeKeyValueEntries = mergeKeyValueEntries;
/**
 * encodes a given form data into a url encoded string
 * @param formData the form data config object
 * @param paramsMapper the params mapper
 * @param defaultStr a default string if nothing comes out of it
 */
function encodeFormData(formData, paramsMapper = (inStr, inVal) => [inStr, inVal], defaultStr = Const_1.EMPTY_STR) {
    if (formData.isAbsent()) {
        return defaultStr;
    }
    let assocValues = formData.value;
    let entries = mona_dish_1.LazyStream.of(...Object.keys(assocValues))
        .filter(key => assocValues.hasOwnProperty(key))
        .flatMap(key => mona_dish_1.Stream.of(...assocValues[key]).map(val => paramsMapper(key, val)))
        //we cannot encode file elements that is handled by multipart requests anyway
        .filter(([, value]) => !(value instanceof ExtDomQuery_1.ExtDomQuery.global().File))
        .map(keyVal => `${encodeURIComponent(keyVal[0])}=${encodeURIComponent(keyVal[1])}`)
        .collect(new mona_dish_1.ArrayCollector());
    return entries.join("&");
}
exports.encodeFormData = encodeFormData;
/**
 * splits and decodes encoded values into strings containing of key=value
 * @param encoded encoded string
 */
function decodeEncodedValues(encoded) {
    return decodeURIComponent(encoded).split(/&/gi)
        .filter(item => !!(item || '')
        .replace(/\s+/g, ''));
}
exports.decodeEncodedValues = decodeEncodedValues;
function mergeFiles(target, dataSource) {
    dataSource
        .querySelectorAllDeep("input[type='file']")
        .stream
        .map(fileInput => [fileInput.name.value || fileInput.id.value, fileInput.filesFromElem(0)])
        .each(([key, files]) => target.append(key).value = files);
}
exports.mergeFiles = mergeFiles;
//# sourceMappingURL=URLCodec.js.map