"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.constructArnInOtherRegion = exports.mrkAwareAwsKmsKeyIdCompare = exports.isMultiRegionAwsKmsIdentifier = exports.isMultiRegionAwsKmsArn = exports.validAwsKmsIdentifier = exports.parseAwsKmsResource = exports.getRegionFromIdentifier = exports.parseAwsKmsKeyArn = exports.KMS_SERVICE = void 0;
const material_management_1 = require("@aws-crypto/material-management");
/* See: https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-kms
 * regex to match: 'resourceType/resourceId' || 'resourceType'
 * This is complicated because the `split(':')`.
 * The valid resourceType resourceId delimiters are `/`, `:`.
 * This means if the delimiter is a `:` it will be split out,
 * when splitting the whole arn.
 */
exports.KMS_SERVICE = 'kms';
const ARN_PREFIX = 'arn';
const KEY_RESOURCE_TYPE = 'key';
const ALIAS_RESOURCE_TYPE = 'alias';
const MRK_RESOURCE_ID_PREFIX = 'mrk-';
const VALID_RESOURCE_TYPES = [KEY_RESOURCE_TYPE, ALIAS_RESOURCE_TYPE];
/**
 * Returns a parsed ARN if a valid AWS KMS Key ARN.
 * If the request is a valid resource the function
 * will return false.
 * However if the ARN is malformed this function throws an error,
 */
function parseAwsKmsKeyArn(kmsKeyArn) {
    /* Precondition: A KMS Key Id must be a non-null string. */
    (0, material_management_1.needs)(kmsKeyArn && typeof kmsKeyArn === 'string', 'KMS key arn must be a non-null string.');
    const parts = kmsKeyArn.split(':');
    /* Check for early return (Postcondition): A valid ARN has 6 parts. */
    if (parts.length === 1) {
        /* Exceptional Postcondition: Only a valid AWS KMS resource.
         * This may result in this function being called twice.
         * However this is the most correct behavior.
         */
        parseAwsKmsResource(kmsKeyArn);
        return false;
    }
    /* See: https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html#arn-syntax-kms
     * arn:aws:kms:us-east-1:123456789012:key/12345678-1234-1234-1234-123456789012
     * arn:aws:kms:us-east-1:123456789012:alias/example-alias
     */
    const [arnLiteral, partition, service, region = '', account = '', resource = '',] = parts;
    const [resourceType, ...resourceSection] = resource.split('/');
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
    //# The resource section MUST be non-empty and MUST be split by a
    //# single "/" any additional "/" are included in the resource id
    const resourceId = resourceSection.join('/');
    /* If this is a valid AWS KMS Key ARN, return the parsed ARN */
    (0, material_management_1.needs)(
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
    //# MUST start with string "arn"
    arnLiteral === ARN_PREFIX &&
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
        //# The partition MUST be a non-empty
        partition &&
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
        //# The service MUST be the string "kms"
        service === exports.KMS_SERVICE &&
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
        //# The region MUST be a non-empty string
        region &&
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
        //# The account MUST be a non-empty string
        account &&
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
        //# The resource type MUST be either "alias" or "key"
        VALID_RESOURCE_TYPES.includes(resourceType) &&
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.5
        //# The resource id MUST be a non-empty string
        resourceId, 'Malformed arn.');
    return {
        Partition: partition,
        Region: region,
        AccountId: account,
        ResourceType: resourceType,
        ResourceId: resourceId,
    };
}
exports.parseAwsKmsKeyArn = parseAwsKmsKeyArn;
function getRegionFromIdentifier(kmsKeyIdentifier) {
    const awsKmsKeyArn = parseAwsKmsKeyArn(kmsKeyIdentifier);
    return awsKmsKeyArn ? awsKmsKeyArn.Region : '';
}
exports.getRegionFromIdentifier = getRegionFromIdentifier;
function parseAwsKmsResource(resource) {
    /* Precondition: An AWS KMS resource can not have a `:`.
     * That would make it an ARNlike.
     */
    (0, material_management_1.needs)(resource.split(':').length === 1, 'Malformed resource.');
    /* `/` is a valid values in an AWS KMS Alias name. */
    const [head, ...tail] = resource.split('/');
    /* Precondition: A raw identifer is only an alias or a key. */
    (0, material_management_1.needs)(head === ALIAS_RESOURCE_TYPE || !tail.length, 'Malformed resource.');
    const [resourceType, resourceId] = head === ALIAS_RESOURCE_TYPE
        ? [ALIAS_RESOURCE_TYPE, tail.join('/')]
        : [KEY_RESOURCE_TYPE, head];
    return {
        ResourceType: resourceType,
        ResourceId: resourceId,
    };
}
exports.parseAwsKmsResource = parseAwsKmsResource;
function validAwsKmsIdentifier(kmsKeyIdentifier) {
    return (parseAwsKmsKeyArn(kmsKeyIdentifier) || parseAwsKmsResource(kmsKeyIdentifier));
}
exports.validAwsKmsIdentifier = validAwsKmsIdentifier;
//= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.8
//# This function MUST take a single AWS KMS ARN
function isMultiRegionAwsKmsArn(kmsIdentifier) {
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.8
    //# If the input is an invalid AWS KMS ARN this function MUST error.
    const awsKmsKeyArn = typeof kmsIdentifier === 'string'
        ? parseAwsKmsKeyArn(kmsIdentifier)
        : kmsIdentifier;
    /* Precondition: The kmsIdentifier must be an ARN. */
    (0, material_management_1.needs)(awsKmsKeyArn, 'AWS KMS identifier is not an ARN');
    const { ResourceType, ResourceId } = awsKmsKeyArn;
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.8
    //# If resource type is "alias", this is an AWS KMS alias ARN and MUST
    //# return false.
    //
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.8
    //# If resource type is "key" and resource ID starts with
    //# "mrk-", this is a AWS KMS multi-Region key ARN and MUST return true.
    //
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.8
    //# If resource type is "key" and resource ID does not start with "mrk-",
    //# this is a (single-region) AWS KMS key ARN and MUST return false.
    return (ResourceType === KEY_RESOURCE_TYPE &&
        ResourceId.startsWith(MRK_RESOURCE_ID_PREFIX));
}
exports.isMultiRegionAwsKmsArn = isMultiRegionAwsKmsArn;
//= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.9
//# This function MUST take a single AWS KMS identifier
function isMultiRegionAwsKmsIdentifier(kmsIdentifier) {
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.9
    //# If the input starts with "arn:", this MUST return the output of
    //# identifying an an AWS KMS multi-Region ARN (aws-kms-key-
    //# arn.md#identifying-an-an-aws-kms-multi-region-arn) called with this
    //# input.
    if (kmsIdentifier.startsWith('arn:')) {
        return isMultiRegionAwsKmsArn(kmsIdentifier);
    }
    else if (kmsIdentifier.startsWith('alias/')) {
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.9
        //# If the input starts with "alias/", this an AWS KMS alias and
        //# not a multi-Region key id and MUST return false.
        return false;
    }
    else if (kmsIdentifier.startsWith(MRK_RESOURCE_ID_PREFIX)) {
        //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.9
        //# If the input starts
        //# with "mrk-", this is a multi-Region key id and MUST return true.
        return true;
    }
    //= compliance/framework/aws-kms/aws-kms-key-arn.txt#2.9
    //# If
    //# the input does not start with any of the above, this is not a multi-
    //# Region key id and MUST return false.
    return false;
}
exports.isMultiRegionAwsKmsIdentifier = isMultiRegionAwsKmsIdentifier;
/* Returns a boolean representing whether two AWS KMS Key IDs should be considered equal.
 * For everything except MRK-indicating ARNs, this is a direct comparison.
 * For MRK-indicating ARNs, this is a comparison of every ARN component except region.
 * Throws an error if the IDs are not explicitly equal and at least one of the IDs
 * is not a valid AWS KMS Key ARN or alias name.
 */
//= compliance/framework/aws-kms/aws-kms-mrk-match-for-decrypt.txt#2.5
//# The caller MUST provide:
function mrkAwareAwsKmsKeyIdCompare(keyId1, keyId2) {
    //= compliance/framework/aws-kms/aws-kms-mrk-match-for-decrypt.txt#2.5
    //# If both identifiers are identical, this function MUST return "true".
    if (keyId1 === keyId2)
        return true;
    //= compliance/framework/aws-kms/aws-kms-mrk-match-for-decrypt.txt#2.5
    //# Otherwise if either input is not identified as a multi-Region key
    //# (aws-kms-key-arn.md#identifying-an-aws-kms-multi-region-key), then
    //# this function MUST return "false".
    const arn1 = parseAwsKmsKeyArn(keyId1);
    const arn2 = parseAwsKmsKeyArn(keyId2);
    if (!arn1 || !arn2)
        return false;
    if (!isMultiRegionAwsKmsArn(arn1) || !isMultiRegionAwsKmsArn(arn2))
        return false;
    //= compliance/framework/aws-kms/aws-kms-mrk-match-for-decrypt.txt#2.5
    //# Otherwise if both inputs are
    //# identified as a multi-Region keys (aws-kms-key-arn.md#identifying-an-
    //# aws-kms-multi-region-key), this function MUST return the result of
    //# comparing the "partition", "service", "accountId", "resourceType",
    //# and "resource" parts of both ARN inputs.
    return (arn1.Partition === arn2.Partition &&
        arn1.AccountId === arn2.AccountId &&
        arn1.ResourceType === arn2.ResourceType &&
        arn1.ResourceId === arn2.ResourceId);
}
exports.mrkAwareAwsKmsKeyIdCompare = mrkAwareAwsKmsKeyIdCompare;
/* Manually construct a new MRK ARN that looks like the old ARN except the region is replaced by a new region.
 * Throws an error if the input parsed ARN is not an MRK
 */
function constructArnInOtherRegion(parsedArn, region) {
    /* Precondition: Only reconstruct a multi region ARN. */
    (0, material_management_1.needs)(isMultiRegionAwsKmsArn(parsedArn), 'Cannot attempt to construct an ARN in a new region from an non-MRK ARN.');
    const { Partition, AccountId, ResourceType, ResourceId } = parsedArn;
    return [
        ARN_PREFIX,
        Partition,
        exports.KMS_SERVICE,
        region,
        AccountId,
        ResourceType + '/' + ResourceId,
    ].join(':');
}
exports.constructArnInOtherRegion = constructArnInOtherRegion;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXJuX3BhcnNpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYXJuX3BhcnNpbmcudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7OztBQUV0Qyx5RUFBdUQ7QUFFdkQ7Ozs7OztHQU1HO0FBQ1UsUUFBQSxXQUFXLEdBQUcsS0FBSyxDQUFBO0FBVWhDLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQTtBQUN4QixNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQTtBQUMvQixNQUFNLG1CQUFtQixHQUFHLE9BQU8sQ0FBQTtBQUNuQyxNQUFNLHNCQUFzQixHQUFHLE1BQU0sQ0FBQTtBQUVyQyxNQUFNLG9CQUFvQixHQUFHLENBQUMsaUJBQWlCLEVBQUUsbUJBQW1CLENBQUMsQ0FBQTtBQUVyRTs7Ozs7R0FLRztBQUNILFNBQWdCLGlCQUFpQixDQUMvQixTQUFpQjtJQUVqQiwyREFBMkQ7SUFDM0QsSUFBQSwyQkFBSyxFQUNILFNBQVMsSUFBSSxPQUFPLFNBQVMsS0FBSyxRQUFRLEVBQzFDLHdDQUF3QyxDQUN6QyxDQUFBO0lBRUQsTUFBTSxLQUFLLEdBQUcsU0FBUyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQTtJQUVsQyxzRUFBc0U7SUFDdEUsSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtRQUN0Qjs7O1dBR0c7UUFDSCxtQkFBbUIsQ0FBQyxTQUFTLENBQUMsQ0FBQTtRQUM5QixPQUFPLEtBQUssQ0FBQTtLQUNiO0lBRUQ7OztPQUdHO0lBQ0gsTUFBTSxDQUNKLFVBQVUsRUFDVixTQUFTLEVBQ1QsT0FBTyxFQUNQLE1BQU0sR0FBRyxFQUFFLEVBQ1gsT0FBTyxHQUFHLEVBQUUsRUFDWixRQUFRLEdBQUcsRUFBRSxFQUNkLEdBQUcsS0FBSyxDQUFBO0lBRVQsTUFBTSxDQUFDLFlBQVksRUFBRSxHQUFHLGVBQWUsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUE7SUFFOUQsd0RBQXdEO0lBQ3hELGlFQUFpRTtJQUNqRSxpRUFBaUU7SUFDakUsTUFBTSxVQUFVLEdBQUcsZUFBZSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQTtJQUU1QywrREFBK0Q7SUFDL0QsSUFBQSwyQkFBSztJQUNILHdEQUF3RDtJQUN4RCxnQ0FBZ0M7SUFDaEMsVUFBVSxLQUFLLFVBQVU7UUFDdkIsd0RBQXdEO1FBQ3hELHFDQUFxQztRQUNyQyxTQUFTO1FBQ1Qsd0RBQXdEO1FBQ3hELHdDQUF3QztRQUN4QyxPQUFPLEtBQUssbUJBQVc7UUFDdkIsd0RBQXdEO1FBQ3hELHlDQUF5QztRQUN6QyxNQUFNO1FBQ04sd0RBQXdEO1FBQ3hELDBDQUEwQztRQUMxQyxPQUFPO1FBQ1Asd0RBQXdEO1FBQ3hELHFEQUFxRDtRQUNyRCxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO1FBQzNDLHdEQUF3RDtRQUN4RCw4Q0FBOEM7UUFDOUMsVUFBVSxFQUNaLGdCQUFnQixDQUNqQixDQUFBO0lBQ0QsT0FBTztRQUNMLFNBQVMsRUFBRSxTQUFTO1FBQ3BCLE1BQU0sRUFBRSxNQUFNO1FBQ2QsU0FBUyxFQUFFLE9BQU87UUFDbEIsWUFBWSxFQUFFLFlBQVk7UUFDMUIsVUFBVSxFQUFFLFVBQVU7S0FDdkIsQ0FBQTtBQUNILENBQUM7QUF6RUQsOENBeUVDO0FBRUQsU0FBZ0IsdUJBQXVCLENBQUMsZ0JBQXdCO0lBQzlELE1BQU0sWUFBWSxHQUFHLGlCQUFpQixDQUFDLGdCQUFnQixDQUFDLENBQUE7SUFDeEQsT0FBTyxZQUFZLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQTtBQUNoRCxDQUFDO0FBSEQsMERBR0M7QUFFRCxTQUFnQixtQkFBbUIsQ0FDakMsUUFBZ0I7SUFFaEI7O09BRUc7SUFDSCxJQUFBLDJCQUFLLEVBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLHFCQUFxQixDQUFDLENBQUE7SUFFOUQscURBQXFEO0lBQ3JELE1BQU0sQ0FBQyxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsR0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFBO0lBRTNDLDhEQUE4RDtJQUM5RCxJQUFBLDJCQUFLLEVBQUMsSUFBSSxLQUFLLG1CQUFtQixJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxxQkFBcUIsQ0FBQyxDQUFBO0lBRTFFLE1BQU0sQ0FBQyxZQUFZLEVBQUUsVUFBVSxDQUFDLEdBQzlCLElBQUksS0FBSyxtQkFBbUI7UUFDMUIsQ0FBQyxDQUFDLENBQUMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUN2QyxDQUFDLENBQUMsQ0FBQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsQ0FBQTtJQUUvQixPQUFPO1FBQ0wsWUFBWSxFQUFFLFlBQVk7UUFDMUIsVUFBVSxFQUFFLFVBQVU7S0FDdkIsQ0FBQTtBQUNILENBQUM7QUF2QkQsa0RBdUJDO0FBRUQsU0FBZ0IscUJBQXFCLENBQ25DLGdCQUF3QjtJQUl4QixPQUFPLENBQ0wsaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxtQkFBbUIsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUM3RSxDQUFBO0FBQ0gsQ0FBQztBQVJELHNEQVFDO0FBRUQsd0RBQXdEO0FBQ3hELGdEQUFnRDtBQUNoRCxTQUFnQixzQkFBc0IsQ0FDcEMsYUFBMEM7SUFFMUMsd0RBQXdEO0lBQ3hELG9FQUFvRTtJQUNwRSxNQUFNLFlBQVksR0FDaEIsT0FBTyxhQUFhLEtBQUssUUFBUTtRQUMvQixDQUFDLENBQUMsaUJBQWlCLENBQUMsYUFBYSxDQUFDO1FBQ2xDLENBQUMsQ0FBQyxhQUFhLENBQUE7SUFFbkIscURBQXFEO0lBQ3JELElBQUEsMkJBQUssRUFBQyxZQUFZLEVBQUUsa0NBQWtDLENBQUMsQ0FBQTtJQUV2RCxNQUFNLEVBQUUsWUFBWSxFQUFFLFVBQVUsRUFBRSxHQUFHLFlBQVksQ0FBQTtJQUVqRCx3REFBd0Q7SUFDeEQsc0VBQXNFO0lBQ3RFLGlCQUFpQjtJQUNqQixFQUFFO0lBQ0Ysd0RBQXdEO0lBQ3hELHlEQUF5RDtJQUN6RCx3RUFBd0U7SUFDeEUsRUFBRTtJQUNGLHdEQUF3RDtJQUN4RCx5RUFBeUU7SUFDekUsb0VBQW9FO0lBQ3BFLE9BQU8sQ0FDTCxZQUFZLEtBQUssaUJBQWlCO1FBQ2xDLFVBQVUsQ0FBQyxVQUFVLENBQUMsc0JBQXNCLENBQUMsQ0FDOUMsQ0FBQTtBQUNILENBQUM7QUE5QkQsd0RBOEJDO0FBRUQsd0RBQXdEO0FBQ3hELHVEQUF1RDtBQUN2RCxTQUFnQiw2QkFBNkIsQ0FBQyxhQUFxQjtJQUNqRSx3REFBd0Q7SUFDeEQsbUVBQW1FO0lBQ25FLDREQUE0RDtJQUM1RCx1RUFBdUU7SUFDdkUsVUFBVTtJQUNWLElBQUksYUFBYSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUNwQyxPQUFPLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxDQUFBO0tBQzdDO1NBQU0sSUFBSSxhQUFhLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQzdDLHdEQUF3RDtRQUN4RCxnRUFBZ0U7UUFDaEUsb0RBQW9EO1FBQ3BELE9BQU8sS0FBSyxDQUFBO0tBQ2I7U0FBTSxJQUFJLGFBQWEsQ0FBQyxVQUFVLENBQUMsc0JBQXNCLENBQUMsRUFBRTtRQUMzRCx3REFBd0Q7UUFDeEQsdUJBQXVCO1FBQ3ZCLG9FQUFvRTtRQUNwRSxPQUFPLElBQUksQ0FBQTtLQUNaO0lBQ0Qsd0RBQXdEO0lBQ3hELE1BQU07SUFDTix3RUFBd0U7SUFDeEUsd0NBQXdDO0lBQ3hDLE9BQU8sS0FBSyxDQUFBO0FBQ2QsQ0FBQztBQXhCRCxzRUF3QkM7QUFFRDs7Ozs7R0FLRztBQUNILHNFQUFzRTtBQUN0RSw0QkFBNEI7QUFDNUIsU0FBZ0IsMEJBQTBCLENBQ3hDLE1BQWMsRUFDZCxNQUFjO0lBRWQsc0VBQXNFO0lBQ3RFLHdFQUF3RTtJQUN4RSxJQUFJLE1BQU0sS0FBSyxNQUFNO1FBQUUsT0FBTyxJQUFJLENBQUE7SUFFbEMsc0VBQXNFO0lBQ3RFLHFFQUFxRTtJQUNyRSxzRUFBc0U7SUFDdEUsc0NBQXNDO0lBQ3RDLE1BQU0sSUFBSSxHQUFHLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFBO0lBQ3RDLE1BQU0sSUFBSSxHQUFHLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFBO0lBQ3RDLElBQUksQ0FBQyxJQUFJLElBQUksQ0FBQyxJQUFJO1FBQUUsT0FBTyxLQUFLLENBQUE7SUFDaEMsSUFBSSxDQUFDLHNCQUFzQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsSUFBSSxDQUFDO1FBQ2hFLE9BQU8sS0FBSyxDQUFBO0lBRWQsc0VBQXNFO0lBQ3RFLGdDQUFnQztJQUNoQyx5RUFBeUU7SUFDekUsc0VBQXNFO0lBQ3RFLHNFQUFzRTtJQUN0RSw0Q0FBNEM7SUFDNUMsT0FBTyxDQUNMLElBQUksQ0FBQyxTQUFTLEtBQUssSUFBSSxDQUFDLFNBQVM7UUFDakMsSUFBSSxDQUFDLFNBQVMsS0FBSyxJQUFJLENBQUMsU0FBUztRQUNqQyxJQUFJLENBQUMsWUFBWSxLQUFLLElBQUksQ0FBQyxZQUFZO1FBQ3ZDLElBQUksQ0FBQyxVQUFVLEtBQUssSUFBSSxDQUFDLFVBQVUsQ0FDcEMsQ0FBQTtBQUNILENBQUM7QUE5QkQsZ0VBOEJDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQix5QkFBeUIsQ0FDdkMsU0FBNkIsRUFDN0IsTUFBYztJQUVkLHdEQUF3RDtJQUN4RCxJQUFBLDJCQUFLLEVBQ0gsc0JBQXNCLENBQUMsU0FBUyxDQUFDLEVBQ2pDLHlFQUF5RSxDQUMxRSxDQUFBO0lBQ0QsTUFBTSxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsWUFBWSxFQUFFLFVBQVUsRUFBRSxHQUFHLFNBQVMsQ0FBQTtJQUNwRSxPQUFPO1FBQ0wsVUFBVTtRQUNWLFNBQVM7UUFDVCxtQkFBVztRQUNYLE1BQU07UUFDTixTQUFTO1FBQ1QsWUFBWSxHQUFHLEdBQUcsR0FBRyxVQUFVO0tBQ2hDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFBO0FBQ2IsQ0FBQztBQWxCRCw4REFrQkMifQ==