import hashlib
import re
import argparse
import sys
from urllib.parse import urlsplit, parse_qs, unquote, quote
from typing import Dict, List, Tuple

_SIGNATURE_REGEX = re.compile(r'[^A-Za-z0-9/?_.=&{}\[\]":, -]')

def compute_signature(method: str, path: str, body: str = '', key: str = '') -> str:
    if not method or not path:
        raise ValueError("Method and path must be provided.")

    url_parts = urlsplit(path)
    base_path = url_parts.path
    
    if not base_path.startswith('/'):
        base_path = '/' + base_path
    
    raw_query_params: Dict[str, List[str]] = parse_qs(
        url_parts.query, keep_blank_values=True, strict_parsing=False
    )
    
    canonical_query: List[Tuple[str, str]] = []
    for k, v_list in raw_query_params.items():
        if k == '_signature':
            continue
            
        value = unquote(v_list[0]) if v_list else ''
        canonical_query.append((k, value))
    
    canonical_query.sort(key=lambda item: item[0])
    
    query_string = '&'.join(f"{k}={quote(v)}" for k, v in canonical_query)

    if query_string:
        canonical_path = f"{base_path}?{query_string}"
    else:
        canonical_path = base_path
        
    canonical_path = re.sub(_SIGNATURE_REGEX, '-', canonical_path)

    body_for_signing = re.sub(_SIGNATURE_REGEX, '-', body)
    
    if not key:
        password_hash = "da39a3ee5e6b4b0d3255bfef95601890afd80709"
    else:
        password_hash = hashlib.sha1(key.encode('utf-8')).hexdigest().lower()

    data = f"{method.upper()}:{canonical_path}:{body_for_signing}:{password_hash}"
    
    return hashlib.sha1(data.encode('utf-8')).hexdigest().lower()

def main():
    parser = argparse.ArgumentParser(description="Computes a SHA1 signature for an HTTP request.")

    parser.add_argument('--method', type=str, required=True,
                        choices=['GET', 'POST', 'PUT', 'DELETE'],
                        help="The HTTP method (e.g., GET).")
    parser.add_argument('--path', type=str, required=True,
                        help="The canonical path (e.g., /api/resource?param=value).")
    parser.add_argument('--key', type=str, default='',
                        help="The secret key. Defaults to an empty string.")
    parser.add_argument('--body', type=str, default='',
                        help="The request body as a string. Defaults to an empty string.")

    try:
        args = parser.parse_args()
        
        signature = compute_signature(
            method=args.method,
            path=args.path,
            body=args.body,
            key=args.key
        )

        print(f"Computed Signature: {signature}")

    except ValueError as e:
        sys.stderr.write(f"Error: {e}\n")
        sys.exit(1)
    except Exception as e:
        sys.stderr.write(f"An unexpected error occurred: {e}\n")
        sys.exit(1)


if __name__ == '__main__':
    main()
