#include <fstream>
#include <string>

#include <apertium/mtx_reader.h>
#include <apertium/perceptron_tagger.h>
#include <apertium/shell_utils.h>
#include <apertium/stream_tagger.h>
#include <lttoolbox/lt_locale.h>

namespace Apertium {

using namespace Apertium::ShellUtils;

int perceptron_trace(int argc, char* argv[])
{
  std::locale::global(std::locale(""));
  if (argc == 3 && std::string(argv[1]) == "model")
  {
    // Dumps a tagger model
    TaggerFlags flags;
    PerceptronTagger pt(flags);
    std::ifstream tagger_model;
    try_open_fstream("MODEL", argv[2], tagger_model);
    pt.deserialise(tagger_model);
    std::cout << pt;
  }
  else if (argc == 3 && std::string(argv[1]) == "mtx")
  {
    PerceptronSpec spec;
    MTXReader mtx_reader(spec);
    mtx_reader.read(argv[2]);
    std::cout << "== Macro definitions ==\n";
    mtx_reader.printTmplDefns();
    std::cout << "== Spec ==\n";
    std::cout << spec;
  }
  else if (argc == 5 && std::string(argv[1]) == "path")
  {
    // Dumps features generated by every wordoid in every path using a correctly tagged path
    // Doesn't use a model
    TaggerFlags flags;
    PerceptronTagger pt(flags);
    pt.read_spec(argv[2]);

    Stream untagged(flags, argv[3]);
    Stream tagged(flags, argv[4]);

    TrainingCorpus tc(tagged, untagged, false, false);

    size_t sent_idx, token_idx, analy_idx, wrd_idx;
    for (sent_idx=0; sent_idx<tc.sentences.size(); sent_idx++)
    {
      TaggedSentence &tagged_sent = tc.sentences[sent_idx].first;
      Sentence &untagged_sent = tc.sentences[sent_idx].second;
      for (token_idx=0; token_idx<untagged_sent.size(); token_idx++)
      {
        if (!untagged_sent[token_idx].TheLexicalUnit)
        {
          continue;
        }
        LexicalUnit &lu = *untagged_sent[token_idx].TheLexicalUnit;
        for (analy_idx=0; analy_idx<lu.TheAnalyses.size(); analy_idx++)
        {
          Optional<Analysis> saved_token = tagged_sent[token_idx];
          tagged_sent[token_idx] = lu.TheAnalyses[analy_idx];
          std::cout << "LU:" << tagged_sent[token_idx] << std::endl ;
          std::vector<Morpheme> &wordoids = lu.TheAnalyses[analy_idx].TheMorphemes;
          for (wrd_idx=0; wrd_idx<wordoids.size(); wrd_idx++)
          {
            UnaryFeatureVec feat_vec;
            pt.spec.get_features(
              tagged_sent, untagged_sent,
              token_idx, wrd_idx,
              feat_vec);
            std::cout << "Sentence " << sent_idx << " of " << tc.sentences.size() << "\t\t"
                       << "Token " << token_idx << " of " << untagged_sent.size() << "\t\t"
                       << "Analysis " << analy_idx << " of " << lu.TheAnalyses.size() << "\t\t"
                       << "Wordoid " << wrd_idx << " of " << wordoids.size() << "\n";
            std::cout << "" << wordoids[wrd_idx] << "\n";
            FeatureVec fv(feat_vec);
            std::cout << fv;
            std::cout << "\n\n";
          }
          tagged_sent[token_idx] = saved_token;
        }
      }
    }
  }
  else
  {
    std::cout << "Run with one of:\n";
    std::cout << argv[0] << " model <binary model file>\n";
    std::cout << "Output features and weights from a model file.\n";
    std::cout << argv[0] << " mtx <mtx file>\n";
    std::cout << "Output macros and features from an mtx file.\n";
    std::cout << argv[0] << " path <mtx file> <untagged> <tagged>\n";
    std::cout << "Trace a particular path through giving which features fire "
               << "and the resulting score. Useful for interactively "
               << "designing feature sets.\n";
  }
  return 0;
}

}


int main(int argc, char* argv[]) {
  LtLocale::tryToSetLocale();
  return Apertium::perceptron_trace(argc, argv);
}
