import { IrcClientConfig } from "../models/IrcClientConfig";
export declare type MembershipSyncKind = "incremental" | "initial";
export interface IrcServerConfig {
    name?: string;
    port?: number;
    icon?: string;
    ca?: string;
    networkId?: string;
    ssl?: boolean;
    sslselfsign?: boolean;
    sasl?: boolean;
    tlsOptions?: Record<string, unknown>;
    password?: string;
    allowExpiredCerts?: boolean;
    additionalAddresses?: string[];
    onlyAdditionalAddresses: boolean;
    dynamicChannels: {
        enabled: boolean;
        published: boolean;
        useHomeserverDirectory: boolean;
        createAlias: boolean;
        joinRule: "public" | "invite";
        federate: boolean;
        aliasTemplate: string;
        whitelist: string[];
        exclude: string[];
        roomVersion?: string;
        groupId?: string;
    };
    quitDebounce: {
        enabled: boolean;
        quitsPerSecond: number;
        delayMinMs: number;
        delayMaxMs: number;
    };
    mappings: {
        [channel: string]: {
            roomIds: string[];
            key?: string;
        };
    };
    modePowerMap?: {
        [mode: string]: number;
    };
    sendConnectionMessages: boolean;
    botConfig: {
        nick: string;
        joinChannelsIfNoUsers: boolean;
        enabled: boolean;
        password?: string;
        username: string;
    };
    privateMessages: {
        enabled: boolean;
        exclude: string[];
        federate: boolean;
    };
    matrixClients: {
        userTemplate: string;
        displayName: string;
        joinAttempts: number;
    };
    ircClients: {
        nickTemplate: string;
        maxClients: number;
        idleTimeout: number;
        reconnectIntervalMs: number;
        concurrentReconnectLimit: number;
        allowNickChanges: boolean;
        ipv6: {
            only: boolean;
            prefix?: string;
            blocks?: {
                homeserver: string;
                startFrom: string;
            }[];
        };
        lineLimit: number;
        userModes?: string;
        realnameFormat?: "mxid" | "reverse-mxid";
        pingTimeoutMs: number;
        pingRateMs: number;
        kickOn: {
            channelJoinFailure: boolean;
            ircConnectionFailure: boolean;
            userQuit: boolean;
        };
    };
    excludedUsers: Array<{
        regex: string;
        kickReason?: string;
    }>;
    membershipLists: {
        enabled: boolean;
        floodDelayMs: number;
        ignoreIdleOnStartup?: {
            enabled: true;
            idleForHours: number;
            exclude: string;
        };
        global: {
            ircToMatrix: {
                initial: boolean;
                incremental: boolean;
                requireMatrixJoined: boolean;
            };
            matrixToIrc: {
                initial: boolean;
                incremental: boolean;
            };
        };
        channels: {
            channel: string;
            ircToMatrix: {
                initial: boolean;
                incremental: boolean;
                requireMatrixJoined: boolean;
            };
        }[];
        rooms: {
            room: string;
            matrixToIrc: {
                initial: boolean;
                incremental: boolean;
            };
        }[];
    };
}
export declare class IrcServer {
    domain: string;
    config: IrcServerConfig;
    readonly homeserverDomain: string;
    private expiryTimeSeconds;
    private addresses;
    private groupIdValid;
    private excludedUsers;
    private idleUsersStartupExcludeRegex?;
    private enforceReconnectInterval;
    /**
     * Construct a new IRC Server.
     * @constructor
     * @param {string} domain : The IRC network address
     * @param {Object} serverConfig : The config options for this network.
     * @param {string} homeserverDomain : The domain of the homeserver
     * e.g "matrix.org"
     * @param {number} expiryTimeSeconds : How old a matrix message can be
     * before it is considered 'expired' and not sent to IRC. If 0, messages
     * will never expire.
     */
    constructor(domain: string, config: IrcServerConfig, homeserverDomain: string, expiryTimeSeconds?: number);
    /**
     * Get how old a matrix message can be (in seconds) before it is considered
     * 'expired' and not sent to IRC.
     * @return {Number} The number of seconds. If 0, they never expire.
     */
    getExpiryTimeSeconds(): number;
    /**
     * Get a string that represents the human-readable name for a server.
     * @return {string} this.config.name if truthy, otherwise it will return
     * an empty string.
     */
    getReadableName(): string;
    /**
     * Get an icon to represent the network
     * The icon URL, if configured.
     */
    getIcon(): string | undefined;
    /**
     * Return a randomised server domain from the default and additional addresses.
     * @return {string}
     */
    randomDomain(): string;
    /**
     * Returns the network ID of this server, which should be unique across all
     * IrcServers on the bridge. Defaults to the domain of this IrcServer.
     * @return {string} this.config.networkId || this.domain
     */
    getNetworkId(): string;
    /**
     * Returns whether the server is configured to wait getQuitDebounceDelayMs before
     * parting a user that has disconnected due to a net-split.
     * @return {Boolean} this.config.quitDebounce.enabled.
     */
    shouldDebounceQuits(): boolean;
    /**
     * Get a random interval to delay a quits for when debouncing. Will be between
     * `delayMinMs` and `delayMaxMs`
     */
    getQuitDebounceDelay(): number;
    /**
     * Get the rate of maximum quits received per second before a net-split is
     * detected. If the rate of quits received becomes higher that this value,
     * a net split is considered ongoing.
     * @return {number}
     */
    getDebounceQuitsPerSecond(): number;
    /**
     * Get a map that converts IRC user modes to Matrix power levels.
     * @return {Object}
     */
    getModePowerMap(): {
        [mode: string]: number;
    };
    getHardCodedRoomIds(): string[];
    getChannelKey(channel: string): string | undefined;
    shouldSendConnectionNotices(): boolean;
    isBotEnabled(): boolean;
    getUserModes(): string;
    getRealNameFormat(): "mxid" | "reverse-mxid";
    getJoinRule(): "invite" | "public";
    areGroupsEnabled(): boolean;
    getGroupId(): string | undefined;
    shouldFederatePMs(): boolean;
    getMemberListFloodDelayMs(): number;
    shouldFederate(): boolean;
    forceRoomVersion(): string | undefined;
    getPort(): number | undefined;
    isInWhitelist(userId: string): boolean;
    getSecureOptions(): Record<string, unknown>;
    useSsl(): boolean;
    useSslSelfSigned(): boolean;
    useSasl(): boolean;
    allowExpiredCerts(): boolean;
    getIdleTimeout(): number;
    toggleReconnectInterval(enable: boolean): void;
    getReconnectIntervalMs(): number;
    getConcurrentReconnectLimit(): number;
    getMaxClients(): number;
    shouldPublishRooms(): boolean;
    shouldPublishRoomsToHomeserverDirectory(): boolean;
    allowsNickChanges(): boolean;
    getBotNickname(): string;
    createBotIrcClientConfig(): IrcClientConfig;
    getIpv6Prefix(): string | undefined;
    getIpv6Only(): boolean;
    getLineLimit(): number;
    getJoinAttempts(): number;
    isExcludedChannel(channel: string): boolean;
    isExcludedUser(userId: string): {
        regex: RegExp;
        kickReason?: string | undefined;
    } | undefined;
    get ignoreIdleUsersOnStartup(): true | undefined;
    get ignoreIdleUsersOnStartupAfterMs(): number;
    get ignoreIdleUsersOnStartupExcludeRegex(): RegExp | undefined;
    get aliasTemplateHasHashPrefix(): boolean;
    /**
     * The amount of time to allow for inactivty on the connection, before considering the connection
     * dead. This usually happens if the IRCd doesn't ping us.
     */
    get pingTimeout(): number;
    /**
     * The rate at which to send pings to the IRCd if the client is being quiet for a while.
     * Whilst the IRCd *should* be sending pings to us to keep the connection alive, it appears
     * that sometimes they don't get around to it and end up ping timing us out.
    */
    get pingRateMs(): number;
    canJoinRooms(userId: string): boolean;
    createsDynamicAliases(): boolean;
    createsPublicAliases(): boolean;
    allowsPms(): boolean;
    shouldSyncMembershipToIrc(kind: MembershipSyncKind, roomId?: string): boolean;
    shouldSyncMembershipToMatrix(kind: MembershipSyncKind, channel: string): boolean;
    private shouldSyncMembership;
    /**
     * Does the server/channel require all Matrix users to be joined?
     * @param channel The IRC channel.
     * @returns True if the server requires all Matrix users to be joined.
     */
    shouldRequireMatrixUserJoined(channel: string): boolean;
    shouldJoinChannelsIfNoUsers(): boolean;
    isMembershipListsEnabled(): boolean;
    getUserLocalpart(nick: string): string;
    claimsUserId(userId: string): boolean;
    getNickFromUserId(userId: string): string | null;
    getUserIdFromNick(nick: string): string;
    getDisplayNameFromNick(nick: string): string;
    claimsAlias(alias: string): boolean;
    getChannelFromAlias(alias: string): string | null;
    getAliasFromChannel(channel: string): string;
    getNick(userId: string, displayName?: string): string;
    getAliasRegex(): string;
    getUserRegex(): string;
    getIpv6BlockForHomeserver(homeserver: string): string | null;
    static get DEFAULT_CONFIG(): IrcServerConfig;
    reconfigure(config: IrcServerConfig, expiryTimeSeconds?: number): void;
    private static templateToRegex;
    private static escapeRegExp;
}
