/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import javax.xml.transform.TransformerException;

import junit.framework.TestCase;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.bool.BoolUtils;
import woolpack.fn.FnUtils;

public class NodeFindUtilsTest extends TestCase {

	private Node generateSampleNode() throws TransformerException {
		return XmlTransformerUtils.convert("<HTML><BODY>" +
				"<SPAN id=\"id0\"/>" +
				"<SPAN id=\"id1\"><SPAN id=\"id11\"/></SPAN>" +
				"</BODY></HTML>");
	}
	
	public void testEvaluateOneException() throws TransformerException {
		final Node node = generateSampleNode();
		try {
			NodeFindUtils.one(
					FnUtils.<Node, Boolean, IllegalStateException>throwing(
							new IllegalStateException()))
							.exec(node);
			fail();
		} catch (final IllegalStateException e) {
		}
	}
	
	public void testEvaluateOneNotFound() throws TransformerException {
		final Node node = generateSampleNode();
		assertNull(NodeFindUtils.one(FnUtils.join(
				XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("DIV")))
				.exec(node));
	}
	
	public void testEvaluateOneFound() throws TransformerException {
		final Node node = generateSampleNode();
		final Node result = NodeFindUtils.one(FnUtils.join(
				XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("SPAN")))
				.exec(node);
		assertEquals("id0", ((Element) result).getAttribute("id"));
	}
	
	public void testEvaluateListNotFound() throws TransformerException {
		final Node node = generateSampleNode();
		final NodeList result = NodeFindUtils.list(FnUtils.join(
				XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("DIV")), true)
				.exec(node);
		assertEquals(0, result.getLength());
	}
	
	public void testEvaluateListShallow() throws TransformerException {
		final Node node = generateSampleNode();
		final NodeList result = NodeFindUtils.list(FnUtils.join(
				XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("SPAN")), false).exec(node);
		assertEquals(2, result.getLength());
		assertEquals("id0", ((Element) result.item(0)).getAttribute("id"));
		assertEquals("id1", ((Element) result.item(1)).getAttribute("id"));
	}
	
	public void testEvaluateListDeep() throws TransformerException {
		final Node node = generateSampleNode();
		final NodeList result = NodeFindUtils.list(FnUtils.join(
				XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("SPAN")), true)
				.exec(node);
		assertEquals(3, result.getLength());
		assertEquals("id0", ((Element) result.item(0)).getAttribute("id"));
		assertEquals("id1", ((Element) result.item(1)).getAttribute("id"));
		assertEquals("id11", ((Element) result.item(2)).getAttribute("id"));
	}
}
