/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.action;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import woolpack.adapter.OGE;
import woolpack.bool.BoolUtils;
import woolpack.el.ELUtils;
import woolpack.el.PathEL;
import woolpack.fn.FnUtils;
import woolpack.test.TestBean;
import woolpack.utils.Utils;

public class ActionInvokerTest extends TestCase {

	public void testFindId() {
		final ForwardDef[] forwardDefArray = new ForwardDef[] {
				new ForwardDef("class", ELUtils.NULL, BoolUtils.matchObject(Integer.class)),
				new ForwardDef("null", ELUtils.NULL, BoolUtils.matchObject(null)),
				new ForwardDef("value", ELUtils.NULL, BoolUtils.matchObject("myValue")), };
		{
			final ActionResult forwardResult = ActionInvoker.findId(Arrays
					.asList(forwardDefArray), Integer.valueOf(3));
			assertEquals(forwardDefArray[0], forwardResult.getForwardDef());
			assertEquals(Integer.valueOf(3), forwardResult.getReturnedObject());
		}
		{
			final ActionResult info = ActionInvoker.findId(Arrays
					.asList(forwardDefArray), null);
			assertEquals(forwardDefArray[1], info.getForwardDef());
			assertNull(info.getReturnedObject());
		}
		{
			final ActionResult info = ActionInvoker.findId(Arrays
					.asList(forwardDefArray), "myValue");
			assertEquals(forwardDefArray[2], info.getForwardDef());
			assertEquals("myValue", info.getReturnedObject());
		}
		try {
			ActionInvoker.findId(Arrays.asList(forwardDefArray), "yourValue");
			fail();
		} catch (final ForwardDefNotFoundRuntimeException expected) {
			assertEquals("yourValue", expected.getReturnedObject());
		}
	}

	public void testNormal() {
		final ForwardDef[] forwardDefs = {
				new ForwardDef("forwardId0", ELUtils.NULL, BoolUtils.matchObject(Integer.class)),
				new ForwardDef("forwardId1", ELUtils.NULL, BoolUtils.matchObject(Integer.class)),
				new ForwardDef("forwardId2", ELUtils.NULL, FnUtils.<Object, Boolean>fix(true)),
				new ForwardDef("forwardId0", ELUtils.NULL, BoolUtils.matchObject(IndexOutOfBoundsException.class)), };
		final ActionInvoker defs = new ActionInvoker(
				FnUtils.switching(Utils
						.map("id0",
								new ActionDef(
										new PathEL("component0"),
										new OGE("component0.executeInt()"),
										forwardDefs[0]))
						.map("id1",
								new ActionDef(
										new PathEL("component0"),
										new OGE("component0.executeVoid()"),
										forwardDefs[1]))
						.map("id2",
								new ActionDef(
										new PathEL("component0"),
										new OGE("component0.executeException()"),
										forwardDefs[3]))
				), forwardDefs[2]);
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections
					.singletonMap("component0", bean), "id0", inputMap);
			assertEquals(forwardDefs[0], forwardResult.getForwardDef());
			assertEquals(Integer.valueOf(1), forwardResult.getReturnedObject());
			assertEquals(7, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("yourInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections
					.singletonMap("component0", bean), "id0", inputMap);
			assertEquals(forwardDefs[0], forwardResult.getForwardDef());
			assertEquals(Integer.valueOf(1), forwardResult.getReturnedObject());
			assertEquals(0, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections
					.singletonMap("component0", bean), "id2", inputMap);
			assertEquals(forwardDefs[3], forwardResult.getForwardDef());
			assertTrue(forwardResult.getReturnedObject() instanceof IndexOutOfBoundsException);
			assertEquals(7, bean.getMyInt());
		}
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			final ActionResult forwardResult = defs.invoke(Collections
					.singletonMap("component0", bean), "id1", inputMap);
			assertEquals(forwardDefs[2], forwardResult.getForwardDef());
			assertNull(forwardResult.getReturnedObject());
			assertEquals(7, bean.getMyInt());
		}
	}

	public void testForwardDefNotFoundException() {
		final ForwardDef forwardDefs = new ForwardDef("forwardId0",
				ELUtils.NULL, BoolUtils.matchObject(Integer.class));
		final ActionInvoker defs = new ActionInvoker(
				FnUtils.switching(Utils
						.map("id0",
								new ActionDef(
										new PathEL("component0"),
										new OGE("component0.executeException()"),
										forwardDefs))
						), forwardDefs);
		{
			final Map<String, String> inputMap = new HashMap<String, String>();
			inputMap.put("myInt", "7");
			final TestBean bean = new TestBean();
			try {
				defs.invoke(Collections.singletonMap("component0", bean),
						"id0", inputMap);
				fail();
			} catch (final ForwardDefNotFoundRuntimeException expected) {
				assertTrue(expected.getReturnedObject() instanceof IndexOutOfBoundsException);
			}
		}
	}

	public void testSetValuesToSingle() {
		final Object o = new TestBean();
		{
			final Map<String, Object> paramMap = new HashMap<String, Object>();
			paramMap.put("myInt", Integer.valueOf(3));
			ActionInvoker.setValuesTo(o, paramMap, null);
			assertEquals(Integer.valueOf(3), new PathEL("myInt").getValue(o));
		}
		{
			final Map<String, Object> paramMap = new HashMap<String, Object>();
			paramMap.put("myInt", "5");
			ActionInvoker.setValuesTo(o, paramMap, null);
			assertEquals(Integer.valueOf(5), new PathEL("myInt").getValue(o));
		}
	}

	public void testSetValuesToInteger() {
		final TestBean o = new TestBean();

		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new Integer[] { Integer.valueOf(3) });
		paramMap.put("myCollection", new Integer[] { Integer.valueOf(11),
				Integer.valueOf(13), });
		paramMap.put("myIntArray", new Integer[] { Integer.valueOf(5),
				Integer.valueOf(7), });

		ActionInvoker.setValuesTo(o, paramMap, null);

		assertEquals(3, o.getMyInt());
		assertEquals(2, o.getMyIntArray().length);
		assertEquals(5, o.getMyIntArray()[0]);
		assertEquals(7, o.getMyIntArray()[1]);
		assertEquals(
				Arrays.asList(11, 13), o.getMyCollection());
	}

	public void testSetValuesToInt() {
		final TestBean o = new TestBean();

		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new int[] { 3 });
		paramMap.put("myCollection", new int[] { 11, 13 });
		paramMap.put("myIntArray", new int[] { 5, 7 });

		ActionInvoker.setValuesTo(o, paramMap, null);

		assertEquals(3, o.getMyInt());
		assertEquals(2, o.getMyIntArray().length);
		assertEquals(5, o.getMyIntArray()[0]);
		assertEquals(7, o.getMyIntArray()[1]);
		assertEquals(Arrays.asList(11, 13), o.getMyCollection());
	}

	public void testSetValuesToString() {
		final TestBean o = new TestBean();

		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", new String[] { "3" });
		paramMap.put("myIntArray", new String[] { "5", "7" });
		paramMap.put("myCollection", new String[] { "11", "13" });

		ActionInvoker.setValuesTo(o, paramMap, null);

		assertEquals(3, o.getMyInt());
		assertEquals(2, o.getMyIntArray().length);
		assertEquals(5, o.getMyIntArray()[0]);
		assertEquals(7, o.getMyIntArray()[1]);
		assertEquals(Arrays.asList("11", "13"), o.getMyCollection());
	}

	public void testSetValuesToException1() {
		final Object o = new Object();

		final Map<String, Object> paramMap = new HashMap<String, Object>();
		paramMap.put("myInt", Integer.valueOf(3));
		ActionInvoker.setValuesTo(o, paramMap, null);
	}
}
