/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.xml;

import java.io.Reader;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.el.AbstractEL;
import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * @author nakamura
 *
 */
public final class XmlUtils {
	
	public static final NodeGetter<RuntimeException> GET_NODE = new NodeGetter<RuntimeException>();
	public static final NodeNameGetter<RuntimeException> GET_NODE_NAME = new NodeNameGetter<RuntimeException>();
	public static final ThisRemover<RuntimeException> REMOVE_THIS = new ThisRemover<RuntimeException>();
	public static final ChildrenRemover<RuntimeException> REMOVE_CHILDREN = new ChildrenRemover<RuntimeException>();
	public static final ChildrenRetain<RuntimeException> RETAIN_CHILDREN = new ChildrenRetain<RuntimeException>();
	public static final NodeClone<RuntimeException> CLONE_NODE = new NodeClone<RuntimeException>();
	public static final NodeNormalizer<RuntimeException> NORMALIZE_NODE = new NodeNormalizer<RuntimeException>();

	/**
	 * {@link NodeContext}のアクセサにアクセスする{@link EL}です。
	 */
	public static final EL NODE_EL = new AbstractEL() {
		@Override
		public Object getValue(final Object root, final Class clazz) {
			return ((NodeContext) root).getNode();
		}
		@Override
		public void setValue(final Object root, final Object value) {
			((NodeContext) root).setNode((Node) value);
		}
	};
	
	private XmlUtils() {
	}
	
	/**
	 * DOM ノードを比較します。
	 * @param node0
	 * @param node1
	 * @return 同一内容を表すなら true。それ以外は false。
	 */
	public static boolean equalsNode(final Node node0, final Node node1) {
		if (node0 == null) {
			return node1 == null;
		}
		if (node1 == null) {
			return false;
		}
		if (node0.getNodeType() != node1.getNodeType()) {
			return false;
		}
		if (node0.getNodeType() == Node.TEXT_NODE
				|| node0.getNodeType() == Node.COMMENT_NODE) {
			return node0.getNodeValue().equals(node1.getNodeValue());
		}
		if (node0.getNodeType() == Node.ATTRIBUTE_NODE) {
			return node0.getNodeName().equals(node1.getNodeName())
					&& node0.getNodeValue().equals(node1.getNodeValue());
		}
		if (node0.getNodeType() == Node.DOCUMENT_NODE) {
			return equalsNode(((Document) node0).getDocumentElement(),
					((Document) node1).getDocumentElement());
		}
		if (!node0.getNodeName().equals(node1.getNodeName())) {
			return false;
		}

		final Element e0 = (Element) node0;
		final Element e1 = (Element) node1;
		final NamedNodeMap map0 = e0.getAttributes();
		final NamedNodeMap map1 = e1.getAttributes();
		if (map0.getLength() != map1.getLength()) {
			return false;
		}
		for (int i = 0; i < map0.getLength(); i++) {
			if (!equalsNode(map0.item(i), map1.item(i))) {
				return false;
			}
		}

		Node child0 = node0.getFirstChild();
		Node child1 = node1.getFirstChild();
		while (child0 != null || child1 != null) {
			if (!equalsNode(child0, child1)) {
				return false;
			}
			child0 = child0.getNextSibling();
			child1 = child1.getNextSibling();
		}
		return true;
	}

	/**
	 * DOM ノード(子ノードを全て含む)を削除します。
	 * @param node
	 */
	public static void removeThis(final Node node) {
		node.getParentNode().removeChild(node);
	}

	/**
	 * 指定された DOM ノードの全ての子ノードを削除します。
	 * @param node
	 */
	public static void removeChildren(final Node node) {
		Node child = null;
		while ((child = node.getFirstChild()) != null) {
			node.removeChild(child);
		}
	}

	/**
	 * 指定された DOM ノードのみを削除します。子ノードは指定された DOM ノードの位置に挿入されます。
	 * @param node
	 */
	public static void retainChildren(final Node node) {
		final Node parent = node.getParentNode();
		Node child = null;
		while ((child = node.getFirstChild()) != null) {
			parent.insertBefore(child, node);
		}
		parent.removeChild(node);
	}

	/**
	 * DOM ノードの子ノードにテキストノードを追加します。
	 * @param node
	 * @param text
	 */
	public static void appendText(final Node node, final String text) {
		node.appendChild(getDocumentNode(node).createTextNode(text));
	}

	/**
	 * DOM ドキュメントを返します。
	 * DOM ドキュメントに対して{@link Node#getOwnerDocument()}
	 * の呼び出しが失敗するためにこのメソッドを定義しました。
	 * @param node
	 * @return ドキュメントノード。
	 */
	public static Document getDocumentNode(final Node node) {
		return (node.getNodeType() == Node.DOCUMENT_NODE)
		? (Document) node
				: node.getOwnerDocument();
	}
	
	public static <C extends NodeContext, E extends Exception> AttrValueBranch<C, E> branchByAttrValue(
			final Iterable<String> attrNames,
			final Fn<String, ? extends Fn<? super C, Void, ? extends E>, ? extends E> fn) {
		return new AttrValueBranch<C, E>(attrNames, fn);
	}
	
	public static <C extends NodeContext, E extends Exception> NodeFinder<C, E> findNode(
			final Fn<? super Node, ? extends NodeList, ? extends E> findable,
			final Fn<? super C, Void, ? extends E> firstFn,
			final Fn<? super C, Void, ? extends E> pluralFn) {
		return new NodeFinder<C, E>(findable, firstFn, pluralFn);
	}
	
	/**
	 * 
	 * @param <C>
	 * @param <E>
	 * @param findable
	 * @param fn 委譲先(ポインタは検索結果)。
	 * @return 関数。
	 */
	public static <C extends NodeContext, E extends Exception> NodeFinder<C, E> findNode(
			final Fn<? super Node, ? extends NodeList, ? extends E> findable,
			final Fn<? super C, Void, ? extends E> fn) {
		return new NodeFinder<C, E>(findable, fn, fn);
	}
	
	public static AttrValueGetter<RuntimeException> getAttrValue(final String attrName) {
		return new AttrValueGetter<RuntimeException>(attrName);
	}
	
	public static <C extends NodeContext, E extends Exception> ChildElementInserter<C, E> insertElementToChild(
			final String elementName,
			final Fn<? super C, Void, ? extends E> fn) {
		return new ChildElementInserter<C, E>(elementName, fn);
	}
	
	/**
	 * 
	 * @param <C>
	 * @param elementName エレメント名。
	 * @return 関数。
	 */
	public static <C extends NodeContext> ChildElementInserter<C, RuntimeException> insertElementToChild(
			final String elementName) {
		return new ChildElementInserter<C, RuntimeException>(
				elementName,
				FnUtils.<NodeContext, Void>fix(null));
	}
	
	public static <C extends NodeContext, E extends Exception> ParentElementInserter<C, E> insertElementToParent(
			final String elementName,
			final Fn<? super C, Void, ? extends E> fn) {
		return new ParentElementInserter<C, E>(elementName, fn);
	}
	
	/**
	 * 
	 * @param <C>
	 * @param elementName エレメント名。
	 * @return 関数。
	 */
	public static <C extends NodeContext> ParentElementInserter<C, RuntimeException> insertElementToParent(
			final String elementName) {
		return new ParentElementInserter<C, RuntimeException>(
				elementName,
				FnUtils.<NodeContext, Void>fix(null));
	}
	
	public static <C extends NodeContext, E extends Exception> TemplateCopier<C, E> copyTemplate(
			final GettingEL collectionEL,
			final EL valueEL,
			final Fn<? super C, Void, ? extends E> fn) {
		return new TemplateCopier<C, E>(collectionEL, valueEL, fn);
	}
	
	public static <E extends Exception> NodeFactory<E> nodeFactory(
			final Fn<? super String, ? extends Reader, ? extends E> readerFactory,
			final Fn<XmlTransformerContext, Void, ? extends E> transformer) {
		return new NodeFactory<E>(readerFactory, transformer);
	}
	
	public static AttrRemover<RuntimeException> removeAttr(final String attrName) {
		return new AttrRemover<RuntimeException>(attrName);
	}
	
	public static <C extends NodeContext, E extends Exception> TextReplacer<C, E> replaceText(
			final Fn<? super C, String, ? extends E> fn) {
		return new TextReplacer<C, E>(fn);
	}
	
	public static <C extends NodeContext, E extends Exception> ChildTextReplacer<C, E> replaceTextToChild(
			final Fn<? super C, String, ? extends E> fn) {
		return new ChildTextReplacer<C, E>(fn);
	}
	
	public static <C extends NodeContext, E extends Exception> NodeSeeker<C, E> seekNode(
			final Fn<? super C, Void, ? extends E> fn) {
		return new NodeSeeker<C, E>(fn);
	}
	
	public static <C extends NodeContext, E extends Exception> NodeSetter<C, E> setNode(
			final Fn<? super C, ? extends Node, ? extends E> fn) {
		return new NodeSetter<C, E>(fn);
	}
	
	public static <C extends NodeContext, E extends Exception> AttrValueUpdater<C, E> updateAttrValue(
			final String attrName,
			final Fn<? super C, String, ? extends E> fn) {
		return new AttrValueUpdater<C, E>(attrName, fn);
	}
}
