/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.visitor;

import java.util.Map;
import java.util.Map.Entry;

import woolpack.fn.Fn;

/**
 * インタープリタ デザインパターンの Expression 役と Context 役との間をダブルディスパッチする方式で定義した
 * Visitor デザインパターンの Visitor 役です。
 * 
 * {@link Visitor#visit(Object)}メソッドは
 * まず引数の element をキーとして委譲先を検索します。
 * 最初にマッチした場合は委譲して処理を終了します。
 * 次に引数の element のクラス名の代入可能性で委譲先を検索します。
 * 最初にマッチした場合は委譲して処理を終了します。
 * いずれもマッチしない場合はなにもしないで処理を終了します。
 * このクラスは LSP(The Liskov Substitution Principle) を満たしません。
 * 
 * <br/>適用しているデザインパターン：Visitor, Interpreter(Context役)。
 * 
 * @author nakamura
 * 
 * @param <S> サブコンテキスト。
 * @param <E> 例外。
 */
public class Visitor<S, E extends Exception> {
	private final Fn<Visitor<?, E>, Void, E> echo = new EchoAcceptor<E>();
	private Map<Object, ? extends Fn<? super Visitor<S, E>, Void, ? extends E>> map;
	private S subContext;
	private Object element;
	
	public void visit(final Object element) throws E {
		visit(element, echo);
	}

	/**
	 * {@link Visitor#setElement(Object)}を一時的に設定して委譲し、
	 * 委譲先から復帰したときに呼び出し時の状態に初期化します。
	 * @param element 一時的に置き換えるエレメント。
	 * @param acceptor 委譲先。
	 */
	public void visit(
			final Object element,
			final Fn<? super Visitor<S, E>, Void, ? extends E> acceptor) throws E {
		final Object baseElement = getElement();
		setElement(element);
		try {
			acceptor.exec(this);
		} finally {
			setElement(baseElement);
		}
	}

	public void visit() throws E {
		{
			final Fn<? super Visitor<S, E>, Void, ? extends E> a = map.get(element);
			if (a != null) {
				a.exec(this);
				return;
			}
		}
		if (element == null) {
			return;
		}
		final Class clazz = element.getClass();
		for (final Entry<Object, ? extends Fn<? super Visitor<S, E>, Void, ? extends E>> entry : map.entrySet()) {
			if ((entry.getKey() instanceof Class)
					&& ((Class<?>) entry.getKey()).isAssignableFrom(clazz)) {
				entry.getValue().exec(this);
				break;
			}
		}
	}

	public S getSubContext() {
		return subContext;
	}
	public void setSubContext(final S context) {
		this.subContext = context;
	}
	public Map<Object, ? extends Fn<? super Visitor<S, E>, Void, ? extends E>> getMap() {
		return map;
	}
	
	/**
	 * 
	 * @param map 変数またはそのクラスオブジェクトと委譲先の対応表。
	 */
	public void setMap(final Map<Object, ? extends Fn<? super Visitor<S, E>, Void, ? extends E>> map) {
		this.map = map;
	}
	public Object getElement() {
		return element;
	}
	public void setElement(final Object element) {
		this.element = element;
	}
}
