/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;

import javax.sql.DataSource;

import woolpack.fn.Fn;

/**
 * 委譲先からクエリを取得して
 * {@link Statement#execute(String)}を実行し、
 * 返却値の生成を委譲する{@link Fn}です。
 * このクラスのメソッドは最初に発生した例外を再スローするため、メソッド内で続けて発生した例外の情報は失われます。
 * @author nakamura
 *
 * @param <C>
 */
public class StatementInput<C, R> implements Fn<C, R, Exception> {
	private DataSource dataSource;
	private Fn<? super C, ? extends String, ? extends Exception> queryFactory;
	private Fn<? super Statement, ? extends R, ? extends Exception> converter;
	
	/**
	 * 
	 * @param dataSource データソース。
	 * @param queryFactory クエリ生成の委譲先。
	 * @param converter 変換の委譲先。
	 */
	public StatementInput(
			final DataSource dataSource,
			final Fn<? super C, ? extends String, ? extends Exception> queryFactory,
			final Fn<? super Statement, ? extends R, ? extends Exception> converter) {
		this.dataSource = dataSource;
		this.queryFactory = queryFactory;
		this.converter = converter;
	}

	public R exec(final C c) throws Exception {
		SQLException e0 = null;
		try {
			final Connection connection = dataSource.getConnection();
			try {
				final Statement statement = connection.createStatement();
				try {
					statement.execute(queryFactory.exec(c));
					return converter.exec(statement);
				} catch (final SQLException e1) {
					e0 = e1;
				} finally {
					statement.close();
				}
			} catch (final SQLException e1) {
				if (e0 == null) {
					e0 = e1;
				}
			} finally {
				connection.close();
			}
		} catch (final SQLException e1) {
			if (e0 == null) {
				e0 = e1;
			}
		}
		// 委譲先でSQLException以外の例外が発生した場合は以下のブロックは実行されない。
		if (e0 != null) {
			throw e0;
		} else {
			return null;
		}
	}

	public DataSource getDataSource() {
		return dataSource;
	}
	public void setDataSource(final DataSource dataSource) {
		this.dataSource = dataSource;
	}
	public Fn<? super C, ? extends String, ? extends Exception> getQueryFactory() {
		return queryFactory;
	}
	public void setQueryFactory(final Fn<? super C, ? extends String, ? extends Exception> queryFactory) {
		this.queryFactory = queryFactory;
	}
	public Fn<? super Statement, ? extends R, ? extends Exception> getConverter() {
		return converter;
	}
	public void setConverter(final Fn<? super Statement, ? extends R, ? extends Exception> converter) {
		this.converter = converter;
	}
}
