/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.sql.fn;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.util.ArrayList;
import java.util.List;

import woolpack.el.EL;
import woolpack.el.PropertyEL;
import woolpack.fn.Fn;
import woolpack.utils.PropertyUtils;

/**
 * 現在の行をBeanにコピーして返す{@link Fn}です。
 * このクラスは{@link PropertyUtils}を利用してSQLの命名規則をJavaの命名規則に変換して類推します。
 * このクラスのインスタンスは{@link ResultSet}とBeanへの対応をキャッシュします。
 * このクラスのひとつのインスタンスは一種類のStatementに対応する{@link ResultSet}だけを処理することができます。
 * 一種類のStatementであれば複数のスレッドで同時に使用することができます。
 * @author nakamura
 *
 * @param <R>
 */
public class BeanResult<R> implements Fn<ResultSet, R, Exception> {
	private Class<R> clazz;
	private Fn<String, EL, ? extends Exception> elFactory;
	private List<EL> list;
	private int size;
	
	/**
	 * 
	 * @param clazz 生成するBeanのクラスオブジェクト。
	 * @param elFactory 属性値から{@link EL}のファクトリ。これを使用してBeanのインスタンスに値を設定する。型変換処理をカスタマイズするために使用する。
	 */
	public BeanResult(final Class<R> clazz, final Fn<String, EL, ? extends Exception> elFactory) {
		this.clazz = clazz;
		this.elFactory = elFactory;
	}
	
	public BeanResult(final Class<R> clazz) {
		this(clazz, new Fn<String, EL, RuntimeException>() {
			public EL exec(final String c) {
				return new PropertyEL(c);
			}
		});
	}

	public R exec(final ResultSet c) throws Exception{
		final R r;
		try {
			r = clazz.newInstance();
		} catch (final InstantiationException e) {
			throw new IllegalStateException(e);
		} catch (final IllegalAccessException e) {
			throw new IllegalStateException(e);
		}
		if (list == null) {
			final BeanInfo beanInfo;
			try {
				beanInfo = Introspector.getBeanInfo(clazz);
			} catch (final IntrospectionException e) {
				throw new IllegalArgumentException(e);
			}
			list = new ArrayList<EL>();
			final ResultSetMetaData metaData = c.getMetaData();
			size = metaData.getColumnCount();
			myLoop:for (int i = 0; i < size; i++) {
				final String dbColName = metaData.getColumnName(i+1);
				for (final PropertyDescriptor p : beanInfo.getPropertyDescriptors()) {
					if (
							dbColName.equals(p.getName()) ||
							dbColName.toUpperCase().equals(PropertyUtils.toSQLName(p.getName()))) {
						list.add(elFactory.exec(p.getName()));
						continue myLoop;
					}
				}
				list.add(null);
			}
		}
		for (int i = 0; i < size; i++) {
			final EL el = list.get(i);
			if (el != null) {
				el.setValue(r, c.getObject(i + 1));
			}
		}
		return r;
	}

	public Class<R> getClazz() {
		return clazz;
	}
	public Fn<String, EL, ? extends Exception> getElFactory() {
		return elFactory;
	}
}
