/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.html;

import java.util.Collection;
import java.util.regex.Pattern;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import woolpack.bool.BoolUtils;
import woolpack.convert.ConvertUtils;
import woolpack.el.EL;
import woolpack.el.GettingEL;
import woolpack.fn.Fn;
import woolpack.fn.FnUtils;
import woolpack.xml.NodeContext;
import woolpack.xml.NodeFindUtils;
import woolpack.xml.XmlUtils;

/**
 * ユーティリティです。
 * 型推論で表記を簡略するためのスタティックメソッドと変数を含みます。
 * 
 * @author nakamura
 * 
 */
public final class HtmlUtils {

	public static final SpaceCompressor<RuntimeException> COMPRESS_SPACE = new SpaceCompressor<RuntimeException>();
	public static final MergeCellAll<RuntimeException> MERGE_CELL_ALL = new MergeCellAll<RuntimeException>();
	public static final CaseNormalizer<RuntimeException> NORMALIZE_CASE = new CaseNormalizer<RuntimeException>();
	public static final CommentRemover<RuntimeException> REMOVE_COMMENT = new CommentRemover<RuntimeException>();
	public static final ThisAndTextsRemover<RuntimeException> REMOVE_THIS_AND_TEXTS = new ThisAndTextsRemover<RuntimeException>();

	static final Fn<Node, NodeList, RuntimeException> TR_LIST = NodeFindUtils.list(
	FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("TR")), false);

	static final Fn<Node, Node, RuntimeException> TR_ONE = NodeFindUtils.one(
	FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("TR")));

	static final Fn<Node, NodeList, RuntimeException> TD_LIST = NodeFindUtils.list(
			FnUtils.join(XmlUtils.GET_NODE_NAME, BoolUtils.checkEquals("TD")), false);

	private HtmlUtils() {
	}

	/**
	 * 指定された DOM ノードを削除し、さらにその後ろにあるテキストノードを全て削除します。
	 * ラジオボタン・チェックボックスを削除するために使用します。
	 * 
	 * @param node
	 */
	public static void removeThisAndText(final Node node) {
		Node nextNode = node.getNextSibling();
		while (nextNode != null && nextNode.getNodeType() == Node.TEXT_NODE) {
			XmlUtils.removeThis(nextNode);
			nextNode = node.getNextSibling();
		}
		XmlUtils.removeThis(node);
	}
	
	public static AutoUpdater updateAuto(
			final Iterable<String> attrNames,
			final GettingEL componentEL,
			final GettingEL configEL,
			final Collection<Class<?>> atomCollection,
			final GettingEL errorEL) {
		return new AutoUpdater<RuntimeException>(attrNames, componentEL, configEL, atomCollection, errorEL);
	}
	
	public static AutoUpdater<RuntimeException> updateAuto(
			final Iterable<String> attrNames,
			final GettingEL componentEL,
			final GettingEL configEL) {
		return new AutoUpdater<RuntimeException>(attrNames, componentEL, configEL);
	}
	
	public static <C extends NodeContext, E extends Exception> BranchPropertyCounter<C, E> branchPropertyCount(
			final EL el,
			final Iterable<String> attrNames,
			final Fn<? super C, Void, ? extends E> firstFn,
			final Fn<? super C, Void, ? extends E> pluralFn) {
		return new BranchPropertyCounter<C, E>(el, attrNames, firstFn, pluralFn);
	}
	
	public static RowAttrConverter convertRowAttr(final String attrName, final String[] attrValueArray) {
		return new RowAttrConverter(attrName, attrValueArray);
	}
	
	public static HiddenAppender<RuntimeException> hiddenAllToChild(
			final GettingEL mapEL,
			final Collection<String> excludeProperties) {
		return new HiddenAppender<RuntimeException>(mapEL, excludeProperties);
	}
	
	public static RowIndexInserter insertRowIndex(final String headValue) {
		return new RowIndexInserter(headValue);
	}
	
	public static RadioRemaker<RuntimeException> makeRadio(final GettingEL mapEL) {
		return new RadioRemaker<RuntimeException>(mapEL);
	}
	
	public static SelectRemaker<RuntimeException> makeSelect(final GettingEL mapEL) {
		return new SelectRemaker<RuntimeException>(mapEL);
	}
	
	public static MergeCell mergeCell(final int colIndex) {
		return new MergeCell(colIndex);
	}
	
	public static SelectedValueUpdater<RuntimeException> updateToSelectedValue(
			final GettingEL valueEL,
			final GettingEL mapEL) {
		return new SelectedValueUpdater<RuntimeException>(valueEL, mapEL);
	}
	
	public static ValueUpdater<RuntimeException> updateValue(
			final GettingEL valueEL,
			final GettingEL mapEL,
			final boolean selectFlag) {
		return new ValueUpdater<RuntimeException>(valueEL, mapEL, selectFlag);
	}
	
	/**
	 * selected または checked 属性の有無を変更するモードです。
	 * @param valueEL 値の取得先。
	 * @param mapEL 値とラベルの{@link java.util.Map}への参照。
	 * @return 関数。
	 */
	public static ValueUpdater<RuntimeException> updateValue(
			final GettingEL valueEL,
			final GettingEL mapEL) {
		return new ValueUpdater<RuntimeException>(valueEL, mapEL, true);
	}
	
	/**
	 * selected または checked 属性の有無を変更するモードです。
	 * @param valueEL
	 *            値の取得先への参照。コンテキスト役に対して副作用が発生すべきではない。
	 * @return 関数。
	 */
	public static ValueUpdater<RuntimeException> updateValue(
			final GettingEL valueEL) {
		return new ValueUpdater<RuntimeException>(valueEL, null, true);
	}

	/**
	 * 拡張子を取り除く{@link Fn}を返します。
	 * @param elementName
	 * @param attrName
	 * @return 拡張子を取り除く{@link Fn}。
	 */
	public static Fn<NodeContext, Void, RuntimeException> removeExtension(final String elementName, final String attrName) {
		return XmlUtils.findNode(
			NodeFindUtils.list(
					new Fn<Node, Boolean, RuntimeException>() {
						public Boolean exec(final Node c) {
							return c.getNodeType() == Node.ELEMENT_NODE
							&& elementName.equals(c.getNodeName())
							&& ((Element) c).hasAttribute(attrName);
						}
					}, true),
			XmlUtils.updateAttrValue(attrName,
				FnUtils.join(
						XmlUtils.getAttrValue(attrName),
						ConvertUtils.convertRegExp(Pattern.compile("^([^\\.]+)\\.[^\\.]+$"), "$1"))));
	}
}
