/*
 * Copyright 2006 Takahiro Nakamura.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package woolpack.adapter;

import java.lang.reflect.Member;
import java.util.Map;

import ognl.TypeConverter;
import woolpack.fn.Fn;
import woolpack.typeconvert.ConvertContext;
import woolpack.typeconvert.DelegationIfNecessityConverter;
import woolpack.typeconvert.SettingFnConverter;
import woolpack.typeconvert.ToArrayConverter;
import woolpack.typeconvert.ToCollectionDecompositionConverter;
import woolpack.typeconvert.ToCollectionViewConverter;
import woolpack.typeconvert.ToMapViewConverter;

/**
 * 一覧を表すオブジェクトを変換する OGNL 用型コンバータです。
 * 本クラスは OGNL のライブラリを使用します。
 * <br/>適用しているデザインパターン：Adapter。
 * 
 * @author nakamura
 * 
 */
class OgnlCollectionTypeConverter implements TypeConverter {
	private final TypeConverter converter;

	/**
	 * @param converter 単純型の変換器。
	 */
	OgnlCollectionTypeConverter(final TypeConverter converter) {
		this.converter = converter;
	}

	/**
	 * OGNL の API から呼び出されます。(called)。
	 * 
	 * @param context OGNL のコンテキスト。
	 * @param target 格納先のオブジェクト。
	 * @param member 不明。
	 * @param propertyName 変換元のプロパティ名。
	 * @param beforeValue 変換前のオブジェクト。
	 * @param toType 変換先の型。
	 * @return 変換後のオブジェクト。
	 */
	public Object convertValue(final Map context, final Object target,
			final Member member, final String propertyName,
			final Object beforeValue, final Class toType) {

		final ConvertContext convertContext = new ConvertContext();
		convertContext.setPropertyName(propertyName);
		convertContext.setToType(toType);
		convertContext.setValue(beforeValue);

		new SettingFnConverter(
		new DelegationIfNecessityConverter(
		new ToArrayConverter(
		new ToCollectionViewConverter(
		new ToCollectionDecompositionConverter(
		new ToMapViewConverter(
				new Fn<ConvertContext, Void, RuntimeException>() {
					public Void exec(final ConvertContext c) {
						c.setValue(converter.convertValue(
								context,
								target,
								member,
								c.getPropertyName(),
								c.getValue(),
								c.getToType()));
						return null;
					}
				})))))).exec(convertContext);
		return convertContext.getValue();
	}
}
