/* 
 * Copyright 2007 Tatooine Project <http://tatooine.sourceforge.jp/> 
 *  
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *  
 *     http://www.apache.org/licenses/LICENSE-2.0
 *  
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.sf.tatooine.base.utils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;

/**
 * NameUtils.
 *
 * @author  Tooru Noda
 * @version 1.0 2007/04/19
 * @since   JDK5.0 Tiger
 */
public final class NameUtils {

	/** アンダースコア化時変換候補抽出用(先頭大文字) */
	private static final Pattern PTN_UNDERSCORE1 = Pattern.compile("([A-Z][^A-Z]+)");
	
	/** アンダースコア化時変換候補抽出用(大文字の連続) */
	private static final Pattern PTN_UNDERSCORE2 = Pattern.compile("([A-Z][A-Z]+)");
	
	/** クラス名化時変換候補抽出用 */
	private static final Pattern PTN_CLASSIFY = Pattern.compile("(_[^_]*)");
	
	private NameUtils() { }
	
	/**
	 * 先頭以外の大文字の直前に"_"を挿入し、大文字は小文字に変換します.
	 * 
	 * @param name	変換前の名前
	 * @return	変換後の名前
	 */
	public static String underscore(String name) {
		
		StringBuffer buf = new StringBuffer();
		String from = StringUtils.capitalize(name);
		Matcher m1 = PTN_UNDERSCORE1.matcher(from);
		while (m1.find()) {
			String s = m1.group();
			m1.appendReplacement(buf, "_" + s);
		}
		m1.appendTail(buf);
		Matcher m2 = PTN_UNDERSCORE2.matcher(buf.toString());
		buf = new StringBuffer();
		while (m2.find()) {
			String s = m2.group();
			m2.appendReplacement(buf, "_" + StringUtils.capitalize(s));
		}
		m2.appendTail(buf);
		String result = StringUtils.lowerCase(buf.toString());
		if (result.startsWith("_")) {
			result = result.substring(1);
		}
		result = result.replaceAll("__", "_");
		return result;
	}
	
	/**
	 * "_"の直後の文字を大文字にし、"_"を削除します.
	 * 
	 * @param name	変換前の名前
	 * @return	変換後の名前
	 */
	public static String classify(String name) {
		
		StringBuffer buf = new StringBuffer();
		String from = StringUtils.capitalize(name);
		Matcher m = PTN_CLASSIFY.matcher(from);
		while (m.find()) {
			String s = m.group();
			m.appendReplacement(buf, StringUtils.capitalize(s.substring(1)));
		}
		m.appendTail(buf);
		return buf.toString();
	}
}
