/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.ContentHandler;
import org.xml.sax.DTDHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;

import shohaku.core.lang.Boxing;

/**
 * XMLドキュメントの解析に使用する、Simple API for XML (SAX) のラッパーを提供します。<br>
 * ライブラリ上での仕様の対応した情報の保管や制限を実現する為に提供されています。<br>
 * SAXへの制御はこのクラスを経由してのみ可能です
 */
public class SAXDocumentParser {

    /* javax.xml.parsers.SAXParserFactory */
    private SAXParserFactory saxParserFactory;

    /* javax.xml.parsers.SAXParser */
    private SAXParser saxParser;

    /* org.xml.sax.XMLReader */
    private XMLReader xmlReader;

    /* org.xml.sax.EntityResolver。 */
    private EntityResolver entityResolver;

    /* org.xml.sax.DTDHandler。 */
    private DTDHandler dtdHandler;

    /* org.xml.sax.ErrorHandler。 */
    private ErrorHandler errorHandler;

    /* SAXの拡張機能を格納する。 */
    private Map features = new HashMap();

    /* SAXのプロパティを格納する。 */
    private Map properties = new HashMap();

    private Boolean isNamespaceAware;

    private Boolean isValidating;

    private final Ginkgo ginkgo;

    SAXDocumentParser(Ginkgo ginkgo) {
        this.ginkgo = ginkgo;
    }

    /**
     * @param input
     * @throws IOException
     * @throws SAXException
     */
    void parse(InputSource input) throws IOException, SAXException {

        prepare();
        XMLReader xmlReader = getXMLReader();
        xmlReader.parse(input);

    }

    /* 解析処理プロセス開始時のフィールド設定を行います。 */
    private void prepare() throws SAXException {

        try {
            SAXParserFactory _saxParserFactory = getSAXParserFactory();
            Boolean _isValidating = getValidating();
            if (_isValidating != null) {
                _saxParserFactory.setValidating(_isValidating.booleanValue());
            }
            Boolean _isNamespaceAware = getNamespaceAware();
            if (_isNamespaceAware != null) {
                _saxParserFactory.setNamespaceAware(_isNamespaceAware.booleanValue());
            }

            for (Iterator i = features.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                String name = (String) e.getKey();
                boolean value = ((Boolean) e.getValue()).booleanValue();
                _saxParserFactory.setFeature(name, value);
            }

            SAXParser _saxParser = getSAXParser();
            for (Iterator i = properties.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                String name = (String) e.getKey();
                Object value = e.getValue();
                _saxParser.setProperty(name, value);
            }

            XMLReader _xmlReader = getXMLReader();
            _xmlReader.setContentHandler(newContentHandler());

            EntityResolver _entityResolver = getEntityResolver();
            if (_entityResolver != null) {
                _xmlReader.setEntityResolver(_entityResolver);
            }
            DTDHandler _dtdHandler = getDTDHandler();
            if (_dtdHandler != null) {
                _xmlReader.setDTDHandler(_dtdHandler);
            }
            ErrorHandler _errorHandler = getErrorHandler();
            if (_errorHandler != null) {
                _xmlReader.setErrorHandler(_errorHandler);
            } else {
                _xmlReader.setErrorHandler(newDefaultErrorHandler());
            }

        } catch (ParserConfigurationException e) {
            ginkgo.getLogger().error("SAXDocumentParser.setFeature:", e);
            throw new SAXException(e);
        }

    }

    /* 解析処理に使用するSAXParserFactoryを生成して返却します。 */
    private SAXParserFactory getSAXParserFactory() throws SAXException {
        try {
            if (saxParserFactory == null) {
                saxParserFactory = SAXParserFactory.newInstance();
            }
            return saxParserFactory;
        } catch (FactoryConfigurationError e) {
            ginkgo.getLogger().error("SAXDocumentParser.getSAXParserFactory:", e);
            throw new SAXException(e.getException());
        }
    }

    /* 解析処理に使用するSAXParserを生成して返却します。 */
    private SAXParser getSAXParser() throws SAXException {
        try {
            if (saxParser == null) {
                saxParser = getSAXParserFactory().newSAXParser();
            }
            return saxParser;

        } catch (ParserConfigurationException e) {
            ginkgo.getLogger().error("SAXDocumentParser.newSAXParser:", e);
            throw new SAXException(e);
        }
    }

    /* 解析処理に使用するXMLReaderを生成して返却します。 */
    private XMLReader getXMLReader() throws SAXException {
        if (xmlReader == null) {
            xmlReader = getSAXParser().getXMLReader();
        }
        return xmlReader;
    }

    /**
     * XMLReader の基本となる実装で要求された特定の機能の値を返却します。
     * 
     * @param name
     *            取り出される機能の名前
     * @return 要求された機能の値
     * @see javax.xml.parsers.SAXParserFactory#getFeature(java.lang.String)
     */
    public boolean getFeature(String name) {
        Boolean feature = (Boolean) features.get(name);
        return (feature != null) ? feature.booleanValue() : false;
    }

    /**
     * XMLReader の基本となる実装に特定の機能の値を設定します。
     * 
     * @param name
     *            設定される機能の名前
     * @param value
     *            設定される機能の値
     * @see javax.xml.parsers.SAXParserFactory#setFeature(java.lang.String, boolean)
     */
    public void setFeature(String name, boolean value) {
        features.put(name, Boxing.box(value));
    }

    /**
     * XMLReader の基本となる実装で要求された特定のプロパティを返却します。
     * 
     * @param name
     *            取り出されるプロパティの名前
     * @return 要求されたプロパティの値
     * @see javax.xml.parsers.SAXParser#getProperty(java.lang.String)
     */
    public Object getProperty(String name) {
        return properties.get(name);
    }

    /**
     * XMLReader の基本となる実装に特定のプロパティを設定します。
     * 
     * @param name
     *            設定されるプロパティの名前
     * @param value
     *            設定されるプロパティの値
     * @see javax.xml.parsers.SAXParser#setProperty(java.lang.String, java.lang.Object)
     */
    public void setProperty(String name, Object value) {
        properties.put(name, value);
    }

    /* 解析に使用するorg.xml.sax.ContentHandlerを生成して返却します。 */
    private ContentHandler newContentHandler() {
        return new SAXContentHandler(this.ginkgo);
    }

    /* 解析に使用するデフォルトのorg.xml.sax.ErrorHandlerを生成して返却します。 */
    private ErrorHandler newDefaultErrorHandler() {
        return new DefaultErrorHandler(this.ginkgo);
    }

    /**
     * 解析に使用する<code>org.xml.sax.EntityResolver</code>を格納します。
     * 
     * @param entityResolver
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setEntityResolver(EntityResolver entityResolver) {
        this.entityResolver = entityResolver;
    }

    /**
     * 解析に使用する<code>org.xml.sax.EntityResolver</code>を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public EntityResolver getEntityResolver() {
        return this.entityResolver;
    }

    /**
     * 解析に使用する<code>org.xml.sax.DTDHandler</code>を格納します。
     * 
     * @param dtdHandler
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setDTDHandler(DTDHandler dtdHandler) {
        this.dtdHandler = dtdHandler;
    }

    /**
     * 解析に使用する<code>org.xml.sax.DTDHandler</code>を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public DTDHandler getDTDHandler() {
        return this.dtdHandler;
    }

    /**
     * 解析に使用する<code>org.xml.sax.ErrorHandler</code> を格納します。
     * 
     * @param errorHandler
     *            設定値、設定を行わない場合<code>null</code>
     */
    public void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    /**
     * 解析に使用する<code>org.xml.sax.ErrorHandler</code> を返却します。
     * 
     * @return 設定値、設定を行わない場合<code>null</code>
     */
    public ErrorHandler getErrorHandler() {
        return this.errorHandler;
    }

    /**
     * isNamespaceAware を返却します。
     * 
     * @return isNamespaceAware。
     */
    public Boolean getNamespaceAware() {
        return isNamespaceAware;
    }

    /**
     * isNamespaceAware を格納します。
     * 
     * @param isNamespaceAware
     *            isNamespaceAware。
     */
    public void setNamespaceAware(Boolean isNamespaceAware) {
        this.isNamespaceAware = isNamespaceAware;
    }

    /**
     * isValidating を返却します。
     * 
     * @return isValidating。
     */
    public Boolean getValidating() {
        return isValidating;
    }

    /**
     * isValidating を格納します。
     * 
     * @param isValidating
     *            isValidating。
     */
    public void setValidating(Boolean isValidating) {
        this.isValidating = isValidating;
    }

    /*
     * class
     */

    /* Default implementation of the ErrorHandler interface. */
    private static class DefaultErrorHandler implements ErrorHandler {

        private final Ginkgo ginkgo;

        DefaultErrorHandler(Ginkgo ginkgo) {
            this.ginkgo = ginkgo;
        }

        public void warning(SAXParseException e) throws SAXException {
            ginkgo.getLogger().warn("ErrorHandler.warning:", e);
        }

        public void error(SAXParseException e) throws SAXException {
            ginkgo.getLogger().error("ErrorHandler.error:", e);
        }

        public void fatalError(SAXParseException e) throws SAXException {
            ginkgo.getLogger().fatal("ErrorHandler.fatalError:", e);
            throw e;
        }

    }
}
