/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo;

import java.io.IOException;
import java.io.InputStream;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import shohaku.core.helpers.Eval;

/**
 * <p>
 * XMLドキュメントをオブジェクトにマッピングし生成されたオブジェクトを構造化する実行機能を提供します。<br>
 * XMLドキュメントからのデータ生成に着目したライブラリです。
 * </p>
 * <p>
 * このライブラリはXMLで定義された情報を<code>shohaku.ginkgo.NodeCompositeRule</code>の指定するルールに基づき<code>shohaku.ginkgo.Node</code>
 * のツリー構造に変換します。 <br>
 * 値型の<code>shohaku.ginkgo.Node</code>は自身と内包する要素に基づきオブジェクトを生成します。 <br>
 * 複数のノードを再起的に合成する事により、単純なデータ型の集合から複雑なデータ構造を生成します。 <br>
 * また一度定義されたノードは一定の依存関係を含むものの相対的に独立するため複数の文脈で再利用することができます。
 * </p>
 * <p>
 * <code>Ginkgo</code>のデフォルト実装には基本型と利用頻度が高いと思われるデータ型、および大半のオブジェクトが生成可能な汎用型のノードが提供されています。 <br>
 * また複雑な構造を容易に生成するために反復処理や条件分岐を実現する制御インターフェースが提供されています。 <br>
 * デフォルトノード以外の独自のノードの追加やデフォルトノードとの交換、使用するノードの選択、制約の追加も容易に実現できます。<br>
 * またXMLタグ名の交換やXML属性に対するデフォルト値の指定や別名の指定を定義レベルで指定出来ます。<br>
 * この仕様の応用からカスタマイズされたデータ構造の再定義や、汎用ノードを基に特殊なノードを定義する事も可能です。
 * </p>
 * <p>
 * <code>Ginkgo</code>は汎用的なデータ構造を基に複雑なデータ構造を生成したい場合に有用に設計されています。 <br>
 * 解決を求められている問題領域によっては他のAPIの使用を検討してください。
 * </p>
 * <p>
 * 使用例として<code>shohaku.kosho</code>パッケージを参照する事が出来ます。
 * </p>
 */
public class Ginkgo {

    /* apache commons logging. */
    private static final Log staticLog = LogFactory.getLog(Ginkgo.class);

    /* 標準のログ. */
    private Log log;

    /* 解析処理に使用するSAXパーサー。 */
    private SAXDocumentParser saxDocumentParser;

    /* 解析処理に使用するClassLoader. */
    private ClassLoader classLoader;

    /* Document. */
    private Document document;

    /* DocumentCompositeRule. */
    private DocumentCompositeRule documentCompositeRule;

    /* NodeCompositeRule. */
    private NodeCompositeRule nodeCompositeRule;

    /* 解析処理内でエラーが発生した場合に例外を発生させるかを指定する（デフォルトはfalse）。 */
    private boolean errThrowable;

    /**
     * デフォルトコンストラクタ。
     */
    public Ginkgo() {
        this(newDefaultDocumentCompositeRule());
    }

    /**
     * ドキュメントの合成ルールを指定して初期化します。
     * 
     * @param docRule
     *            ドキュメントの合成ルール
     */
    public Ginkgo(DocumentCompositeRule docRule) {
        if (null == docRule) {
            throw new NullPointerException("DocumentCompositeRule is null");
        }
        initialize(docRule);
    }

    /* このオブジェクトを初期化します。 */
    private void initialize(DocumentCompositeRule docRule) {
        this.documentCompositeRule = docRule;
        this.log = staticLog;
        this.saxDocumentParser = new SAXDocumentParser(this);
        this.classLoader = Ginkgo.class.getClassLoader();
        this.nodeCompositeRule = null;
        this.errThrowable = false;
    }

    /*
     * Parser
     */

    /**
     * 解析処理を実行します。
     * 
     * @param rule
     *            構成ルール
     * @param inStream
     *            XMLデータ入力ストリーム
     * @throws GinkgoException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(NodeCompositeRule rule, InputStream inStream) {
        parse(rule, inStream, null);
    }

    /**
     * 解析処理を実行します。
     * 
     * @param rule
     *            構成ルール
     * @param inStream
     *            XMLデータ入力ストリーム
     * @param parent
     *            親のドキュメントオブジェクト
     * @throws GinkgoException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(NodeCompositeRule rule, InputStream inStream, Document parent) {
        if (parent != null && !parent.isPreserve()) {
            parent = parent.getPreserveDocument();
        }
        parse(rule, new InputSource(inStream), parent);
    }

    /**
     * 解析処理を実行します。
     * 
     * @param rule
     *            構成ルール
     * @param input
     *            XMLデータソース
     * @throws GinkgoException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(NodeCompositeRule rule, InputSource input) {
        parse(rule, input, null);
    }

    /**
     * 解析処理を実行します。
     * 
     * @param rule
     *            構成ルール
     * @param input
     *            XMLデータソース
     * @param parent
     *            親のドキュメントオブジェクト
     * @throws GinkgoException
     *             構成する情報に何らかの誤りが在る場合発生します。
     */
    public void parse(NodeCompositeRule rule, InputSource input, Document parent) {

        if (Eval.isOrNull(rule, input)) {
            throw new NullPointerException("argument is null");
        }

        prepare(rule, parent);

        try {

            getSAXDocumentParser().parse(input);

        } catch (SAXException e) {
            throw new GinkgoException("ginkgo parse err", e);
        } catch (IOException e) {
            throw new GinkgoException("ginkgo parse err", e);
        }

    }

    /* 解析処理プロセス開始時のフィールド設定を行います。 */
    private void prepare(NodeCompositeRule rule, Document parent) {

        this.nodeCompositeRule = rule;
        this.document = new Document(this, rule, parent);
        this.nodeCompositeRule.parse(this, this.document);

    }

    /*
     * getter and seter
     */

    /**
     * 解析処理プロセス内で使用するログを返却します。
     * 
     * @return ログ
     */
    public Log getLogger() {
        return log;
    }

    /**
     * 解析処理プロセス内で使用するログを格納します。
     * 
     * @param log
     *            ログ
     */
    public void setLogger(Log log) {
        if (log == null) {
            throw new NullPointerException();
        }
        this.log = log;
    }

    /**
     * 解析処理に使用するSAXパーサーを返却します。
     * 
     * @return 解析処理に使用するSAXパーサー
     */
    public SAXDocumentParser getSAXDocumentParser() {
        return saxDocumentParser;
    }

    /**
     * 解析処理に使用する<code>ClassLoader</code>を返却します.
     * 
     * @return 解析処理に使用する<code>ClassLoader</code>
     */
    public ClassLoader getClassLoader() {
        return this.classLoader;
    }

    /**
     * 解析処理に使用する<code>ClassLoader</code>を設定します.
     * 
     * @param classLoader
     *            解析処理に使用する<code>ClassLoader</code>
     */
    public void setClassLoader(ClassLoader classLoader) {
        this.classLoader = classLoader;

    }

    /**
     * 解析処理内でエラーが発生した場合に例外を発生させるかを示すフラグを返却します（デフォルトはfalse）。
     * 
     * @return <code>true</code>の場合は解析処理内でエラーが発生した場合に例外を発生させる
     */
    public boolean isErrThrowable() {
        return errThrowable;
    }

    /**
     * 解析処理内でエラーが発生した場合に例外を発生させるかを示すフラグを格納します。<br>
     * デフォルトでは、この値は<code>false</code>に設定されます。
     * 
     * @param errThrowable
     *            <code>true</code>の場合は解析処理内でエラーが発生した場合に例外を発生させる
     */
    public void setErrThrowable(boolean errThrowable) {
        this.errThrowable = errThrowable;
    }

    /**
     * 解析中又は直前に解析したドキュメントを返却します。
     * 
     * @return 解析中又は直前に解析したドキュメント
     */
    public Document getDocument() {
        return document;
    }

    /**
     * 解析中又は直前に使用した構成ルールを返却します。
     * 
     * @return 解析中又は直前に使用した構成ルール
     */
    public NodeCompositeRule getNodeCompositeRule() {
        return nodeCompositeRule;
    }

    /**
     * ドキュメントの合成ルールを返却します。
     * 
     * @return ドキュメントの合成ルール
     */
    public DocumentCompositeRule getDocumentCompositeRule() {
        return documentCompositeRule;
    }

    /*
     * private
     */

    /* 複数のノードとドキュメントを合成するためのルールのデフォルト実装を生成して返却します。 */
    private static DefaultDocumentCompositeRule newDefaultDocumentCompositeRule() {
        return new DefaultDocumentCompositeRule();
    }

}
