/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.collections.MapUtils;

/**
 * XMLドキュメントの解析コンテキスト情報を提供します。
 */
public class DocumentContext {

    /* Ginkgo。 */
    private final Ginkgo ginkgo;

    /* ドキュメント解析の構成ルール。 */
    private final NodeCompositeRule rule;

    /* ルートノードを保管します。 */
    private Node root;

    /* 属性情報を保管します。 */
    private final HashMap attributes = new HashMap();

    /* 初期化します（Ginkgo解析実行時に生成されます）。 */
    DocumentContext(Ginkgo ginkgo, NodeCompositeRule rule) {
        this.ginkgo = ginkgo;
        this.rule = rule;

    }

    /**
     * ドキュメント解析の構成ルールを返却します。
     * 
     * @return ドキュメント解析の構成ルール
     */
    public NodeCompositeRule getNodeCompositeRule() {
        return rule;
    }

    /**
     * <code>Ginkgo</code>を返却します。
     * 
     * @return <code>Ginkgo</code>
     */
    public Ginkgo getGinkgo() {
        return ginkgo;
    }

    /**
     * 解析処理に使用する<code>ClassLoader</code>を返却します.
     * 
     * @return 解析処理に使用する<code>ClassLoader</code>
     */
    public ClassLoader getClassLoader() {
        return getGinkgo().getClassLoader();
    }

    /**
     * ドキュメントのルートノードを返却します。
     * 
     * @return ドキュメントのルートノード
     */
    public Node getRoot() {
        return root;
    }

    /* ドキュメントのルートノードを格納します。 */
    void setRoot(Node root) {
        this.root = root;
    }

    /*
     * Attribute
     */

    /**
     * 属性名の反復子を返却します。
     * 
     * @return 属性名の反復子
     */
    public Iterator getAttributeNameIterator() {
        return this.attributes.keySet().iterator();
    }

    /**
     * 引数の名前を持つ属性が存在する場合<code>true</code>を返却します。
     * 
     * @param name
     *            属性名
     * @return 引数の名前を持つ属性が存在する場合<code>true</code>
     */
    public boolean containsName(String name) {
        return this.attributes.containsKey(name);
    }

    /**
     * 引数の名前を持つ属性値を返却します。 <br>
     * 指定された属性が存在しない場合は<code>null</code>が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public Object getAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.get(name);
    }

    /**
     * 引数の名前を持つ属性が存在する場合その属性値を存在しない場合は第二引数を返却します。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            属性が存在しない場合に返却される値
     * @return 属性値
     */
    public Object getAttribute(String name, Object defaultValue) {
        Object value = getAttribute(name);
        if (value == null) {
            return defaultValue;
        }
        return value;
    }

    /**
     * 接頭辞の一致する属性を<code>Map</code>に格納して返却します。
     * 
     * @param prefix
     *            属性名の接頭辞
     * @return 接頭辞の一致する属性を格納する<code>Map</code>
     */
    public Map getAttributePrefix(String prefix) {
        return MapUtils.putPrefixKey(this.attributes, prefix, new HashMap());
    }

    /**
     * 属性を登録します、既存の属性が存在する場合は既存の属性値を返却し上書きします。 <br>
     * 既存の属性が存在しない場合は<code>null</code>が返されます。
     * 
     * @param name
     *            属性名
     * @param value
     *            属性値
     * @return 既存の属性値、存在しない場合は<code>null</code>
     */
    public Object setAttribute(String name, Object value) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.put(name, value);
    }

    /**
     * 属性を削除し削除された属性値を返却します。 属性が存在しない場合は<code>null</code>が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public Object removeAttribute(String name) {
        if (name == null) {
            throw new NullPointerException();
        }
        return this.attributes.remove(name);
    }

    /**
     * 接頭辞の一致する属性を削除し削除された属性を<code>Map</code>に格納して返却します。
     * 
     * @param prefix
     *            属性名の接頭辞
     * @return 接頭辞の一致する属性の<code>Map</code>
     */
    public Map removeAttributePrefix(String prefix) {
        return MapUtils.removePrefixKey(this.attributes, prefix, new HashMap());
    }

    /**
     * 属性を全て削除します。
     */
    public void removeAttributeAll() {
        this.attributes.clear();
    }

}
