/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo;

import java.util.Iterator;

import shohaku.core.helpers.Eval;

/**
 * ノードの抽象実装を提供し実装の負担を最小化します。
 */
public abstract class AbstractNode implements Node {

    /* ノードのコンテキスト情報 */
    private NodeContext nodeContext;

    /**
     * 指定されたノードの種別に属する場合<code>true</code>を返却します。<br>
     * デフォルト実装として<code>false</code>を返却します。
     * 
     * @param type
     *            検証する種別
     * @return 指定されたノードの種別に属する場合True
     */
    public boolean isType(int type) {
        return false;
    }

    /**
     * ノードのスコープを返却します。
     * 
     * @return ノードのスコープ
     */
    public int getScope() {
        int scope = getNodeScope();
        if (nodeContext.getParent() != null) {
            int parentScope = nodeContext.getParent().getScope();
            scope = (parentScope > scope) ? parentScope : scope;
        }
        return scope;
    }

    /**
     * ノードの固有のスコープを返却します。
     * 
     * @return ノードの固有のスコープ
     */
    protected int getNodeScope() {
        return SCOPE_DOCUMENT;
    }

    /**
     * ノードの識別子を返却します。
     * 
     * @return ノードの識別子
     */
    public String getId() {
        return getAttribute("id");
    }

    /**
     * ノードの値表現を返却します。<br>
     * デフォルト実装として<code>null</code>を返却します。
     * 
     * @return null
     */
    public Object getNodeValue() {
        return null;
    }

    /**
     * ノードのコンテキスト情報を返却します。
     * 
     * @return ノードのコンテキスト情報
     */
    public NodeContext getContext() {
        return nodeContext;
    }

    /**
     * 引数の名前を持つノードの属性を検索し値を返却します。 <br>
     * 属性が存在しない場合<code>null</code>を返します。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public String getAttribute(String name) {
        String value = null;
        NodeAttributesRule rule = getContext().getNodeAttributesRule(name);
        if (rule != null) {
            if (!Eval.isBlank(rule.getAlias())) {
                name = rule.getAlias();
            }
            value = getContext().getTagAttribute(name);
            if (Eval.isBlank(value)) {
                value = rule.getDefaultValue();
            }
        } else {
            value = getContext().getTagAttribute(name);
        }
        return value;
    }

    /**
     * 引数の名前を持つノードの属性を検索し値を返却します。 <br>
     * 属性が存在しない場合<code>defaultValue</code>を返します。
     * 
     * @param name
     *            属性名
     * @param defaultValue
     *            属性が存在しない場合に返却される値
     * @return 属性値
     */
    public String getAttribute(String name, String defaultValue) {
        String value = getAttribute(name);
        return (value != null) ? value : defaultValue;
    }

    /**
     * テキスト情報を評価して返却します。 <br>
     * 全ての子のテキストノードを一つの文字列に連結して返却します。
     * 
     * @return テキスト情報
     */
    public String getText() {
        StringBuffer sb = new StringBuffer();
        Iterator i = nodeContext.textIterator();
        while (i.hasNext()) {
            sb.append(((Node) i.next()).getText());
        }
        return sb.toString();
    }

    /*
     * Event
     */

    /**
     * 解析処理を開始する直前に初期化の通知を受ける。 <br>
     * オーバライドした場合必ず親クラスの同メソッドを呼び出してください。
     * 
     * @param nodeContext
     *            ノードのコンテキスト情報
     * @throws GinkgoException
     *             構成情報例外。
     */
    public void initialize(NodeContext nodeContext) {
        this.nodeContext = nodeContext;
    }

    /**
     * 解析処理プロセス開始の通知を受ける(空実装)。開始タグの解析に対応します。
     * 
     * @throws GinkgoException
     *             構成情報例外。
     */
    public void begin() {
        // no op
    }

    /**
     * 子のノードの追加の通知を受ける(空実装)。 子ノードの解析終了直後に対応します。
     * 
     * @param node
     *            子のノード
     * @throws GinkgoException
     *             構成情報例外。
     */
    public void child(Node node) {
        // no op
    }

    /**
     * 解析処理プロセス完了通知を受ける(空実装)。終了タグの解析に対応します。 <br>
     * この通知が完了した段階で値を持つ情報は値の生成を完了している必要がある。
     * 
     * @throws GinkgoException
     *             構成情報例外。
     */
    public void end() {
        // no op
    }

    /**
     * 解析処理プロセスを完了の通知を受ける(空実装)。ドキュメント解析処理全体の終了に対応します。
     * 
     * @throws GinkgoException
     *             構成情報例外。
     */
    public void finish() {
        // no op
    }

}
