/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.xml.sax;

import java.io.IOException;
import java.io.InputStream;

import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;

/**
 * 指定された外部DTD実体をクラスルートからのパスとして検索を行うエンティティリゾルバです。 <br>
 * 検索で外部DTDを発見できない場合 <code>java.io.IOException</code> が発生します。
 * 
 * @see org.xml.sax.EntityResolver
 */
public class CalssPathEntityResolver implements EntityResolver {

    /* 外部DTD実体へのパス。 */
    private String dtdResource;

    /* 外部DTD実体を読み込むクラスリーダ。 */
    private ClassLoader classLoader;

    /**
     * デフォルトコンストラクタ。
     */
    public CalssPathEntityResolver() {
        this(null, CalssPathEntityResolver.class.getClassLoader());
    }

    /**
     * 外部DTD実体へのパスを指定して初期化します。
     * 
     * @param dtdResource
     *            外部DTD実体へのパス
     */
    public CalssPathEntityResolver(String dtdResource) {
        this(dtdResource, CalssPathEntityResolver.class.getClassLoader());
    }

    /**
     * 外部DTD実体へのパスを指定して初期化します。
     * 
     * @param dtdResource
     *            外部DTD実体へのパス
     */
    public CalssPathEntityResolver(String dtdResource, ClassLoader classLoader) {
        this.dtdResource = dtdResource;
        this.classLoader = classLoader;
    }

    /**
     * 外部DTD実体を設定して <code>InputSource</code> 返却します。
     * 
     * @param publicId
     *            公開識別子
     * @param systemId
     *            システム識別子
     * @see org.xml.sax.EntityResolver#resolveEntity(java.lang.String, java.lang.String)
     */
    public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
        // 特権を使用して取得する
        InputStream stream = (InputStream) java.security.AccessController
                .doPrivileged(new java.security.PrivilegedAction() {
                    public Object run() {
                        InputStream inStream = null;
                        try {
                            inStream = ResourceLoader.getResourceAsStream(CalssPathEntityResolver.this.dtdResource,
                                    CalssPathEntityResolver.this.classLoader);
                        } catch (NoSuchResourceException e) {
                            // no op;
                        }
                        return inStream;
                    }
                });
        if (stream == null) {
            throw new IOException("not find dtd path:" + this.dtdResource);
        }
        return new InputSource(stream);
    }

    /**
     * 外部DTD実体を読み込むクラスリーダを返却します。
     * 
     * @return クラスリーダ
     */
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    /**
     * 外部DTD実体へのパスを返却します。
     * 
     * @return 外部DTD実体へのパス
     */
    public String getDtdResource() {
        return dtdResource;
    }

    /**
     * 外部DTD実体を読み込むクラスリーダを格納します。
     * 
     * @param loader
     *            クラスリーダ
     */
    public void setClassLoader(ClassLoader loader) {
        classLoader = loader;
    }

    /**
     * 外部DTD実体へのパスを格納します。
     * 
     * @param string
     *            外部DTD実体へのパス
     */
    public void setDtdResource(String string) {
        dtdResource = string;
    }

}