/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util.debug;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * 経過時間を指定の出力ストリームへ出力するストップウオッチを提供します。 <br>
 * <br>
 * 簡易的な性能テスト等への利用が想定されています。
 */
public class StopWatch {

    /** 標準ストリームに出力する <code>StopWatch</code> を定義します、複数のスレッドから同時に使用できません。 */
    public static final StopWatch out = new StopWatch(System.out);

    // running states
    private static final int STATE_UNSTARTED = 0;

    private static final int STATE_RUNNING = 1;

    private static final int STATE_STOPPED = 2;

    private static final int STATE_SUSPENDED = 3;

    /* 出力先のプリントライター. */
    private final PrintWriter writer;

    /* タイマーの現在の状態示します。 */
    private int runningState = STATE_UNSTARTED;

    /* 時刻の出力フォーマットを示します。 */
    private String timeFormat = "HH:mm:ss.SSS";

    /* タイマー開始時刻。 */
    private long startTime = -1;

    /* タイマー停止時刻。 */
    private long stopTime = -1;

    /* タイマー時刻位置。 */
    private long splitTime = -1;

    /**
     * 標準の出力ストリームを出力先として初期化します。
     */
    public StopWatch() {
        this(System.out);
    }

    /**
     * 指定されたプリントストリームを出力先として初期化します。
     * 
     * @param pstream
     *            出力先のプリントストリーム
     */
    public StopWatch(PrintStream pstream) {
        this(new PrintWriter(pstream, true));
    }

    /**
     * 指定されたプリントライターを出力先として初期化します。
     * 
     * @param writer
     *            出力先のプリントライター
     */
    public StopWatch(PrintWriter writer) {
        this.writer = writer;
    }

    /**
     * タイマーを開始し開始時刻をミリ秒で出力します。
     * 
     * @return 開始時刻のミリ秒
     */
    public long start() {
        synchronized (this) {
            if (this.runningState == STATE_STOPPED) {
                throw new IllegalStateException("Timer must be reset before being restarted. ");
            }
            if (this.runningState != STATE_UNSTARTED) {
                throw new IllegalStateException("Timer already started. ");
            }
            this.stopTime = -1;
            this.startTime = System.currentTimeMillis();
            this.splitTime = this.startTime;
            this.printDate("start", startTime);
            this.runningState = STATE_RUNNING;
            return this.startTime;
        }
    }

    /**
     * 開始時刻からの経過ミリ秒を出力します（休止状態の経過時間は含まれまん）。
     * 
     * @return 開始時刻からの経過ミリ秒
     */
    public long time() {
        synchronized (this) {
            long time = System.currentTimeMillis() - this.startTime;
            this.printTimeMillis("time ", time);
            return time;
        }
    }

    /**
     * 保存時刻からの経過ミリ秒を出力し、現在の時刻を新たな保存時刻として登録します。
     * 
     * @return 保存時刻からの経過ミリ秒
     */
    public long split() {
        synchronized (this) {
            if (this.runningState != STATE_RUNNING) {
                throw new IllegalStateException("Timer is not running. ");
            }
            long psplit = this.splitTime;
            this.splitTime = System.currentTimeMillis();

            long time = this.splitTime - psplit;
            this.printTimeMillis("split", time);
            return time;
        }
    }

    /**
     * タイマーを終了し開始時刻から終了時刻の経過ミリ秒を出力します。
     * 
     * @return 開始時刻から終了時刻の経過ミリ秒
     */
    public long stop() {
        synchronized (this) {
            if (this.runningState != STATE_RUNNING && this.runningState != STATE_SUSPENDED) {
                throw new IllegalStateException("Timer is not running. ");
            }

            long start = this.startTime;
            long end = System.currentTimeMillis();
            long time = end - start;

            this.printDate("stop ", end);
            if (start < this.splitTime) {
                this.printTimeMillis("split", (end - this.splitTime));
            }
            this.printTimeMillis("time ", time);

            this.startTime = -1;
            this.stopTime = -1;
            this.splitTime = -1;
            this.runningState = STATE_STOPPED;

            return time;
        }
    }

    /**
     * タイマーを休止します。
     */
    public void suspend() {
        synchronized (this) {
            if (this.runningState != STATE_RUNNING) {
                throw new IllegalStateException("Timer must be running to suspend. ");
            }
            this.stopTime = System.currentTimeMillis();
            this.runningState = STATE_SUSPENDED;
        }
    }

    /**
     * タイマーを休止状態から再開します。
     */
    public void resume() {
        synchronized (this) {
            if (this.runningState != STATE_SUSPENDED) {
                throw new IllegalStateException("Timer must be suspended to resume. ");
            }
            this.startTime += (System.currentTimeMillis() - stopTime);
            this.stopTime = -1;
            this.runningState = STATE_RUNNING;
        }
    }

    /**
     * 現在時刻を出力します。
     * 
     * @return 現在時刻のミリ秒
     */
    public long currentTime() {
        synchronized (this) {
            long time = System.currentTimeMillis();
            this.printDate("time ", time);
            return time;
        }
    }

    /**
     * タイマーをリセットします。
     */
    public void reset() {
        synchronized (this) {
            this.runningState = STATE_UNSTARTED;
            this.startTime = -1;
            this.stopTime = -1;
            this.splitTime = -1;
        }
    }

    /**
     * Timer{start=date, stop=date, split=date}の文字列を返します。
     * 
     * @return このオブジェクトの文字列表現。
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer sb = new StringBuffer();
        sb.append("Timer{");
        sb.append("start=");
        sb.append(startTime);
        sb.append(", ");
        sb.append("stop=");
        sb.append(stopTime);
        sb.append(", ");
        sb.append("split=");
        sb.append(splitTime);
        sb.append("}");
        return sb.toString();
    }

    /**
     * 時刻の出力フォーマットを返却します．
     * 
     * @return 時刻の出力フォーマット
     */
    public String getTimeFormat() {
        synchronized (this) {
            return timeFormat;
        }
    }

    /**
     * 時刻の出力フォーマットを格納します．
     * 
     * @param timeFormat
     *            時刻の出力フォーマット
     */
    public void setTimeFormat(String timeFormat) {
        synchronized (this) {
            this.timeFormat = timeFormat;
        }
    }

    /*
     * private
     */

    /* title : time の書式で出力します。 */
    private void printTimeMillis(Object title, long time) {
        //呼出基で同期を取る必要が有ります
        writer.print(title);
        writer.print(" : ");
        writer.print(time);
        writer.println();
    }

    /* 時刻をformatDate(Date)で変換して title : time の書式で出力します。 */
    private void printDate(Object title, long time) {
        //呼出基で同期を取る必要が有ります
        writer.print(title);
        writer.print(" : ");
        writer.print(formatDate(new Date(time)));
        writer.println();
    }

    private String formatDate(Date date) {
        SimpleDateFormat formatter = new SimpleDateFormat(timeFormat);
        return formatter.format(date);
    }

}
