/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.util.debug;

import java.io.InputStream;
import java.io.Serializable;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

import org.apache.commons.logging.Log;

/**
 * 「標準」出力ストリームへ出力する <code>DebugPrintStream</code> を使用した <code>Apache Commons Logging API</code> の拡張ログ機能を提供します。 <br>
 * オブジェクトの内部情報を出力するデバッグ用のシンプルなログ機能です。 <br>
 * <br>
 * 出力には <code>Debug.out</code> および <code>Debug.err</code>
 * を使用するため、各ライターにプリンタ（DebugPrintStream.Printer）を追加する事で出力フォーマットをカスタマイズ出来ます。 <br>
 * <br>
 * このクラスは <code>org.apache.commons.logging.impl.SimpleLog</code> と同等の機能を提供しています。 <br>
 * <br>
 * この機能を使用するにはクラスルートに配備する <code>Logging API</code> の構成ファイル <code>commons-logging.properties</code> に以下の設定をします。 <br>
 * 
 * <pre>
 * org.apache.commons.logging.LogFactory=org.apache.commons.logging.impl.LogFactoryImpl
 * org.apache.commons.logging.Log=shohaku.core.util.debug.IntrospectLog
 * </pre>
 * 
 * @see shohaku.core.util.debug.Debug
 * @see shohaku.core.util.debug.DebugPrintStream
 * @see shohaku.core.util.debug.DebugPrintStream.Printer
 */
public class IntrospectLog implements Log, Serializable {

    /*
     * private static final
     */

    /* システム変数の接頭辞。 */
    private static final String systemPrefix = "shohaku.core.util.debug.introspectlog.";

    /* 設定プロパティ。 */
    private static final Properties introspectLogProperties = new Properties();

    /* 設定プロパティファイル名。 */
    private static final String introspectLogPropertiesName = "shohaku-core-introspectlog.properties";

    /* デフォルトの日付フォーマット。 */
    private static final String DEFAULT_DATE_TIME_FORMAT = "yyyy/MM/dd HH:mm:ss:SSS zzz";

    /*
     * private static
     */

    /* ログ名をメッセージ出力に含めるかを示す。 */
    private static boolean showLogName = false;

    /* ログ名を短縮形で出力するかを示す。 */
    private static boolean showShortName = true;

    /* 時刻をメッセージ出力に含めるかを示す。 */
    private static boolean showDateTime = false;

    /* 時刻をメッセージ出力する際の日付フォーマット。 */
    private static String dateTimeFormat = DEFAULT_DATE_TIME_FORMAT;

    /* 使用する日付フォーマッタ。 */
    private static DateFormat dateFormatter = null;

    /*
     * Log Level Constants
     */

    /** "Trace" レベルのログを示す。 */
    public static final int LOG_LEVEL_TRACE = 1;

    /** "Debug" レベルのログを示す。 */
    public static final int LOG_LEVEL_DEBUG = 2;

    /** "Info" レベルのログを示す。 */
    public static final int LOG_LEVEL_INFO = 3;

    /** "Warn" レベルのログを示す。 */
    public static final int LOG_LEVEL_WARN = 4;

    /** "Error" レベルのログを示す。 */
    public static final int LOG_LEVEL_ERROR = 5;

    /** "Fatal" レベルのログを示す。 */
    public static final int LOG_LEVEL_FATAL = 6;

    /** 全てのログレベルが有効で有ることを示す。 */
    public static final int LOG_LEVEL_ALL = (LOG_LEVEL_TRACE - 1);

    /** 全てのログレベルが無効で有ることを示す。 */
    public static final int LOG_LEVEL_OFF = (LOG_LEVEL_FATAL + 1);

    /*
     * Initializer
     */

    private static String getStringProperty(String name) {
        String prop = null;
        try {
            prop = System.getProperty(name);
        } catch (SecurityException e) {
            //no op
        }
        return (prop == null) ? introspectLogProperties.getProperty(name) : prop;
    }

    private static String getStringProperty(String name, String dephault) {
        String prop = getStringProperty(name);
        return (prop == null) ? dephault : prop;
    }

    private static boolean getBooleanProperty(String name, boolean dephault) {
        String prop = getStringProperty(name);
        return (prop == null) ? dephault : "true".equalsIgnoreCase(prop);
    }

    static {
        InputStream in = getResourceAsStream(introspectLogPropertiesName);
        if (null != in) {
            try {
                introspectLogProperties.load(in);
                in.close();
            } catch (java.io.IOException e) {
                //no op
            }
        }

        showLogName = getBooleanProperty(systemPrefix + "showlogname", showLogName);
        showShortName = getBooleanProperty(systemPrefix + "showShortLogname", showShortName);
        showDateTime = getBooleanProperty(systemPrefix + "showdatetime", showDateTime);

        if (showDateTime) {
            dateTimeFormat = getStringProperty(systemPrefix + "dateTimeFormat", dateTimeFormat);
            try {
                dateFormatter = new SimpleDateFormat(dateTimeFormat);
            } catch (IllegalArgumentException e) {
                dateTimeFormat = DEFAULT_DATE_TIME_FORMAT;
                dateFormatter = new SimpleDateFormat(dateTimeFormat);
            }
        }
    }

    /*
     * Attributes
     */

    /* このログインスタンスに対する名前。 */
    private String logName = null;

    /* 現在のログレベル。 */
    private int currentLogLevel;

    /* このログインスタンスに対する短い名前。 */
    private String shortLogName = null;

    /*
     * Constructor
     */

    /**
     * ログの名前を指定して初期化します。
     * 
     * @param name
     *            ログの名前
     */
    public IntrospectLog(String name) {

        logName = name;

        setLevel(IntrospectLog.LOG_LEVEL_INFO);

        // Set log level from properties
        String lvl = getStringProperty(systemPrefix + "log." + logName);
        int i = String.valueOf(name).lastIndexOf(".");
        while (null == lvl && i > -1) {
            name = name.substring(0, i);
            lvl = getStringProperty(systemPrefix + "log." + name);
            i = String.valueOf(name).lastIndexOf(".");
        }

        if (null == lvl) {
            lvl = getStringProperty(systemPrefix + "defaultlog");
        }

        if ("all".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_ALL);
        } else if ("trace".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_TRACE);
        } else if ("debug".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_DEBUG);
        } else if ("info".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_INFO);
        } else if ("warn".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_WARN);
        } else if ("error".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_ERROR);
        } else if ("fatal".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_FATAL);
        } else if ("off".equalsIgnoreCase(lvl)) {
            setLevel(IntrospectLog.LOG_LEVEL_OFF);
        }

    }

    /*
     * Properties
     */

    /**
     * 現在のログレベルを設定します。
     * 
     * @param currentLogLevel
     *            現在のログレベル
     */
    public void setLevel(int currentLogLevel) {
        this.currentLogLevel = currentLogLevel;

    }

    /**
     * 現在のログレベルを取得します。
     * 
     * @return 現在のログレベル
     */
    public int getLevel() {
        return currentLogLevel;
    }

    /*
     * Logging Methods
     */

    /* 各レベルのログの出力を実行します。 */
    private void log(int type, Object message, Throwable t) {
        // Use a string buffer for better performance
        StringBuffer buf = new StringBuffer();

        // Append date-time if so configured
        if (showDateTime) {
            buf.append(dateFormatter.format(new Date()));
            buf.append(" ");
        }

        // Append a readable representation of the log level
        switch (type) {
        case IntrospectLog.LOG_LEVEL_TRACE:
            buf.append("[TRACE] ");
            break;
        case IntrospectLog.LOG_LEVEL_DEBUG:
            buf.append("[DEBUG] ");
            break;
        case IntrospectLog.LOG_LEVEL_INFO:
            buf.append("[INFO] ");
            break;
        case IntrospectLog.LOG_LEVEL_WARN:
            buf.append("[WARN] ");
            break;
        case IntrospectLog.LOG_LEVEL_ERROR:
            buf.append("[ERROR] ");
            break;
        case IntrospectLog.LOG_LEVEL_FATAL:
            buf.append("[FATAL] ");
            break;
        }

        // Append the name of the log instance if so
        // configured
        if (showShortName) {
            if (shortLogName == null) {
                // Cut all but the last component of the
                // name for both styles
                shortLogName = logName.substring(logName.lastIndexOf(".") + 1);
                shortLogName = shortLogName.substring(shortLogName.lastIndexOf("/") + 1);
            }
            buf.append(String.valueOf(shortLogName)).append(" - ");
        } else if (showLogName) {
            buf.append(String.valueOf(logName)).append(" - ");
        }

        // Append the message
        buf.append(String.valueOf(message));

        // Append stack trace if not null
        if (t != null) {
            buf.append(" <");
            buf.append(t.toString());
            buf.append(">");

            java.io.StringWriter sw = new java.io.StringWriter(1024);
            java.io.PrintWriter pw = new java.io.PrintWriter(sw);
            t.printStackTrace(pw);
            pw.close();
            buf.append(sw.toString());
        }

        // Print to the appropriate destination
        write(type, buf);

    }

    /* ログの書き込みを実行します。 */
    private void write(int type, StringBuffer buffer) {

        if (type > IntrospectLog.LOG_LEVEL_INFO) {
            //WARN, ERROR, FATAL : error
            Debug.err.println(buffer.toString());
        } else {
            //TRACE, DEBUG, INFO : output
            Debug.out.println(buffer.toString());
        }
    }

    /* 指定されたログレベルが現在有効か検証して返す。 */
    private boolean isLevelEnabled(int logLevel) {
        // log level are numerically ordered so can use
        // introspect numeric
        // comparison
        return (logLevel >= currentLogLevel);
    }

    /*
     * Log Implementation
     */

    /**
     * <code>Debug</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @see org.apache.commons.logging.Log#debug(java.lang.Object)
     */
    public final void debug(Object message) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_DEBUG)) {
            log(IntrospectLog.LOG_LEVEL_DEBUG, message, null);
        }
    }

    /**
     * <code>Debug</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     * @see org.apache.commons.logging.Log#debug(java.lang.Object, java.lang.Throwable)
     */
    public final void debug(Object message, Throwable t) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_DEBUG)) {
            log(IntrospectLog.LOG_LEVEL_DEBUG, message, t);
        }
    }

    /**
     * <code>Trace</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @see org.apache.commons.logging.Log#trace(java.lang.Object)
     */
    public final void trace(Object message) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_TRACE)) {
            log(IntrospectLog.LOG_LEVEL_TRACE, message, null);
        }
    }

    /**
     * <code>Trace</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     * @see org.apache.commons.logging.Log#trace(java.lang.Object, java.lang.Throwable)
     */
    public final void trace(Object message, Throwable t) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_TRACE)) {
            log(IntrospectLog.LOG_LEVEL_TRACE, message, t);
        }
    }

    /**
     * <code>Info</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @see org.apache.commons.logging.Log#info(java.lang.Object)
     */
    public final void info(Object message) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_INFO)) {
            log(IntrospectLog.LOG_LEVEL_INFO, message, null);
        }
    }

    /**
     * <code>Info</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     * @see org.apache.commons.logging.Log#info(java.lang.Object, java.lang.Throwable)
     */
    public final void info(Object message, Throwable t) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_INFO)) {
            log(IntrospectLog.LOG_LEVEL_INFO, message, t);
        }
    }

    /**
     * <code>Warn</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @see org.apache.commons.logging.Log#warn(java.lang.Object)
     */
    public final void warn(Object message) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_WARN)) {
            log(IntrospectLog.LOG_LEVEL_WARN, message, null);
        }
    }

    /**
     * <code>Warn</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     * @see org.apache.commons.logging.Log#warn(java.lang.Object, java.lang.Throwable)
     */
    public final void warn(Object message, Throwable t) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_WARN)) {
            log(IntrospectLog.LOG_LEVEL_WARN, message, t);
        }
    }

    /**
     * <code>Error</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @see org.apache.commons.logging.Log#error(java.lang.Object)
     */
    public final void error(Object message) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_ERROR)) {
            log(IntrospectLog.LOG_LEVEL_ERROR, message, null);
        }
    }

    /**
     * <code>Error</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     * @see org.apache.commons.logging.Log#error(java.lang.Object, java.lang.Throwable)
     */
    public final void error(Object message, Throwable t) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_ERROR)) {
            log(IntrospectLog.LOG_LEVEL_ERROR, message, t);
        }
    }

    /**
     * <code>Fatal</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @see org.apache.commons.logging.Log#fatal(java.lang.Object, java.lang.Throwable)
     */
    public final void fatal(Object message) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_FATAL)) {
            log(IntrospectLog.LOG_LEVEL_FATAL, message, null);
        }
    }

    /**
     * <code>Fatal</code> レベルでログを出力します。
     * 
     * @param message
     *            メッセージ
     * @param t
     *            エラーまたは例外
     * @see org.apache.commons.logging.Log#fatal(java.lang.Object, java.lang.Throwable)
     */
    public final void fatal(Object message, Throwable t) {
        if (isLevelEnabled(IntrospectLog.LOG_LEVEL_FATAL)) {
            log(IntrospectLog.LOG_LEVEL_FATAL, message, t);
        }
    }

    /**
     * <code>Debug</code> レベルが有効の場合 <code>true</code> を返します。
     * 
     * @return <code>Debug</code> レベルが有効で有るか
     * @see org.apache.commons.logging.Log#isDebugEnabled()
     */
    public final boolean isDebugEnabled() {
        return isLevelEnabled(IntrospectLog.LOG_LEVEL_DEBUG);
    }

    /**
     * <code>Error</code> レベルが有効の場合 <code>true</code> を返します。
     * 
     * @return <code>Error</code> レベルが有効で有るか
     * @see org.apache.commons.logging.Log#isErrorEnabled()
     */
    public final boolean isErrorEnabled() {
        return isLevelEnabled(IntrospectLog.LOG_LEVEL_ERROR);
    }

    /**
     * <code>Fatal</code> レベルが有効の場合 <code>true</code> を返します。
     * 
     * @return <code>Fatal</code> レベルが有効で有るか
     * @see org.apache.commons.logging.Log#isFatalEnabled()
     */
    public final boolean isFatalEnabled() {
        return isLevelEnabled(IntrospectLog.LOG_LEVEL_FATAL);
    }

    /**
     * <code>Info</code> レベルが有効の場合 <code>true</code> を返します。
     * 
     * @return <code>Info</code> レベルが有効で有るか
     * @see org.apache.commons.logging.Log#isInfoEnabled()
     */
    public final boolean isInfoEnabled() {
        return isLevelEnabled(IntrospectLog.LOG_LEVEL_INFO);
    }

    /**
     * <code>Trace</code> レベルが有効の場合 <code>true</code> を返します。
     * 
     * @return <code>Trace</code> レベルが有効で有るか
     * @see org.apache.commons.logging.Log#isTraceEnabled()
     */
    public final boolean isTraceEnabled() {
        return isLevelEnabled(IntrospectLog.LOG_LEVEL_TRACE);
    }

    /**
     * <code>Warn</code> レベルが有効の場合 <code>true</code> を返します。
     * 
     * @return <code>Warn</code> レベルが有効で有るか
     * @see org.apache.commons.logging.Log#isWarnEnabled()
     */
    public final boolean isWarnEnabled() {
        return isLevelEnabled(IntrospectLog.LOG_LEVEL_WARN);
    }

    /* 引数が示すリソースの入力ストリームを特権で取得して返します。 */
    private static InputStream getResourceAsStream(final String name) {
        return (InputStream) AccessController.doPrivileged(new PrivilegedAction() {
            public Object run() {
                ClassLoader threadCL = Thread.currentThread().getContextClassLoader();
                if (threadCL != null) {
                    return threadCL.getResourceAsStream(name);
                } else {
                    return ClassLoader.getSystemResourceAsStream(name);
                }
            }
        });
    }
}
