/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util.debug;

import java.io.OutputStream;
import java.lang.reflect.Array;
import java.sql.Time;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import shohaku.core.beans.BeanUtilities;
import shohaku.core.io.IntrospectPrintStream;

/**
 * デバック用のプリント文を出力する様々なユーティリティ機能を持つ拡張プリントストリームを提供します。 <br>
 * <br>
 * 親クラスが持つ機能以外に日付のフォーマット出力やコレクションの表形式の出力、タイマー出力等の機能が追加されています。 <br>
 * 以外にも様々なフォーマットで出力するユーティリティメソッドが多数提供されています。 <br>
 */
public class DebugPrintStream extends IntrospectPrintStream {

    /** <code>DebugPrintStream.print(Object o)</code> で使用される出力機能を定義します。 */
    public static interface Printer {

        /**
         * 出力を実行します。 <code>isAssignable(Object)</code> が <code>true</code> の場合のみ呼び出されます。
         * 
         * @param stream
         *            ストリーム
         * @param value
         *            出力する値
         */
        void print(DebugPrintStream stream, Object value);

        /**
         * 指定された値が出力対象とする型である場合 <code>true</code> を返します。
         * 
         * @param value
         *            出力する値
         * @return 出力対象とする型である場合 <code>true</code>
         */
        boolean isAssignable(Object value);
    }

    /*
     * protected static
     */

    /** デフォルトの出力プリンタを格納します。 */
    protected static final List DEFAULT_PRINTERS;
    static {
        List dps = new ArrayList();
        /* 配列プリンタ。 */
        dps.add(new Printer() {
            public void print(DebugPrintStream stream, Object o) {
                stream.printArray(o);
            }

            public boolean isAssignable(Object o) {
                return (o != null && o.getClass().isArray());
            }
        });
        /* Mapプリンタ。 */
        dps.add(new Printer() {
            public void print(DebugPrintStream stream, Object o) {
                stream.printMap((Map) o);
            }

            public boolean isAssignable(Object o) {
                return (o instanceof Map);
            }
        });
        /* Collectionプリンタ。 */
        dps.add(new Printer() {
            public void print(DebugPrintStream stream, Object o) {
                stream.printColl((Collection) o);
            }

            public boolean isAssignable(Object o) {
                return (o instanceof Collection);
            }
        });
        /* 日付型プリンタ。 */
        dps.add(new Printer() {
            public void print(DebugPrintStream stream, Object o) {
                stream.printDate((Date) o);
            }

            public boolean isAssignable(Object o) {
                return (o instanceof java.util.Date);
            }
        });
        /* 正規表現パターンのプリンタ。 */
        dps.add(new Printer() {
            public void print(DebugPrintStream stream, Object o) {
                synchronized (stream) {
                    stream.print(((Pattern) o).pattern());
                    stream.print(',');
                    stream.print(((Pattern) o).flags());
                }
            }

            public boolean isAssignable(Object o) {
                return (o instanceof Pattern);
            }
        });
        DEFAULT_PRINTERS = Collections.unmodifiableList(dps);
    }

    /** このインスタンスで使用する出力プリンタ。 */
    protected final List printers = new ArrayList(DEFAULT_PRINTERS);

    /**
     * 標準の出力ストリームを出力先として初期化します。
     */
    public DebugPrintStream() {
        this(System.out);
    }

    /**
     * 標準の出力ストリームを出力先として初期化します。
     * 
     * @param autoFlush
     *            <code>true</code> の場合 <code>println()</code> メソッドでは出力バッファをフラッシュする
     */
    public DebugPrintStream(boolean autoFlush) {
        this(System.out, autoFlush);
    }

    /**
     * 指定された出力ストリームを出力先として初期化します。
     * 
     * @param out
     *            出力ストリーム
     */
    public DebugPrintStream(OutputStream out) {
        this(out, false);
    }

    /**
     * 指定された出力ストリームを出力先として初期化します。
     * 
     * @param out
     *            出力ストリーム
     * @param autoFlush
     *            <code>true</code> の場合 <code>println()</code> メソッドでは出力バッファをフラッシュする
     */
    public DebugPrintStream(OutputStream out, boolean autoFlush) {
        super(out, autoFlush);
    }

    /*
     * Printer
     */

    /**
     * 登録済みの出力プリンタを返却します。
     * 
     * @return 出力プリンタ
     */
    public Printer[] getPrinters() {
        synchronized (printers) {
            return (Printer[]) printers.toArray(new Printer[0]);
        }
    }

    /**
     * 出力プリンタを追加します。
     * 
     * @param printer
     *            出力プリンタ
     */
    public void addPrinter(Printer printer) {
        synchronized (printers) {
            this.printers.add(printer);
        }
    }

    /**
     * 出力プリンタを追加します。
     * 
     * @param index
     *            登録するインデックス位置
     * @param printer
     *            出力プリンタ
     * @throws IndexOutOfBoundsException
     *             インデックス位置が範囲外の場合
     */
    public void addPrinter(int index, Printer printer) {
        synchronized (printers) {
            this.printers.add(index, printer);
        }
    }

    /**
     * 出力プリンタを削除します。
     * 
     * @param index
     *            登録するインデックス位置
     * @return 削除が実際に行われた場合 <code>true</code> を返す
     */
    public boolean removePrinter(int index) {
        synchronized (printers) {
            try {
                return (null != this.printers.remove(index));
            } catch (IndexOutOfBoundsException e) {
                return false;
            }
        }
    }

    /**
     * 出力プリンタを削除します。
     * 
     * @param printer
     *            出力プリンタ
     * @return 削除が実際に行われた場合 <code>true</code> を返す
     */
    public boolean removePrinter(Printer printer) {
        synchronized (printers) {
            return this.printers.remove(printer);
        }
    }

    /*
     * print
     */

    /**
     * 登録済みの出力プリンタを使用して出力ストリームへ出力します。
     * 
     * @param o
     *            出力値
     */
    public void print(Object o) {
        try {
            synchronized (printers) {
                Printer printer = null;
                for (Iterator i = printers.iterator(); i.hasNext();) {
                    Printer p = (Printer) i.next();
                    if (p.isAssignable(o)) {
                        printer = p;
                        break;
                    }
                }
                if (printer != null) {
                    synchronized (this) {
                        printer.print(this, o);
                    }
                } else {
                    super.print(o);
                }
            }
        } catch (Exception e) {
            // no op
        }
    }

    /**
     * 出力プリンタを使用して出力ストリームへ出力します。 <br>
     * <code>Printer.isAssignable(Object)</code> が <code>true</code> の場合のみ出力されます。
     * 
     * @param o
     *            出力値
     * @param printer
     *            出力プリンタ
     */
    public void print(Object o, Printer printer) {
        if (printer.isAssignable(o)) {
            synchronized (this) {
                printer.print(this, o);
            }
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, boolean value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, char value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, double value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, float value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, int value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, long value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, Object value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルと名前=値を付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value
     *            出力値
     */
    public void print(Object title, Object name, Object value) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(name);
            this.print('=');
            this.print(value);
        }
    }

    /**
     * タイトルと名前=値１、値２を付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value1
     *            出力値１
     * @param value2
     *            出力値２
     */
    public void print(Object title, Object name, Object value1, Object value2) {
        synchronized (this) {
            this.print(title);
            this.print(" : ");
            this.print(name);
            this.print('=');
            this.print(value1);
            this.print(", ");
            this.print(value2);
        }
    }

    /*
     * println
     */

    /**
     * 出力プリンタを使用して出力ストリームへ出力して改行します。 <br>
     * <code>Printer.isAssignable(Object)</code> が <code>true</code> の場合のみ出力されます。
     * 
     * @param o
     *            出力値
     * @param printer
     *            出力プリンタ
     */
    public void println(Object o, Printer printer) {
        synchronized (this) {
            if (printer.isAssignable(o)) {
                printer.print(this, o);
                this.println();
            }
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, boolean value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, char value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, double value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, float value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, int value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, long value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, Object value) {
        synchronized (this) {
            this.print(title, value);
            this.println();
        }
    }

    /**
     * タイトルと名前=値を付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value
     *            出力値
     */
    public void println(Object title, Object name, Object value) {
        synchronized (this) {
            this.print(title, name, value);
            this.println();
        }
    }

    /**
     * タイトルと名前=値１、値２を付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value1
     *            出力値１
     * @param value2
     *            出力値２
     */
    public void println(Object title, Object name, Object value1, Object value2) {
        synchronized (this) {
            this.print(title, name, value1, value2);
            this.println();
        }
    }

    /*
     * 線
     */

    /**
     * 破線を出力します。
     * 
     * @param len
     *            破線数
     */
    public void printLine(int len) {
        synchronized (this) {
            char[] c = new char[len];
            for (int i = 0; i < len; i++) {
                c[i] = '-';
            }
            this.print(c);
        }
    }

    /**
     * 破線を出力して１行分改行します。
     * 
     * @param len
     *            破線数
     */
    public void printLineln(int len) {
        synchronized (this) {
            this.printLine(len);
            this.println();
        }
    }

    /**
     * タイトルを指定して破線を出力します。 "------ タイトル ------"
     * 
     * @param title
     *            タイトル
     * @param len
     *            破線数
     */
    public void printLineln(Object title, int len) {
        synchronized (this) {
            this.println();
            this.printLine(len);
            this.printsp(2);
            this.print(title);
            this.printsp(2);
            this.printLine(len);
            this.println();
        }
    }

    /**
     * 破線前後の改行数を指定して破線を出力します。
     * 
     * @param len
     *            破線数
     * @param beginLine
     *            開始改行
     * @param endLine
     *            終了改行
     */
    public void printLineln(int len, int beginLine, int endLine) {
        synchronized (this) {
            this.printlf(beginLine);
            this.printLine(len);
            this.printlf(endLine);
        }
    }

    /**
     * 二重破線を出力します。
     * 
     * @param len
     *            破線数
     */
    public void printDLine(int len) {
        synchronized (this) {
            char[] c = new char[len];
            for (int i = 0; i < len; i++) {
                c[i] = '=';
            }
            this.print(c);
        }
    }

    /**
     * 二重破線を出力して前後１行分改行します。
     * 
     * @param len
     *            破線数
     */
    public void printDLineln(int len) {
        this.printDLineln(len, 1, 1);
    }

    /**
     * タイトルを指定して二重破線を出力します。 "====== タイトル ======"
     * 
     * @param title
     *            タイトル
     * @param len
     *            破線数
     */
    public void printDLineln(Object title, int len) {
        synchronized (this) {
            this.println();
            this.printDLine(len);
            this.printsp(2);
            this.print(title);
            this.printsp(2);
            this.printDLine(len);
            this.println();
        }
    }

    /**
     * 破線前後の改行数を指定して二重破線を出力します。
     * 
     * @param len
     *            破線数
     * @param beginLine
     *            開始改行
     * @param endLine
     *            終了改行
     */
    public void printDLineln(int len, int beginLine, int endLine) {
        synchronized (this) {
            this.printlf(beginLine);
            this.printDLine(len);
            this.printlf(endLine);
        }
    }

    /*
     * コレクション
     */
    
    
    /**
     * 要素のクラス名を含めコレクションを出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;[ &lt;ClassName&gt;item1, &lt;ClassName&gt;item2, … ]
     * </pre>
     * 
     * @param c
     *            出力値
     */
    public void printTypeColl(Collection c) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
            } else if (c.size() == 0) {
                this.printClass(c);
                this.print("[]");
            } else {
                this.printClass(c);
                super.print('[');
                Iterator i = c.iterator();
                boolean hasNext = i.hasNext();
                while (hasNext) {
                    Object o = i.next();
                    this.printClass(o);
                    this.print((o == c) ? "(this Collection)" : o);
                    hasNext = i.hasNext();
                    if (hasNext) {
                        super.print(", ");
                    }
                }
                super.print(']');
            }
        }
    }
    

    /**
     * 要素のクラス名を含めマップを出力します。
     * 
     * <pre>
     *  &lt;ClassName&gt;{ &lt;ClassName&gt;key1:&lt;ClassName&gt;value1, &lt;ClassName&gt;key2:&lt;ClassName&gt;value2, … }
     * </pre>
     * 
     * @param m
     *            出力値
     */
    public void printTypeMap(Map m) {
        synchronized (this) {
            if (m == null) {
                this.printNull();
            } else if (m.size() == 0) {
                this.printClass(m);
                this.print("{}");
            } else {
                this.printClass(m);
                this.print('{');
                boolean st = true;
                for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    if (st) {
                        st = false;
                    } else {
                        this.print(", ");
                    }
                    this.printClass(key);
                    this.print((key == m) ? "(this Map)" : key);
                    this.print(':');
                    this.printClass(value);
                    this.print((value == m) ? "(this Map)" : value);
                }
                this.print('}');
            }
        }
    }

    /**
     * 多次元配列を表形式で出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[
     *      item1
     *      &lt;ArrayClassType(length)&gt;[item2-1, item2-2, item2-3]
     *      item3
     *      …
     *     ]
     * </pre>
     * 
     * @param a
     *            出力値
     */
    public void printArrayTbl(Object a) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
                super.println();
            } else if (a.getClass().isArray()) {

                this.printClass(a.getClass());
                super.println('[');

                int size = Array.getLength(a);
                for (int i = 0; i < size; i++) {
                    this.printsp(2);
                    this.printArray(Array.get(a, i));
                    super.println();
                }

                super.println(']');
            } else {
                super.println(a);
            }
        }
    }

    /**
     * 多次元配列を表形式で出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[
     *       item1, item2, item3
     *       item4, &lt;ArrayClassType(length)&gt;[item5-1, item5-2, item5-3], item6
     *       item7, …
     *     ]
     * </pre>
     * 
     * @param a
     *            出力値
     * @param column
     *            １列分の要素数
     */
    public void printArrayTbl(Object a, int column) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
                super.println();

            } else if (a.getClass().isArray()) {
                int size = Array.getLength(a);

                if (size == 0 || column <= 0) {

                    this.printClass(a);
                    this.println("[]");

                } else {

                    this.printClass(a);
                    super.println('[');

                    int row = (size % column == 0) ? (size / column) : (size / column + 1);
                    for (int inx = 0, i = 0; (i < row && inx < size); i++) {
                        this.printsp(2);
                        for (int j = 0; (j < column && inx < size); j++) {
                            this.printArray(Array.get(a, i));
                            if ((j + 1) < column && (inx + 1) < size) {
                                this.print(", ");
                            }
                            inx++;
                        }
                        this.println();
                    }

                    super.println(']');
                }
            } else {
                super.println(a);
            }
        }

    }

    /**
     * コレクションを表形式で出力します。
     * 
     * <pre>
     *    &lt;ClassName&gt;[
     *      item1
     *      item2
     *      item3
     *      …
     *    ]
     * </pre>
     * 
     * @param c
     *            出力値
     */
    public void printCollTbl(Collection c) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
                super.println();
            } else if (c.size() == 0) {
                this.printClass(c);
                this.println("[]");
                return;
            } else {
                this.printClass(c);
                this.println('[');
                Iterator i = c.iterator();
                while (i.hasNext()) {
                    this.printsp(2);
                    Object o = i.next();
                    this.println((o == c) ? "(this Collection)" : o);
                }
                this.println(']');
            }
        }
    }

    /**
     * コレクションを表形式で出力します。
     * 
     * <pre>     
     *     &lt;ClassName&gt;[
     *       item1, item2, item3
     *       item4, item5, item6
     *       item7, …
     *     ]
     * </pre>
     * 
     * @param c
     *            出力値
     * @param column
     *            １列分の要素数
     */
    public void printCollTbl(Collection c, int column) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
                super.println();
            } else if (c.size() == 0 || column <= 0) {
                this.printClass(c);
                this.println("[]");
                return;
            } else {
                Iterator i = c.iterator();
                boolean hasNext = i.hasNext();
                this.printClass(c);
                this.println('[');
                while (hasNext) {
                    this.printsp(2);
                    int icolumn = 0;
                    while (hasNext && icolumn < column) {
                        Object o = i.next();
                        this.print((o == c) ? "(this Collection)" : o);
                        hasNext = i.hasNext();
                        if (hasNext) {
                            super.print(", ");
                        }
                        icolumn++;
                    }
                    this.println();
                }
                this.println(']');
            }
        }
    }

    /**
     * マップを表形式で出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;{
     *       key1:value1
     *       key2:value2
     *       key3:value3
     *       …
     *     }
     * </pre>
     * 
     * @param m
     *            出力値
     */
    public void printMapTbl(Map m) {
        synchronized (this) {
            if (m == null) {
                this.printNull();
                super.println();
            } else if (m.size() == 0) {
                this.printClass(m);
                this.println("{}");
            } else {
                this.printClass(m);
                this.println('{');
                for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    this.printsp(2);
                    this.print((key == m) ? "(this Map)" : key);
                    this.print(':');
                    this.println((value == m) ? "(this Map)" : value);
                }
                this.println('}');
            }
        }
    }

    /*
     * JavaBean
     */

    /**
     * <code>JavaBean</code> の保有するプロパティを全て表形式で出力します。
     * 
     * <pre>        
     *     &lt;ClassName&gt;{
     *       property1=value1
     *       property2=value2
     *       property3=value3
     *       …
     *     }
     * </pre>
     * 
     * @param bean
     *            出力する <code>JavaBean</code>
     */
    public void printBeanTbl(Object bean) {
        synchronized (this) {
            //プロパティをMapに格納して全取得
            Map props;
            try {
                props = BeanUtilities.getProperties(bean);
            } catch (Exception e) {
                this.println();
                return;
            }
            //出力開始
            //クラス名を出力する
            this.print(bean.getClass().getName());
            //全プロパティを出力する
            this.println('{');
            for (Iterator i = props.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                if (!"class".equals(e.getKey())) {
                    this.printsp(2);
                    this.print((String) e.getKey());
                    this.print('=');
                    this.print(e.getValue());
                    this.println();
                }
            }
            this.println('}');
        }

    }

    /**
     * 全ての <code>JavaBean</code> の保有するプロパティを全て表形式で出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[length]
     *     [
     *       [0]&lt;ClassName&gt;{
     *         property1=value1
     *         property2=value2
     *         property3=value3
     *         …
     *       }
     *       [1]&lt;ClassName&gt;{
     *         property1=value1
     *         property2=value2
     *         property3=value3
     *         …
     *       }
     *       …
     *     ]
     * </pre>
     * 
     * @param beans
     *            出力する <code>JavaBean</code> の配列
     */
    public void printBeansTbl(Object[] beans) {
        synchronized (this) {
            //全Beanを出力
            this.printClass(beans);
            this.println();
            this.println('[');
            for (int i = 0; i < beans.length; i++) {
                this.printsp(2);
                this.print('[' + i + ']');
                this.printBeanTbl(beans[i]);
            }

            this.println(']');
        }
    }

    /*
     * 時刻型
     */

    /**
     * 日付型を出力します。
     * 
     * @param date
     *            出力値
     */
    public void printDate(Date date) {
        this.print(formatDate(date));
    }

    /**
     * 現在時刻を出力します。
     */
    public void printCurrentTime() {
        this.printDate(new java.util.Date(System.currentTimeMillis()));
    }

    /*
     * 比較
     */

    /**
     * <code>JavaBean</code> の保有する同一名のプロパティを全て比較しその情報を出力する
     * 
     * @param from
     *            比較元
     * @param to
     *            比較先
     */
    public void printCompBean(Object from, Object to) {
        synchronized (this) {
            //プロパティをMapに格納して全取得
            Map fromProp;
            Map toProp;
            try {
                fromProp = BeanUtilities.getProperties(from);
                toProp = BeanUtilities.getProperties(to);
            } catch (Exception e) {
                this.println();
                return;
            }
            //出力開始
            this.printCompMap(fromProp, toProp);
        }

    }

    /**
     * <code>JavaBean</code> の保有する同一名のプロパティを全て比較しその情報を出力する
     * 
     * @param from
     *            比較元
     * @param to
     *            比較先
     */
    public void printCompMap(Map from, Map to) {
        synchronized (this) {
            //出力開始
            for (Iterator i = from.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                Object fromValue = e.getValue();
                Object toValue = to.get(e.getKey());

                //要素を全て厳密に比較する
                if (deepEquals(fromValue, toValue)) {
                    this.println("TRUE", e.getKey(), fromValue, toValue);
                } else {
                    this.println("FALSE", e.getKey(), fromValue, toValue);
                }
            }
        }
    }

    /**
     * 型情報を付けて出力します。
     * 
     * @param value
     *            出力値
     */
    public void printType(Object value) {
        synchronized (this) {
            this.printClass(value);
            this.print(':');
            this.print(value);
        }
    }

    /**
     * 型情報を付けて出力し改行します。
     * 
     * @param value
     *            出力値
     */
    public void printTypeln(Object value) {
        synchronized (this) {
            this.printClass(value);
            this.print(':');
            this.println(value);
        }
    }

    /*
     * Helper
     */

    /**
     * 日付型の書式で日付文字列に変換します。
     * <p>
     * 変換形式：
     * <ul>
     * <li>java.sql.Timestamp = yyyy-mm-dd hh:mm:ss.fffffffff
     * <li>java.sql.Time = hh:mm:ss
     * <li>java.util.Date = yyyy-mm-dd hh:mm:ss.sss
     * <ul>
     * </p>
     * 
     * @param date
     *            日付
     * @return 日付文字列
     */
    protected static Object formatDate(Object date) {
        if (date instanceof Timestamp) {
            String s = formatDate((java.util.Date) date, "yyyy-MM-dd HH:mm:ss");
            return s + '.' + ((Timestamp) date).getNanos();
        } else if (date instanceof Time) {
            return formatDate((java.util.Date) date, "HH:mm:ss");
        } else if (date instanceof Date) {
            return formatDate((java.util.Date) date, "yyyy-MM-dd HH:mm:ss.SSS");
        } else {
            return date;
        }
    }

    /**
     * 日付型をフォーマット変換します。
     * 
     * @param date
     *            日付
     * @param pattern
     *            書式
     * @return 日付文字列
     */
    protected static String formatDate(java.util.Date date, String pattern) {
        SimpleDateFormat formatter = new SimpleDateFormat(pattern);
        return formatter.format(date);
    }

    /**
     * 厳密な同一性比較を行う。 <br>
     * <code>null</code> の比較と 配列・多次元配列の内部要素を含む同値性比較を行い、以外は通常の比較と同一。
     * 
     * @param from
     *            比較元
     * @param to
     *            比較先
     * @return 同一の場合True
     */
    protected static boolean deepEquals(Object from, Object to) {
        //どちらかがNullの場合
        if (from == null || to == null) {
            //双方Nullの場合True
            return (from == to);
        }
        //双方配列の場合内部要素を含め比較する
        if (from.getClass().isArray() && to.getClass().isArray()) {
            //配列のコンポーネント型が違う場合False
            if (!from.getClass().getComponentType().equals(from.getClass().getComponentType())) {
                return false;
            }
            //配列長が違う場合False
            int size = Array.getLength(from);
            if (size != Array.getLength(to)) {
                return false;
            }
            //内部要素を全比較
            for (int i = 0; i < size; i++) {
                Object o1 = Array.get(from, i);
                Object o2 = Array.get(to, i);
                //片方がNullの場合False
                if (o1 == null || o2 == null) {
                    if (o1 != o2) {
                        return false;
                    }
                } else {
                    //再起的に比較
                    if (!deepEquals(o1, o2)) {
                        return false;
                    }
                }
            }
            //全要素の比較結果同一
            return true;
        }
        //以外通常比較
        return from.equals(to);

    }
}
