/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the GNU Lesser General
 * Public License as published by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this library; if not, write to
 * the Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.util.cel;

import shohaku.core.collections.Parameters;
import shohaku.core.lang.Boxing;
import shohaku.core.lang.Eval;
import shohaku.core.util.codec.UnicodeEscapesCoder;

/**
 * プリミティブ型のラッパー型を生成する式を提供します。
 */
public class PrimitiveExpression implements Expression {

    /* この式のリテラル文字を定義します。 */
    private static final Literal[] literals = new Literal[] { new Literal('\u0000', '\u0000'), new Literal('\'', '\'') };

    private static final String NUM_CHARS = "-+xX0123456789abcdefABCDEF.BSILFD";

    /**
     * リテラル文字を持たない特殊リテラル '\u0000', '\u0000' と文字リテラル '\'', '\'' を返します。
     * 
     * @return リテラル文字
     */
    public Literal[] getLiteral() {
        return literals;
    }

    /**
     * 式を実行して値を返します。
     * 
     * @param binder
     *            処理基のバインダー
     * @param literal
     *            リテラル文字
     * @param expression
     *            式
     * @param begin
     *            開始インデックス
     * @param end
     *            終了インデックス
     * @param values
     *            引数値の一覧
     * @return 式の結果
     */
    public Object execute(CELBinder binder, Literal literal, String expression, int begin, int end, Parameters values) {
        String exp = expression.substring(begin, end).trim();
        if (literals[0].begin == literal.begin) {
            return executeNoLiteral(exp);
        } else {
            return executeCharacter(exp);
        }
    }

    private Object executeNoLiteral(String exp) {
        //Null
        if ("null".equals(exp)) {
            return null;
        }
        //Boolean
        if (Eval.isOrEquals(exp, new Object[] { "true", "false" })) {
            return Boolean.valueOf(exp);
        }
        //Number
        char numLiteral = exp.charAt(exp.length() - 1);
        String num = exp.substring(0, exp.length() - 1);
        switch (numLiteral) {
        case 'B':
            return Byte.decode(num);
        case 'S':
            return Short.decode(num);
        case 'I':
            return Integer.decode(num);
        case 'L':
            return Long.decode(num);
        case 'F':
            return Float.valueOf(num);
        case 'D':
            return Double.valueOf(num);
        default:
            if (Eval.isContains(num, '.')) {
                return Double.valueOf(exp);
            } else {
                return Integer.decode(exp);
            }
        }
    }

    private Object executeCharacter(String exp) {
        String schar = exp.replaceAll("\\\\'", "'");
        if (schar.length() > 1) {
            schar = UnicodeEscapesCoder.decode(schar);
        }
        if (schar.length() != 1) {
            //err
        }
        return Boxing.box(schar.charAt(0));
    }

    /**
     * 式の終端のインデックスを検索して返します。 発見出来ない場合は -1 を返します。
     * 
     * @param binder
     *            処理基のバインダー
     * @param expression
     *            式
     * @param literal
     *            リテラル文字
     * @param begin
     *            開始インデックス
     * @return 式の終端のインデックス
     */
    public int findNextIndex(CELBinder binder, Literal literal, String expression, int begin) {
        if (literals[0].equals(literal)) {
            return endLiteralIndexOfByNoLiteral(expression, begin);
        } else {
            return endLiteralIndexOfyByCharacter(expression, begin);
        }
    }

    private int endLiteralIndexOfByNoLiteral(String expression, int begin) {
        //Null
        if (Eval.isStartsWith(expression, "null", begin)) {
            return begin + "null".length();
        }
        //Boolean
        if (Eval.isStartsWith(expression, "true", begin)) {
            return begin + "true".length();
        }
        if (Eval.isStartsWith(expression, "false", begin)) {
            return begin + "false".length();
        }
        //Number
        int end = begin;
        while (end < expression.length()) {
            if (!Eval.isContains(NUM_CHARS, expression.charAt(end))) {
                break;
            }
            end++;
        }
        if(end == begin){
            return -1;
        }
        return end;
    }

    private int endLiteralIndexOfyByCharacter(String expression, int begin) {
        int end = (begin + 1);//'\''.length() == 1
        while (end < expression.length()) {
            end = expression.indexOf('\'', end);
            if (-1 == end) {
                return -1;
            }
            if (0 == end || (0 < end && '\\' != expression.charAt(end - 1))) {
                break;
            }
            end++;
        }
        return (end + 1);//end to next
    }

}