/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util;

import java.sql.Timestamp;
import java.util.Calendar;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * オブジェクトとその状態に関する記述文を生成するユーティリティメソッドを提供します。
 */
class ToStringHelper {

    /* 通常の出力。 */
    static final int NORMAL = 0;

    /* 内部情報の配列とクラス名の出力。 */
    static final int CLASSES = 1;

    /* 「深層表現」の配列とクラス名の出力。 */
    static final int DEEP_CLASSES = 2;

    /*
     * toString
     */

    /**
     * 指定のスタイルでオブジェクトのより詳細な文字列表現を構築します。 <br>
     * 指定されたオブジェクトが配列の場合は配列の内容の文字列表現を返します。 <br>
     * MapまたはCollectionの要素が配列の場合にその要素の内容の文字列表現を含んだ文字列を返します。 <br>
     * 以外の場合は <code>String.valueOf(o)</code> により文字列に変換されます。 <br>
     * <br>
     * o が配列の場合は、角括弧 (「[]」) で囲まれた配列要素のリストで構成されます。 <br>
     * 隣接する要素は、文字「, 」(コンマおよび空白文字) で区切られます。 <br>
     * 要素はその型に対応する <code>String.valueOf(#type# o)</code> により文字列に変換されます。 <br>
     * <br>
     * o が <code> null </code> の場合、 <code> 「null」 </code> を返します。
     * 
     * @param o
     *            文字列表現を返す配列
     * @param style
     *            書式スタイル
     * @return o の文字列表現
     */
    static String toString(Object o, int style) {
        if (null == o) {
            return "null";
        }
        Class c = o.getClass();
        StringBuffer buf = new StringBuffer();
        appendClass(o, buf, 0, style);
        if (c.isArray()) {
            Class type = c.getComponentType();
            if (type == Character.TYPE) {
                buf.append(toString((char[]) o));
            } else if (type == Boolean.TYPE) {
                buf.append(toString((boolean[]) o));
            } else if (type == Byte.TYPE) {
                buf.append(toString((byte[]) o));
            } else if (type == Short.TYPE) {
                buf.append(toString((short[]) o));
            } else if (type == Integer.TYPE) {
                buf.append(toString((int[]) o));
            } else if (type == Long.TYPE) {
                buf.append(toString((long[]) o));
            } else if (type == Float.TYPE) {
                buf.append(toString((float[]) o));
            } else if (type == Double.TYPE) {
                buf.append(toString((double[]) o));
            } else {
                buf.append(toString((Object[]) o, style));
            }
        } else {
            if (o instanceof Map) {
                buf.append(toString((Map) o, style));
            } else if (o instanceof Collection) {
                buf.append(toString((Collection) o, style));
            } else if (o instanceof Timestamp) {
                buf.append(String.valueOf(o));
            } else if (o instanceof java.sql.Time) {
                buf.append(String.valueOf(o));
            } else if (o instanceof java.util.Date) {
                buf.append(DateUtils.format(o));
            } else if (o instanceof Calendar) {
                buf.append(DateUtils.format(o));
            } else {
                buf.append(String.valueOf(o));
            }
        }
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(boolean[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(char[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(byte[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(short[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(int[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(long[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(float[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(double[] a) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        buf.append('[');
        buf.append(a[0]);

        for (int i = 1; i < a.length; i++) {
            buf.append(", ");
            buf.append(a[i]);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定された配列の内容の文字列表現を返します。 */
    static String toString(Object[] a, int style) {
        if (a == null) {
            return "null";
        }
        if (a.length == 0) {
            return "[]";
        }

        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < a.length; i++) {
            if (i == 0) {
                buf.append('[');
            } else {
                buf.append(", ");
            }
            appendObject(a[i], buf, style);
        }

        buf.append(']');
        return buf.toString();
    }

    /* 指定されたマップの内容の文字列表現を返します。 */
    static String toString(Map m, int style) {
        if (m == null) {
            return "null";
        }
        StringBuffer buf = new StringBuffer();
        if (m.size() == 0) {
            buf.append("{}");
        } else {
            buf.append('{');
            boolean st = true;
            for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                Object key = e.getKey();
                Object value = e.getValue();
                if (st) {
                    st = false;
                } else {
                    buf.append(", ");
                }
                if (key != null && key.getClass().isArray()) {
                    buf.append(toString(key, style));
                } else {
                    appendObject(key, buf, style);
                }
                buf.append(':');
                if (value != null && value.getClass().isArray()) {
                    buf.append(toString(value, style));
                } else {
                    appendObject(value, buf, style);
                }
            }
            buf.append('}');
        }
        return buf.toString();
    }

    /* 指定されたコレクションの内容の文字列表現を返します。 */
    static String toString(Collection c, int style) {
        if (c == null) {
            return "null";
        }
        StringBuffer buf = new StringBuffer();
        if (c.size() == 0) {
            buf.append("[]");
        } else {
            buf.append('[');
            Iterator i = c.iterator();
            boolean hasNext = i.hasNext();
            while (hasNext) {
                Object e = i.next();
                if (e != null && e.getClass().isArray()) {
                    toString(e, style);
                } else {
                    appendObject(e, buf, style);
                }
                hasNext = i.hasNext();
                if (hasNext) {
                    buf.append(", ");
                }
            }
            buf.append(']');
        }
        return buf.toString();
    }

    /*
     * deepToString
     */

    /**
     * 指定のスタイルでオブジェクトの「深層内容」を含むより詳細な文字列表現を構築します。 <br>
     * 指定されたオブジェクトが配列の場合は「深層内容」の文字列表現を返します。 <br>
     * MapまたはCollectionの要素が配列の場合にその「深層内容」の文字列表現を含んだ文字列を返します。 <br>
     * 以外の場合は <code>String.valueOf(o)</code> により文字列に変換されます。 <br>
     * このメソッドは、多次元配列の文字列への変換用に設計されています。 <br>
     * <br>
     * o が配列の場合は、角括弧 (「[]」) で囲まれた配列要素のリストで構成されます。 <br>
     * 隣接する要素は、文字「, 」(コンマおよび空白文字) で区切られます。 <br>
     * 要素は、それ自体が配列でないかぎり、その型に対応する <code>String.valueOf(#type# o)</code> により文字列に変換されます。 <br>
     * <br>
     * 要素 e が参照型の配列である場合、このメソッドを再帰的に呼び出すことで文字列に変換されます。 <br>
     * <br>
     * 無限の再帰を避けるため、指定された配列に自身が要素として含まれるか、1 つまたは複数の配列レベルを介した自身への間接参照が含まれる場合、 <br>
     * 自己参照は文字列「[...]」に変換されます。 たとえば、自身への参照だけを含む配列は、「[[...]]」としてレンダリングされます。 <br>
     * <br>
     * o が <code> null </code> の場合、 <code> 「null」 </code> を返します。
     * 
     * @param o
     *            文字列表現を返す配列
     * @param style
     *            書式スタイル
     * @return o の文字列表現
     */
    static String deepToString(Object o, int style) {
        if (null == o) {
            return "null";
        }
        Class c = o.getClass();
        if (c.isArray()) {
            Class type = c.getComponentType();
            if (type.isPrimitive()) {
                return toString(o, style);
            } else {
                return deepToString((Object[]) o, style);
            }
        } else {
            if (o instanceof Map) {
                return deepToString((Map) o, style);
            } else if (o instanceof Collection) {
                return deepToString((Collection) o, style);
            } else {
                return String.valueOf(o);
            }
        }
    }

    /* 指定された配列の「深層内容」の文字列表現を返します。 */
    static String deepToString(Object[] a, int style) {
        if (a == null) {
            return "null";
        }

        int bufLen = 20 * a.length;
        if (a.length != 0 && bufLen <= 0) {
            bufLen = Integer.MAX_VALUE;
        }
        StringBuffer buf = new StringBuffer(bufLen);
        deepToString(a, buf, new HashSet(), 0, style);
        return buf.toString();
    }

    /* deepToString(Object[], int) の再起メソッドです。 */
    private static void deepToString(Object[] a, StringBuffer buf, Set dejaVu, int row, int style) {
        if (a == null) {
            buf.append("null");
            return;
        }
        dejaVu.add(a);
        appendClass(a, buf, row, style);
        buf.append('[');
        for (int i = 0; i < a.length; i++) {
            if (i != 0) {
                buf.append(", ");
            }

            Object e = a[i];
            if (e == null) {
                buf.append("null");
            } else if (dejaVu.contains(e)) {
                buf.append("[...]");
            } else {
                Class eClass = e.getClass();
                if (eClass.isArray()) {
                    Class type = eClass.getComponentType();
                    if (type == Character.TYPE) {
                        buf.append(toString((char[]) e));
                    } else if (type == Boolean.TYPE) {
                        buf.append(toString((boolean[]) e));
                    } else if (type == Byte.TYPE) {
                        buf.append(toString((byte[]) e));
                    } else if (type == Short.TYPE) {
                        buf.append(toString((short[]) e));
                    } else if (type == Integer.TYPE) {
                        buf.append(toString((int[]) e));
                    } else if (type == Long.TYPE) {
                        buf.append(toString((long[]) e));
                    } else if (type == Float.TYPE) {
                        buf.append(toString((float[]) e));
                    } else if (type == Double.TYPE) {
                        buf.append(toString((double[]) e));
                    } else {
                        deepToString((Object[]) e, buf, dejaVu, (row + 1), style);
                    }
                } else {
                    if (e instanceof Map) {
                        deepToString((Map) e, buf, dejaVu, style);
                    } else if (e instanceof Collection) {
                        deepToString((Collection) e, buf, dejaVu, style);
                    } else {
                        appendObject(e, buf, style);
                    }
                }
            }
        }
        buf.append("]");
        dejaVu.remove(a);
    }

    /* 指定されたマップの「深層内容」の文字列表現を返します。 */
    static String deepToString(Map m, int style) {
        if (m == null) {
            return "null";
        }

        int bufLen = 20 * m.size();
        if (m.size() != 0 && bufLen <= 0) {
            bufLen = Integer.MAX_VALUE;
        }
        StringBuffer buf = new StringBuffer(bufLen);
        deepToString(m, buf, new HashSet(), style);
        return buf.toString();
    }

    /* deepToString(Map, int) の再起メソッドです。 */
    private static void deepToString(Map m, StringBuffer buf, Set dejaVu, int style) {
        if (m == null) {
            buf.append("null");
            return;
        }
        dejaVu.add(m);
        appendClass(m, buf, style);
        if (m.size() == 0) {
            buf.append("{}");
        } else {
            buf.append('{');
            boolean st = true;
            for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                Object key = e.getKey();
                Object value = e.getValue();
                if (st) {
                    st = false;
                } else {
                    buf.append(", ");
                }
                append(m, buf, key, dejaVu, style);
                buf.append(':');
                append(m, buf, value, dejaVu, style);
            }
            buf.append('}');
        }
    }

    /* deepToString(Map, int) の値を追加します。 */
    private static void append(Map m, StringBuffer buf, Object o, Set dejaVu, int style) {
        if (o == null) {
            buf.append("null");
            return;
        }
        if (dejaVu.contains(o)) {
            buf.append("[...]");
        } else {
            if (o == m) {
                buf.append("(this Map)");
            } else if (o.getClass().isArray()) {
                deepToString((Object[]) o, buf, dejaVu, 0, style);
            } else if (o instanceof Map) {
                deepToString((Map) o, buf, dejaVu, style);
            } else if (o instanceof Collection) {
                deepToString((Collection) o, buf, dejaVu, style);
            } else {
                appendObject(o, buf, style);
            }
        }
    }

    /* 指定されたコレクションの「深層内容」の文字列表現を返します。 */
    static String deepToString(Collection c, int style) {
        if (c == null) {
            return "null";
        }

        int bufLen = 20 * c.size();
        if (c.size() != 0 && bufLen <= 0) {
            bufLen = Integer.MAX_VALUE;
        }
        StringBuffer buf = new StringBuffer(bufLen);
        deepToString(c, buf, new HashSet(), style);
        return buf.toString();
    }

    /* deepToString(Collection, int) の再起メソッドです。 */
    private static void deepToString(Collection c, StringBuffer buf, Set dejaVu, int style) {
        if (c == null) {
            buf.append("null");
            return;
        }
        dejaVu.add(c);
        appendClass(c, buf, style);
        if (c.size() == 0) {
            buf.append("[]");
        } else {
            buf.append('[');
            Iterator i = c.iterator();
            boolean hasNext = i.hasNext();
            while (hasNext) {
                Object e = i.next();
                append(c, buf, e, dejaVu, style);
                hasNext = i.hasNext();
                if (hasNext) {
                    buf.append(", ");
                }
            }
            buf.append(']');
        }
    }

    /* deepToString(Collection, int) の値を追加します。 */
    private static void append(Collection c, StringBuffer buf, Object o, Set dejaVu, int style) {
        if (o == null) {
            buf.append("null");
        } else if (o == c) {
            buf.append("(this Collection)");
        } else if (o.getClass().isArray()) {
            deepToString((Object[]) o, buf, dejaVu, 0, style);
        } else if (o instanceof Map) {
            deepToString((Map) o, buf, dejaVu, style);
        } else if (o instanceof Collection) {
            deepToString((Collection) o, buf, dejaVu, style);
        } else {
            appendObject(o, buf, style);
        }
    }

    /* deepToString(...) の値を追加します。 */
    private static void appendObject(Object o, StringBuffer buf, int style) {
        if (o == null) {
            buf.append("null");
            return;
        }
        if (DEEP_CLASSES == style) {
            appendClass(o, buf, style);
        }
        if (o instanceof Timestamp) {
            buf.append(String.valueOf(o));
        } else if (o instanceof java.sql.Time) {
            buf.append(String.valueOf(o));
        } else if (o instanceof java.util.Date) {
            buf.append(DateUtils.format(o));
        } else if (o instanceof Calendar) {
            buf.append(DateUtils.format(o));
        } else {
            buf.append(o);
        }
    }

    /* deepToString(...) の値のクラス情報を追加します。 */
    private static void appendClass(Object o, StringBuffer buf, int style) {
        appendClass(o, buf, -1, style);
    }

    /* deepToString(...) の値のクラス情報を追加します。 */
    private static void appendClass(Object o, StringBuffer buf, int row, int style) {
        if ((CLASSES == style && row == 0) || DEEP_CLASSES == style) {
            buf.append('<');
            buf.append(o.getClass().getName());
            buf.append('>');
        }
    }
}
