/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util;

import java.lang.reflect.Array;
import java.text.ChoiceFormat;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.Format;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import shohaku.core.beans.BeanUtilities;
import shohaku.core.beans.InvocationBeansException;
import shohaku.core.lang.Eval;
import shohaku.core.lang.Seek;

/**
 * 識別子でマッピングされたメッセージ書式化機能を束ねる機能を提供します。
 */
public class MessageFormatBinder {

    /**
     * 識別子とフォーマッタをマッピングし実行する機能を定義します。
     */
    public interface Formater {

        /**
         * 識別子の一覧を返します。
         * 
         * @return 識別子の一覧
         */
        String[] getIds();

        /**
         * フォーマットを実行します。
         * 
         * @param locale
         *            ロケール
         * @param id
         *            フォーマット識別子
         * @param arg
         *            フォーマット引数
         * @param value
         *            処理対象の値
         * @param values
         *            引数値の一覧
         * @return フォーマットされた文字列
         */
        CharSequence format(Locale locale, String id, String arg, Object value, Object[] values);
    }

    /* フォーマットの正規表現パターン。 */
    private static final Pattern formatPattern;
    static {
        StringBuffer sb = new StringBuffer();
        sb.append("\\{");
        sb.append("(\\d+)");
        sb.append("(?:,([_a-z-A-Z]\\w+))?");
        sb.append("(?:,((?>\\\\}|[^}])+))?");
        sb.append("\\}");
        formatPattern = Pattern.compile(sb.toString());
    }

    /* 括弧を削除する正規表現パターン。 */
    private static final Pattern bracePattern = Pattern.compile("\\\\}");

    /* デフォルトで使用するフォーマット。 */
    private static final Formater[] defaultFormaters;
    static {
        List fs = new ArrayList();
        fs.add(new BeansFormater());
        fs.add(new CollectionFormater());
        fs.add(new ListFormater());
        fs.add(new SetFormater());
        fs.add(new MapFormater());
        fs.add(new ArrayFormater());
        fs.add(new NumberFormater());
        fs.add(new DateFormater());
        fs.add(new TimeFormater());
        fs.add(new ChoiceFormater());
        defaultFormaters = (Formater[]) fs.toArray(new Formater[0]);
    }

    /* デフォルトのフォーマッターインスタンス。 */
    private static final MessageFormatBinder defaultBinder = new MessageFormatBinder();

    /* このインスタンスで使用するフォーマットのマッピング情報。 */
    private final Map formaterMap;

    /* このインスタンスで使用するフォーマット。 */
    private final Formater[] formaters;

    /* このインスタンスで使用するロケール。 */
    private final Locale locale;

    /**
     * デフォルトのロケールとフォーマットで初期化します。
     */
    public MessageFormatBinder() {
        this(Locale.getDefault(), defaultFormaters);
    }

    /**
     * ロケールを指定してデフォルトフォーマットで初期化します。
     * 
     * @param locale
     *            ロケール
     */
    public MessageFormatBinder(Locale locale) {
        this(locale, defaultFormaters);
    }

    /**
     * ロケールとフォーマットを指定して初期化します。
     * 
     * @param locale
     *            ロケール
     * @param fs
     *            フォーマット
     */
    public MessageFormatBinder(Locale locale, Formater[] fs) {
        this(locale, fs, toMappingFormaters(fs));
    }

    /* フォーマッタを初期化します。 */
    private MessageFormatBinder(Locale locale, Formater[] fs, Map fsm) {
        this.formaters = fs;
        this.locale = locale;
        this.formaterMap = fsm;
    }

    /**
     * 登録されているフォーマット一覧を返却します．
     * 
     * @return 登録されているフォーマット一覧
     */
    public Formater[] getFormaters() {
        return (Formater[]) formaters.clone();
    }

    /**
     * ロケールを返却します．
     * 
     * @return ロケール
     */
    public Locale getLocale() {
        return locale;
    }

    /**
     * フォーマット変換を実行します。
     * 
     * @param pattern
     *            書式パターン
     * @param args
     *            書式パターンに割り当てる値
     * @return フォーマット変換された文字列
     */
    public String format(String pattern, Object[] args) {
        StringBuffer sb = new StringBuffer();
        Matcher m = formatPattern.matcher(pattern);

        int i = 0;
        while (i < pattern.length()) {
            if (m.find(i)) {
                sb.append(pattern.substring(i, m.start()));

                String[] sa = new String[3];
                for (int j = 0; j < m.groupCount(); j++) {
                    sa[j] = m.group(j + 1);
                }
                formatString(sb, sa[1], replaceBrace(sa[2]), args[getIndex(sa[0])], args);
                i = m.end();
            } else {
                sb.append(pattern.substring(i));
                break;
            }
        }

        return sb.toString();
    }

    /* 文字列を参照インデックスに変換して返します。 */
    private int getIndex(String sindex) {
        return Integer.parseInt(sindex);
    }

    /* エスケープシーケンスされた区切り文字を区切り文字に置換します。 */
    private String replaceBrace(String sbrace) {
        return (sbrace != null) ? bracePattern.matcher(sbrace).replaceAll("}") : null;
    }

    /* 個々の要素のフォーマット変換を実行します。 */
    private void formatString(StringBuffer sb, String id, String arg, Object value, Object[] values) {
        if (value == null) {
            sb.append("null");
            return;
        }
        if (!Eval.isBlank(id)) {
            Formater f = (Formater) formaterMap.get(id);
            sb.append(f.format(locale, id, arg, value, values));
        } else {
            sb.append(value);
        }
    }

    /*
     * static
     */

    /**
     * デフォルトの書式化オブジェクトを返却します．
     * 
     * @return デフォルトの書式化オブジェクト
     */
    public static MessageFormatBinder getDefaultBinder() {
        return defaultBinder;
    }

    /* フォーマットを識別子単位でマッピングして返却します。 */
    private static Map toMappingFormaters(Formater[] fs) {
        Map m = new HashMap((int) (fs.length * 1.5));
        for (int i = 0; i < fs.length; i++) {
            String[] ids = fs[i].getIds();
            for (int j = 0; j < ids.length; j++) {
                m.put(ids[j], fs[i]);
            }
        }
        return Collections.unmodifiableMap(m);
    }

    /*
     * implements default Formater
     */

    /* Java Bean を処理するデフォルトフォーマット。 */
    private static final class BeansFormater implements Formater {
        public String[] getIds() {
            return new String[] { "bean" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            try {
                if (!Eval.isBlank(arg)) {
                    Object ret = BeanUtilities.getNestedProperty(value, arg.trim(), values);
                    return ToStringHelper.toString(ret, ToStringHelper.NORMAL);
                } else {
                    return BeanUtilities.toBeanString(value);
                }
            } catch (InvocationBeansException e) {
                throw new IllegalArgumentException("id:" + id + ", arg:" + arg);
            }
        }
    }

    /* コレクションを処理するデフォルトフォーマット。 */
    private static final class CollectionFormater implements Formater {
        public String[] getIds() {
            return new String[] { "coll" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            if (!(value instanceof Collection)) {
                throw new IllegalArgumentException("no Collection type:" + value);
            }
            if (!Eval.isBlank(arg)) {
                if ("class".equalsIgnoreCase(arg)) {
                    return ToStringHelper.toString(value, ToStringHelper.CLASSES);
                }
                if ("deep".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.NORMAL);
                }
                if ("detail".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.CLASSES);
                }
                if ("size".equalsIgnoreCase(arg)) {
                    return String.valueOf(((Collection) value).size());
                }
                //unsupported
                throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
            } else {
                return ToStringHelper.toString(value, ToStringHelper.NORMAL);
            }
        }
    }

    /* リストを処理するデフォルトフォーマット。 */
    private static final class ListFormater implements Formater {
        public String[] getIds() {
            return new String[] { "list" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            if (!(value instanceof List)) {
                throw new IllegalArgumentException("no List type:" + value);
            }
            if (!Eval.isBlank(arg)) {
                if ("class".equalsIgnoreCase(arg)) {
                    return ToStringHelper.toString(value, ToStringHelper.CLASSES);
                }
                if ("deep".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.NORMAL);
                }
                if ("detail".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.CLASSES);
                }
                if ("size".equalsIgnoreCase(arg)) {
                    return String.valueOf(((List) value).size());
                }
                //index
                try {
                    int index = Integer.parseInt(arg);
                    return ToStringHelper.toString(((List) value).get(index), ToStringHelper.NORMAL);
                } catch (NumberFormatException e) {
                    throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
                }
            } else {
                return ToStringHelper.toString(value, ToStringHelper.NORMAL);
            }
        }
    }

    /* 集合を処理するデフォルトフォーマット。 */
    private static final class SetFormater implements Formater {
        public String[] getIds() {
            return new String[] { "set" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            if (!(value instanceof Set)) {
                throw new IllegalArgumentException("no Set type:" + value);
            }
            if (!Eval.isBlank(arg)) {
                if ("class".equalsIgnoreCase(arg)) {
                    return ToStringHelper.toString(value, ToStringHelper.CLASSES);
                }
                if ("deep".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.NORMAL);
                }
                if ("detail".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.CLASSES);
                }
                if ("size".equalsIgnoreCase(arg)) {
                    return String.valueOf(((Set) value).size());
                }
                //unsupported
                throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
            } else {
                return ToStringHelper.toString(value, ToStringHelper.NORMAL);
            }
        }
    }

    /* マップを処理するデフォルトフォーマット。 */
    private static final class MapFormater implements Formater {
        public String[] getIds() {
            return new String[] { "map" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            if (!(value instanceof Map)) {
                throw new IllegalArgumentException("no Map type:" + value);
            }
            if (!Eval.isBlank(arg)) {
                if ("class".equalsIgnoreCase(arg)) {
                    return ToStringHelper.toString(value, ToStringHelper.CLASSES);
                }
                if ("deep".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.NORMAL);
                }
                if ("detail".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.CLASSES);
                }
                if ("size".equalsIgnoreCase(arg)) {
                    return String.valueOf(((Map) value).size());
                }
                if ("get{".startsWith(arg) && '}' == arg.charAt(arg.length() - 1)) {
                    try {
                        int index = Integer.parseInt(arg.substring(4, arg.length() - 1));
                        Object e = ((Map) value).get(values[index]);
                        return ToStringHelper.toString(e, ToStringHelper.NORMAL);
                    } catch (NumberFormatException e) {
                        throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
                    }
                }
                if ("get[".startsWith(arg) && ']' == arg.charAt(arg.length() - 1)) {
                    try {
                        Object key = arg.substring(4, arg.length() - 1);
                        Object e = ((Map) value).get(key);
                        return ToStringHelper.toString(e, ToStringHelper.NORMAL);
                    } catch (NumberFormatException e) {
                        throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
                    }
                }
                //unsupported
                throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
            } else {
                return ToStringHelper.toString(value, ToStringHelper.NORMAL);
            }
        }
    }

    /* 配列を処理するデフォルトフォーマット。 */
    private static final class ArrayFormater implements Formater {
        public String[] getIds() {
            return new String[] { "array" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            if (!value.getClass().isArray()) {
                throw new IllegalArgumentException("no Array type:" + value);
            }
            if (!Eval.isBlank(arg)) {
                if ("class".equalsIgnoreCase(arg)) {
                    return ToStringHelper.toString(value, ToStringHelper.CLASSES);
                }
                if ("deep".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.NORMAL);
                }
                if ("detail".equalsIgnoreCase(arg)) {
                    return ToStringHelper.deepToString(value, ToStringHelper.CLASSES);
                }
                if ("size".equalsIgnoreCase(arg)) {
                    return String.valueOf(Array.getLength(value));
                }
                //index
                try {
                    int index = Integer.parseInt(arg);
                    return ToStringHelper.toString(Array.get(value, index), ToStringHelper.NORMAL);
                } catch (NumberFormatException e) {
                    throw new IllegalArgumentException("unsupported id:" + id + ", arg:" + arg);
                }
            } else {
                return ToStringHelper.toString(value, ToStringHelper.NORMAL);
            }
        }
    }

    /* 数値を処理するデフォルトフォーマット（java.text.NumberFormat を使用）。 */
    private static final class NumberFormater implements Formater {
        String[] keyword = new String[] { "default", "currency", "percent", "integer" };

        public String[] getIds() {
            return new String[] { "number" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            //get the format
            Format newFormat = null;
            arg = arg.trim().toLowerCase();
            if (Eval.isBlank(arg)) {
                switch (Seek.indexOf(keyword, arg, 0, keyword.length)) {
                case 0: // default;
                    newFormat = NumberFormat.getInstance(locale);
                    break;
                case 1:// currency
                    newFormat = NumberFormat.getCurrencyInstance(locale);
                    break;
                case 2:// percent
                    newFormat = NumberFormat.getPercentInstance(locale);
                    break;
                case 3:// integer
                    newFormat = NumberFormat.getIntegerInstance(locale);
                    break;
                default: // pattern
                    newFormat = new DecimalFormat(arg, new DecimalFormatSymbols(locale));
                    break;
                }
            } else {
                newFormat = NumberFormat.getInstance(locale);
            }
            return newFormat.format(value);
        }

    }

    /* 日付を処理するデフォルトフォーマット（java.text.DateFormat を使用）。 */
    private static final class DateFormater implements Formater {
        String[] keyword = new String[] { "default", "short", "medium", "long", "full" };

        public String[] getIds() {
            return new String[] { "date" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            Date date = null;
            if (value instanceof java.util.Date) {
                date = (Date) value;
            } else if (value instanceof Calendar) {
                date = ((Calendar) value).getTime();
            } else {
                throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
            }
            //get the format
            Format newFormat = null;
            arg = arg.trim().toLowerCase();
            if (Eval.isBlank(arg)) {
                switch (Seek.indexOf(keyword, arg, 0, keyword.length)) {
                case 0: // default
                    newFormat = DateFormat.getDateInstance(DateFormat.DEFAULT, locale);
                    break;
                case 1: // short
                    newFormat = DateFormat.getDateInstance(DateFormat.SHORT, locale);
                    break;
                case 2: // medium
                    newFormat = DateFormat.getDateInstance(DateFormat.DEFAULT, locale);
                    break;
                case 3: // long
                    newFormat = DateFormat.getDateInstance(DateFormat.LONG, locale);
                    break;
                case 4: // full
                    newFormat = DateFormat.getDateInstance(DateFormat.FULL, locale);
                    break;
                default: // pattern
                    newFormat = new SimpleDateFormat(arg, locale);
                    break;
                }
            } else {
                newFormat = DateFormat.getDateInstance(DateFormat.DEFAULT, locale);
            }
            return newFormat.format(date);
        }
    }

    /* 時刻を処理するデフォルトフォーマット（java.text.DateFormat を使用）。 */
    private static final class TimeFormater implements Formater {
        String[] keyword = new String[] { "default", "short", "medium", "long", "full" };

        public String[] getIds() {
            return new String[] { "time" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            Date date = null;
            if (value instanceof java.util.Date) {
                date = (Date) value;
            } else if (value instanceof Calendar) {
                date = ((Calendar) value).getTime();
            } else {
                throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
            }
            //get the format
            Format newFormat = null;
            arg = arg.trim().toLowerCase();
            if (Eval.isBlank(arg)) {
                switch (Seek.indexOf(keyword, arg, 0, keyword.length)) {
                case 0: // default
                    newFormat = DateFormat.getTimeInstance(DateFormat.DEFAULT, locale);
                    break;
                case 1: // short
                    newFormat = DateFormat.getTimeInstance(DateFormat.SHORT, locale);
                    break;
                case 2: // medium
                    newFormat = DateFormat.getTimeInstance(DateFormat.DEFAULT, locale);
                    break;
                case 3: // long
                    newFormat = DateFormat.getTimeInstance(DateFormat.LONG, locale);
                    break;
                case 4: // full
                    newFormat = DateFormat.getTimeInstance(DateFormat.FULL, locale);
                    break;
                default: // pattern
                    newFormat = new SimpleDateFormat(arg, locale);
                    break;
                }
            } else {
                newFormat = DateFormat.getDateInstance(DateFormat.DEFAULT, locale);
            }
            return newFormat.format(date);
        }
    }

    /* 数値を処理するデフォルトフォーマット（java.text.ChoiceFormat を使用）。 */
    private static final class ChoiceFormater implements Formater {
        public String[] getIds() {
            return new String[] { "choice" };
        }

        public CharSequence format(Locale locale, String id, String arg, Object value, Object[] values) {
            //get the format
            Format newFormat = null;
            try {
                newFormat = new ChoiceFormat(arg);
            } catch (Exception e) {
                throw new IllegalArgumentException("Choice Pattern incorrect");
            }
            return newFormat.format(value);
        }
    }

}
