/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

import shohaku.core.lang.Eval;

/**
 * 日付型のユーティリティメソッドを提供します。
 */
public class DateUtils {

    /** デフォルトの日付フォーマット（yyyy-MM-dd HH:mm:ss.SSS）。 */
    public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss.SSS";
    
    /**
     * 文字列を指定の書式として厳密な解析を行い日付型に変換します。 変換に失敗した場合NULLを返す。
     * 
     * @param text
     *            文字列
     * @param pattern
     *            書式パターン
     * @return 日付
     */
    public static Date parse(String text, String pattern) {
        return parse(text, pattern, false, true);
    }

    /**
     * 文字列を指定の書式と厳密さで解析を行い日付型に変換します。 変換に失敗した場合NULLを返す。
     * 
     * @param text
     *            文字列
     * @param pattern
     *            書式パターン
     * @param strict
     *            書式解析を厳密に行うか設定する、true=厳密な解析
     * @return 日付
     */
    public static Date parse(String text, String pattern, boolean strict) {
        return parse(text, pattern, false, strict);
    }

    /**
     * 文字列を日付に変換します。変換に失敗した場合NULLを返す。
     * 
     * @param text
     *            文字列
     * @param pattern
     *            書式パターン
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @param strict
     *            書式解析を厳密に行うか設定する、true=厳密な解析
     * @return 日付
     */
    private static Date parse(String text, String pattern, boolean lenient, boolean strict) {
        try {
            if (strict && pattern.length() != text.length()) {
                return null;
            }
            DateFormat format = new SimpleDateFormat(pattern);
            format.setLenient(lenient);
            return format.parse(text);
        } catch (ParseException e) {
            return null;
        }
    }

    /**
     * 文字列を指定の書式の配列順に厳密な解析を行い最初に成功した書式で返します。 全ての変換に失敗した場合NULLを返す。
     * 
     * @param text
     *            文字列
     * @param patterns
     *            書式パターン
     * @return 日付
     */
    public static Date parse(String text, String[] patterns) {
        return parse(text, patterns, false, true);
    }

    /**
     * 文字列を指定の厳密さと書式の配列順に解析を行い最初に成功した書式で返します。 全ての変換に失敗した場合NULLを返す。
     * 
     * @param text
     *            文字列
     * @param patterns
     *            書式パターン
     * @param strict
     *            書式解析を厳密に行うか設定する、true=厳密な解析
     * @return 日付
     */
    public static Date parse(String text, String[] patterns, boolean strict) {
        return parse(text, patterns, false, strict);
    }

    /**
     * 文字列を書式の配列順に解析を行い最初に成功した書式で日付に変換して返します。 全ての変換に失敗した場合NULLを返す。
     * 
     * @param text
     *            文字列
     * @param patterns
     *            書式パターン
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @param strict
     *            書式解析を厳密に行うか設定する、true=厳密な解析
     * @return 日付
     */
    private static Date parse(String text, String[] patterns, boolean lenient, boolean strict) {
        SimpleDateFormat format = new SimpleDateFormat();
        format.setLenient(lenient);
        int textLen = text.length();
        Date d = null;
        for (int i = 0; i < patterns.length; i++) {
            if (strict && patterns[i].length() != textLen) {
                continue;
            }
            format.applyPattern(patterns[i]);
            try {
                d = format.parse(text);
                break;
            } catch (ParseException e) {
                //no op
            }
        }
        return d;
    }

    /*
     * Date or Calendar
     */

    /**
     * 日付型またはカレンダーをデフォルトの書式で日付文字列に変換します。
     * 
     * @param date
     *            日付型またはカレンダー
     * @return 日付文字列
     */
    public static String format(Object date) {
        if (Eval.isAndInstanceOf(date, new Class[] { Date.class, Calendar.class })) {
            return format(Locale.getDefault(), date, DEFAULT_DATE_FORMAT);
        } else {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
    }

    /**
     * 日付型またはカレンダーを指定の書式で日付文字列に変換します。
     * 
     * @param date
     *            日付型またはカレンダー
     * @param pattern
     *            書式パターン
     * @return 日付文字列
     */
    public static String format(Object date, String pattern) {
        if (Eval.isAndInstanceOf(date, new Class[] { Date.class, Calendar.class })) {
            return format(Locale.getDefault(), date, pattern);
        } else {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
    }

    /**
     * 日付型またはカレンダーをデフォルトの書式で日付文字列に変換します。
     * 
     * @param locale
     * @param date
     *            日付型またはカレンダー
     * @param pattern
     *            書式パターン
     * @return 日付文字列
     */
    public static String format(Locale locale, Object date, String pattern) {
        if (!(date instanceof Date || date instanceof Calendar)) {
            throw new IllegalArgumentException("argument o is not Date or Calendar :" + date);
        }
        if (date instanceof Calendar) {
            date = ((Calendar) date).getTime();
        }
        SimpleDateFormat formatter = new SimpleDateFormat(pattern, locale);
        return formatter.format(date);
    }

    /**
     * 日付型またはカレンダーをデフォルトの書式で日付文字列に変換します。
     * 
     * @param locale
     * @param date
     *            日付型またはカレンダー
     * @param style
     *            書式スタイル
     * @return 日付文字列
     */
    public static String format(Locale locale, Object date, int style) {
        if (date instanceof Calendar) {
            date = ((Calendar) date).getTime();
        }
        DateFormat formatter = DateFormat.getDateInstance(style, locale);
        return formatter.format(date);
    }

}
