/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.lang;

import java.io.Serializable;
import java.util.AbstractList;
import java.util.Arrays;
import java.util.List;
import java.util.RandomAccess;

/**
 * プリミティブ型とラッパークラスを相互に交換する機能を提供します。 明示的な Boxing/Unboxing 機能です。
 */
public class Boxing {

    /*
     * primitive class
     */

    /**
     * プリミティブ型のクラスをラッパー型のクラスに変換し返します。 <br>
     * 引数がプリミティブ型でない場合 <code>IllegalArgumentException</code> を発生します。
     * 
     * @param primitiveType
     *            プリミティブ型のクラス
     * @return ラッパー型のクラス
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Class boxClass(Class primitiveType) {

        if (primitiveType == null) {
            throw new NullPointerException();
        }

        if (Boolean.TYPE.equals(primitiveType)) {
            return Boolean.class;
        } else if (Character.TYPE.equals(primitiveType)) {
            return Character.class;
        } else if (Byte.TYPE.equals(primitiveType)) {
            return Byte.class;
        } else if (Short.TYPE.equals(primitiveType)) {
            return Short.class;
        } else if (Integer.TYPE.equals(primitiveType)) {
            return Integer.class;
        } else if (Long.TYPE.equals(primitiveType)) {
            return Long.class;
        } else if (Float.TYPE.equals(primitiveType)) {
            return Float.class;
        } else if (Double.TYPE.equals(primitiveType)) {
            return Double.class;
        } else if (Void.TYPE.equals(primitiveType)) {
            return Void.class;
        } else {
            throw new IllegalArgumentException("It is not a primitive type.");
        }
    }

    /**
     * ラッパー型のクラスをプリミティブ型のクラスに変換し返します。 <br>
     * 引数がラッパークラスでない場合 <code>IllegalArgumentException</code> を発生します。
     * 
     * @param wrpperType
     *            ラッパー型のクラス
     * @return プリミティブ型のクラス
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Class unboxClass(Class wrpperType) {

        if (wrpperType == null) {
            throw new NullPointerException();
        }

        if (Boolean.class.equals(wrpperType)) {
            return Boolean.TYPE;
        } else if (Character.class.equals(wrpperType)) {
            return Character.TYPE;
        } else if (Byte.class.equals(wrpperType)) {
            return Byte.TYPE;
        } else if (Short.class.equals(wrpperType)) {
            return Short.TYPE;
        } else if (Integer.class.equals(wrpperType)) {
            return Integer.TYPE;
        } else if (Long.class.equals(wrpperType)) {
            return Long.TYPE;
        } else if (Float.class.equals(wrpperType)) {
            return Float.TYPE;
        } else if (Double.class.equals(wrpperType)) {
            return Double.TYPE;
        } else if (Void.class.equals(wrpperType)) {
            return Void.TYPE;
        } else {
            throw new IllegalArgumentException("It is not a primitive wrpper type.");
        }
    }

    /*
     * wrapper object to primitive (throws NullPointerException)
     */

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static char unbox(Character src) {
        return src.charValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static boolean unbox(Boolean src) {
        return src.booleanValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static byte unbox(Byte src) {
        return src.byteValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static short unbox(Short src) {
        return src.shortValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static int unbox(Integer src) {
        return src.intValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static long unbox(Long src) {
        return src.longValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static float unbox(Float src) {
        return src.floatValue();
    }

    /**
     * プリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @return 変換されたプリミティブ値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static double unbox(Double src) {
        return src.doubleValue();
    }

    /*
     * wrapper object to primitive (null as nullValue)
     */

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static char unbox(Character src, char nullValue) {
        return (src == null) ? nullValue : src.charValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static boolean unbox(Boolean src, boolean nullValue) {
        return (src == null) ? nullValue : src.booleanValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static byte unbox(Byte src, byte nullValue) {
        return (src == null) ? nullValue : src.byteValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static short unbox(Short src, short nullValue) {
        return (src == null) ? nullValue : src.shortValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static int unbox(Integer src, int nullValue) {
        return (src == null) ? nullValue : src.intValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static long unbox(Long src, long nullValue) {
        return (src == null) ? nullValue : src.longValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static float unbox(Float src, float nullValue) {
        return (src == null) ? nullValue : src.floatValue();
    }

    /**
     * <code>null</code> の場合 <code>nullValue</code> を以外はプリミティブ値に変換して返します。
     * 
     * @param src
     *            変換元の値
     * @param nullValue
     *            変換元が null の場合に設定する値
     * @return 変換されたプリミティブ値
     */
    public static double unbox(Double src, double nullValue) {
        return (src == null) ? nullValue : src.doubleValue();
    }

    /*
     * primitive to wrapper object
     */

    private static final Character[] characterCache = new Character[127 + 1];
    static {
        for (int i = 0; i < characterCache.length; i++)
            characterCache[i] = new Character((char) i);
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Character box(char src) {
        if (src <= 127) { // 
            return characterCache[src];
        }
        return new Character(src);
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Boolean box(boolean src) {
        return Boolean.valueOf(src);
    }

    private static final Byte[] byteCache = new Byte[-(-128) + 127 + 1];
    static {
        for (int i = 0; i < byteCache.length; i++)
            byteCache[i] = new Byte((byte) (i - 128));
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Byte box(byte src) {
        return byteCache[src + 128];//(src + offset)
    }

    private static final Short[] shortCache = new Short[-(-128) + 127 + 1];
    static {
        for (int i = 0; i < shortCache.length; i++)
            shortCache[i] = new Short((short) (i - 128));
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Short box(short src) {
        int asInt = src;
        if (asInt >= -128 && asInt <= 127) { // 
            return shortCache[asInt + 128];//(src + offset)
        }
        return new Short(src);
    }

    private static final Integer[] integerCache = new Integer[-(-128) + 127 + 1];
    static {
        for (int i = 0; i < integerCache.length; i++)
            integerCache[i] = new Integer(i - 128);
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Integer box(int src) {
        if (src >= -128 && src <= 127) { // 
            return integerCache[src + 128];//(src + offset)
        }
        return new Integer(src);
    }

    private static final Long[] longCache = new Long[-(-128) + 127 + 1];
    static {
        for (int i = 0; i < longCache.length; i++)
            longCache[i] = new Long(i - 128);
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Long box(long src) {
        if (src >= -128 && src <= 127) { // will cache
            return longCache[(int) src + 128];//(src + offset)
        }
        return new Long(src);
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Float box(float src) {
        return new Float(src);
    }

    /**
     * プリミティブ型をラッパー型に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型
     * @return 変換されたラッパー型
     */
    public static Double box(double src) {
        return new Double(src);
    }

    /*
     * wrapper object to primitive (throws NullPointerException)
     */

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static char[] unbox(Character[] src) {
        int size = src.length;
        char[] a = new char[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static boolean[] unbox(Boolean[] src) {
        int size = src.length;
        boolean[] a = new boolean[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static byte[] unbox(Byte[] src) {
        int size = src.length;
        byte[] a = new byte[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static short[] unbox(Short[] src) {
        int size = src.length;
        short[] a = new short[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static int[] unbox(Integer[] src) {
        int size = src.length;
        int[] a = new int[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static long[] unbox(Long[] src) {
        int size = src.length;
        long[] a = new long[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static float[] unbox(Float[] src) {
        int size = src.length;
        float[] a = new float[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @throws NullPointerException
     *             引数がまたは要素が <code>null</code> の場合
     */
    public static double[] unbox(Double[] src) {
        int size = src.length;
        double[] a = new double[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i]);
        }
        return a;
    }

    /*
     * wrapper object to primitive (element null as nullValue)
     */

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static char[] unbox(Character[] src, char nullValue) {
        int size = src.length;
        char[] a = new char[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static boolean[] unbox(Boolean[] src, boolean nullValue) {
        int size = src.length;
        boolean[] a = new boolean[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static byte[] unbox(Byte[] src, byte nullValue) {
        int size = src.length;
        byte[] a = new byte[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static short[] unbox(Short[] src, short nullValue) {
        int size = src.length;
        short[] a = new short[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static int[] unbox(Integer[] src, int nullValue) {
        int size = src.length;
        int[] a = new int[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static long[] unbox(Long[] src, long nullValue) {
        int size = src.length;
        long[] a = new long[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static float[] unbox(Float[] src, float nullValue) {
        int size = src.length;
        float[] a = new float[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /**
     * プリミティブ型の配列に変換して返します、要素が <code>null</code> の場合 <code>nullValue</code> を割り当てます。
     * 
     * @param src
     *            変換元の配列
     * @return 変換されたプリミティブ型配列
     * @param nullValue
     *            変換元の要素が null の場合に設定する値
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static double[] unbox(Double[] src, double nullValue) {
        int size = src.length;
        double[] a = new double[size];
        for (int i = 0; i < size; i++) {
            a[i] = unbox(src[i], nullValue);
        }
        return a;
    }

    /*
     * primitive to wrapper object
     */

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Character[] box(char[] src) {
        int len = src.length;
        Character[] a = new Character[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Boolean[] box(boolean[] src) {
        int len = src.length;
        Boolean[] a = new Boolean[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Byte[] box(byte[] src) {
        int len = src.length;
        Byte[] a = new Byte[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Short[] box(short[] src) {
        int len = src.length;
        Short[] a = new Short[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Integer[] box(int[] src) {
        int len = src.length;
        Integer[] a = new Integer[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Long[] box(long[] src) {
        int len = src.length;
        Long[] a = new Long[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Float[] box(float[] src) {
        int len = src.length;
        Float[] a = new Float[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /**
     * プリミティブ型の配列をラッパー型の配列に変換して返します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換されたラッパー型の配列
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static Double[] box(double[] src) {
        int len = src.length;
        Double[] a = new Double[len];
        for (int i = 0; i < len; i++) {
            a[i] = box(src[i]);
        }
        return a;
    }

    /*
     * primitive array as List
     */

    /**
     * プリミティブ型配列かオブジェクト型配列か識別して固定サイズのリストに変換して返します。 <br>
     * 引数が配列型でない場合 <code>IllegalArgumentException</code> が発生します。
     * 
     * @param array
     *            変換元の配列
     * @return 変換された <code>List</code> 表現
     * @throws IllegalArgumentException
     *             引数が配列型でない場合
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(Object array) {
        Class c = array.getClass();
        if (c.isArray()) {
            Class type = c.getComponentType();
            if (Boolean.TYPE.equals(type)) {
                return asList((boolean[]) array);
            } else if (Character.TYPE.equals(type)) {
                return asList((char[]) array);
            } else if (Byte.TYPE.equals(type)) {
                return asList((byte[]) array);
            } else if (Short.TYPE.equals(type)) {
                return asList((short[]) array);
            } else if (Integer.TYPE.equals(type)) {
                return asList((int[]) array);
            } else if (Long.TYPE.equals(type)) {
                return asList((long[]) array);
            } else if (Float.TYPE.equals(type)) {
                return asList((float[]) array);
            } else if (Double.TYPE.equals(type)) {
                return asList((double[]) array);
            } else {
                return Arrays.asList((Object[]) array);
            }
        } else {
            throw new IllegalArgumentException("It is not an array class. class:" + c);
        }

    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(char[] src) {
        return new BoxingCharacterList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(boolean[] src) {
        return new BoxingBooleanList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(byte[] src) {
        return new BoxingByteList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(short[] src) {
        return new BoxingShortList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(int[] src) {
        return new BoxingIntegerList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(long[] src) {
        return new BoxingLongList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(float[] src) {
        return new BoxingFloatList(src);
    }

    /**
     * 指定されたプリミティブ型の配列を基にする固定サイズのリストを返します。 <br>
     * 返されたリストへの変更は、そのまま配列に書き込まれます。 <br>
     * このメソッドは、 <code>Collection.toArray</code> と組み合わせることで、 配列ベースの API とコレクションベースの API の橋渡し役として機能します。 <br>
     * また、返されるリストは直列化可能で、 <code>RandomAccess</code> を実装します。
     * 
     * @param src
     *            変換元のプリミティブ型配列
     * @return 変換された <code>List</code> 表現
     * @throws NullPointerException
     *             引数が <code>null</code> の場合
     */
    public static List asList(double[] src) {
        return new BoxingDoubleList(src);
    }

    /*
     * class
     */

    /* char[]のリストビューを定義します。 */
    private static class BoxingCharacterList extends AbstractList implements RandomAccess, Serializable {
        private final Character[] oa;

        private final char[] a;

        private final int size;

        private BoxingCharacterList(char[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Character[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Character e = oa[index];
            char c = a[index];
            if (e == null || e.charValue() != c) {
                e = box(c);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((char[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Character e = (Character) element;
            oa[index] = e;
            a[index] = e.charValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Character) {
                char c = ((Character) o).charValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* boolean[]のリストビューを定義します。 */
    private static class BoxingBooleanList extends AbstractList implements RandomAccess, Serializable {
        private final Boolean[] oa;

        private final boolean[] a;

        private final int size;

        private BoxingBooleanList(boolean[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Boolean[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Boolean e = oa[index];
            boolean b = a[index];
            if (e == null || e.booleanValue() != b) {
                e = box(b);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((boolean[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Boolean e = (Boolean) element;
            oa[index] = e;
            a[index] = e.booleanValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Boolean) {
                boolean c = ((Boolean) o).booleanValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* byte[]のリストビューを定義します。 */
    private static class BoxingByteList extends AbstractList implements RandomAccess, Serializable {
        private final Byte[] oa;

        private final byte[] a;

        private final int size;

        private BoxingByteList(byte[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Byte[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Byte e = oa[index];
            byte n = a[index];
            if (e == null || e.byteValue() != n) {
                e = box(n);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((byte[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Byte e = (Byte) element;
            oa[index] = e;
            a[index] = e.byteValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Byte) {
                byte c = ((Byte) o).byteValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* short[]のリストビューを定義します。 */
    private static class BoxingShortList extends AbstractList implements RandomAccess, Serializable {
        private final Short[] oa;

        private final short[] a;

        private final int size;

        private BoxingShortList(short[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Short[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Short e = oa[index];
            short n = a[index];
            if (e == null || e.shortValue() != n) {
                e = box(n);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((short[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Short e = (Short) element;
            oa[index] = e;
            a[index] = e.shortValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Short) {
                short c = ((Short) o).shortValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* int[]のリストビューを定義します。 */
    private static class BoxingIntegerList extends AbstractList implements RandomAccess, Serializable {
        private final Integer[] oa;

        private final int[] a;

        private final int size;

        private BoxingIntegerList(int[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Integer[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Integer e = oa[index];
            int n = a[index];
            if (e == null || e.intValue() != n) {
                e = box(n);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((int[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Integer e = (Integer) element;
            oa[index] = e;
            a[index] = e.intValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Integer) {
                int c = ((Integer) o).intValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* long[]のリストビューを定義します。 */
    private static class BoxingLongList extends AbstractList implements RandomAccess, Serializable {
        private final Long[] oa;

        private final long[] a;

        private final int size;

        private BoxingLongList(long[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Long[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Long e = oa[index];
            long n = a[index];
            if (e == null || e.longValue() != n) {
                e = box(n);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((long[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Long e = (Long) element;
            oa[index] = e;
            a[index] = e.longValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Long) {
                long c = ((Long) o).longValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* float[]のリストビューを定義します。 */
    private static class BoxingFloatList extends AbstractList implements RandomAccess, Serializable {
        private final Float[] oa;

        private final float[] a;

        private final int size;

        private BoxingFloatList(float[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Float[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Float e = oa[index];
            float n = a[index];
            if (e == null || e.floatValue() != n) {
                e = box(n);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((float[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Float e = (Float) element;
            oa[index] = e;
            a[index] = e.floatValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Float) {
                float c = ((Float) o).floatValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

    /* double[]のリストビューを定義します。 */
    private static class BoxingDoubleList extends AbstractList implements RandomAccess, Serializable {
        private final Double[] oa;

        private final double[] a;

        private final int size;

        private BoxingDoubleList(double[] src) {
            super();
            if (src == null) {
                throw new NullPointerException();
            }
            this.a = src;
            this.oa = new Double[src.length];
            this.size = src.length;
        }

        public Object get(int index) {
            Double e = oa[index];
            double n = a[index];
            if (e == null || e.doubleValue() != n) {
                e = box(n);
                oa[index] = e;
            }
            return e;
        }

        public Object[] toArray() {
            return box((double[]) a);
        }

        public int size() {
            return size;
        }

        public Object set(int index, Object element) {
            Object oldValue = get(index);
            Double e = (Double) element;
            oa[index] = e;
            a[index] = e.doubleValue();
            return oldValue;
        }

        public int indexOf(Object o) {
            if (o != null && o instanceof Double) {
                double c = ((Double) o).doubleValue();
                for (int i = 0; i < a.length; i++) {
                    if (c == a[i]) {
                        return i;
                    }
                }
            }
            return -1;
        }

        public boolean contains(Object o) {
            return (indexOf(o) != -1);
        }
    }

}
