/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.util.Collection;
import java.util.List;
import java.util.ListIterator;

/**
 * 指定されたリストをラッピングするビューを定義します。 <br>
 * <br>
 * このクラスはリストの実装を隠蔽する目的で使用されます。またリストをラッピングするクラスの基底実装の役割を持ちます。
 */
public class WrappedList extends WrappedCollection implements List {

    /** 基となるリスト。 */
    protected final List list;

    /**
     * ラップするリストを指定して初期化します。
     * 
     * @param list
     *            ラップするリスト
     */
    public WrappedList(List list) {
        super(list);
        this.list = list;
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#get(int)
     */
    public Object get(int index) {
        return list.get(index);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#remove(int)
     */
    public Object remove(int index) {
        return list.remove(index);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#add(int, java.lang.Object)
     */
    public void add(int index, Object element) {
        list.add(index, element);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#indexOf(java.lang.Object)
     */
    public int indexOf(Object o) {
        return list.indexOf(o);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#lastIndexOf(java.lang.Object)
     */
    public int lastIndexOf(Object o) {
        return list.lastIndexOf(o);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#addAll(int, java.util.Collection)
     */
    public boolean addAll(int index, Collection c) {
        return list.addAll(index, c);
    }

    /**
     * 基となるリストの同メソッドの戻り値を <code>WrappedList</code> でラッピングしたリストを返します。
     * 
     * @see java.util.List#subList(int, int)
     */
    public List subList(int fromIndex, int toIndex) {
        return new WrappedList(list.subList(fromIndex, toIndex));
    }

    /**
     * 基となるリストの同メソッドの戻り値を <code>WrappedListIterator</code> でラッピングしたリストの反復子を返します。
     * 
     * @see java.util.List#listIterator()
     */
    public ListIterator listIterator() {
        return new WrappedListIterator(list.listIterator());
    }

    /**
     * 基となるリストの同メソッドの戻り値を <code>WrappedListIterator</code> でラッピングしたリストの反復子を返します。
     * 
     * @see java.util.List#listIterator(int)
     */
    public ListIterator listIterator(final int index) {
        return new WrappedListIterator(list.listIterator(index));
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.util.List#set(int, java.lang.Object)
     */
    public Object set(int index, Object element) {
        return list.set(index, element);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        return list.equals(o);
    }

    /**
     * 基となるリストの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return list.hashCode();
    }
}