/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import shohaku.core.collections.Group;

/**
 * 指定されたグループをラッピングするビューを定義します。 <br>
 * <br>
 * このクラスはグループの実装を隠蔽する目的で使用されます。またグループをラッピングするクラスの基底実装の役割を持ちます。 <br>
 * ラッピングされたグループが直列化可能の場合は直列化可能です。
 */
public class WrappedGroup implements Group, Serializable {

    /** 基となるグループ。 */
    protected final Group group;

    /**
     * ラップするグループを指定して初期化します。
     * 
     * @param g
     *            ラップするグループ
     */
    public WrappedGroup(Group g) {
        if (g == null) {
            throw new NullPointerException();
        }
        this.group = g;
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedCollection</code> でラッピングしたコレクションを返します。
     * 
     * @see shohaku.core.collections.Group#get(java.lang.Object)
     */
    public Collection get(Object key) {
        return new WrappedCollection(group.get(key));
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#size()
     */
    public int size() {
        return group.size();
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#size(java.lang.Object)
     */
    public int size(Object key) {
        return group.size(key);
    }

    /** キーのセットをキャッシュします。 */
    protected transient Set keySet = null;

    /** エントリのセットをキャッシュします。 */
    protected transient Set entrySet = null;

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedSet</code> でラッピングしたセットを返します。
     * 
     * @see shohaku.core.collections.Group#keySet()
     */
    public Set keySet() {
        if (keySet == null) {
            keySet = new WrappedSet(group.keySet());
        }
        return keySet;
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedGroup.WrappedGroupEntrySet</code> でラッピングしたセットを返します。
     * 
     * @see shohaku.core.collections.Group#entrySet()
     */
    public Set entrySet() {
        if (entrySet == null) {
            entrySet = new WrappedGroupEntrySet(group.entrySet());
        }
        return entrySet;
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#isEmpty()
     */
    public boolean isEmpty() {
        return group.isEmpty();
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#containsKey(java.lang.Object)
     */
    public boolean containsKey(Object key) {
        return group.containsKey(key);
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#containsValue(java.lang.Object, java.lang.Object)
     */
    public boolean containsValue(Object key, Object value) {
        return group.containsValue(key, value);
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedIterator</code> でラッピングした反復子を返します。
     * 
     * @see shohaku.core.collections.Group#iterator()
     */
    public Iterator iterator() {
        return new WrappedIterator(group.iterator());
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedIterator</code> でラッピングした反復子を返します。
     * 
     * @see shohaku.core.collections.Group#iterator(java.lang.Object)
     */
    public Iterator iterator(Object key) {
        return new WrappedIterator(group.iterator(key));
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object)
     */
    public boolean add(Object key) {
        return group.add(key);
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#add(java.lang.Object, java.lang.Object)
     */
    public boolean add(Object key, Object value) {
        return group.add(key, value);
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#addAll(shohaku.core.collections.Group)
     */
    public void addAll(Group g) {
        group.addAll(g);
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#addAll(java.lang.Object, java.util.Collection)
     */
    public boolean addAll(Object key, Collection c) {
        return group.addAll(key, c);
    }

    /**
     * 基となるグループの同メソッドの戻り値を <code>WrappedCollection</code> でラッピングしたコレクションを返します。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object)
     */
    public Collection remove(Object key) {
        return new WrappedCollection(group.remove(key));
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#remove(java.lang.Object, java.lang.Object)
     */
    public boolean remove(Object key, Object value) {
        return group.remove(key, value);
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see shohaku.core.collections.Group#clear()
     */
    public void clear() {
        group.clear();
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        return group.equals(o);
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return group.hashCode();
    }

    /**
     * 基となるグループの同メソッドを呼び出します。
     * 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return group.toString();
    }

    /**
     * <code>Group.Entry</code> をラッピングするセットビューを定義します。
     */
    protected class WrappedGroupEntrySet extends WrappedSet {

        /**
         * ラップするセットを指定して初期化します。
         * 
         * @param s
         *            ラップするセット
         */
        protected WrappedGroupEntrySet(Set s) {
            super(s);
        }

        /**
         * <code>Group.Entry</code> をラップする反復子を返します。
         * 
         * @see java.util.Collection#iterator()
         */
        public Iterator iterator() {
            return wrappedIterator(coll.iterator());
        }

        /**
         * <code>Group.Entry</code> をラップする <code>WrappedGroup.WrappedGroupEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray()
         */
        public Object[] toArray() {
            Object[] a = coll.toArray();
            for (int i = 0; i < a.length; i++) {
                a[i] = wrappedEntry((Group.Entry) a[i]);
            }
            return a;
        }

        /**
         * <code>Group.Entry</code> をラップする <code>WrappedGroup.WrappedGroupEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray(java.lang.Object[])
         */
        public Object[] toArray(Object a[]) {
            Object[] arr = coll.toArray(a.length == 0 ? a : (Object[]) java.lang.reflect.Array.newInstance(a.getClass()
                    .getComponentType(), 0));
            for (int i = 0; i < arr.length; i++) {
                arr[i] = wrappedEntry((Group.Entry) arr[i]);
            }
            if (arr.length > a.length) {
                return arr;
            }

            System.arraycopy(arr, 0, a, 0, arr.length);
            if (a.length > arr.length) {
                a[arr.length] = null;
            }
            return a;
        }

        /**
         * 
         * <code>Group.Entry</code> をラップする <code>WrappedGroup.WrappedGroupEntry</code> に変換して検索します。
         * 
         * @see java.util.Collection#contains(java.lang.Object)
         */
        public boolean contains(Object o) {
            if (!(o instanceof Group.Entry)) {
                return false;
            }
            return coll.contains(wrappedEntry((Group.Entry) o));
        }

        /**
         * <code>WrappedGroup.WrappedGroupEntry</code> で検索するため <code>Set#contains(Object o)</code> を呼び出して実装します。
         * 
         * @see java.util.Collection#containsAll(java.util.Collection)
         */
        public boolean containsAll(Collection coll) {
            Iterator e = coll.iterator();
            while (e.hasNext()) {
                if (!contains(e.next())) { // Invokes safe contains() above
                    return false;
                }
            }
            return true;
        }

        /**
         * 反復子をラップします。
         * 
         * @param iterator
         *            反復子
         * @return ラップした反復子
         */
        protected Iterator wrappedIterator(Iterator iterator) {
            return new WrappedIterator(iterator) {
                public Object next() {
                    return wrappedEntry((Group.Entry) i.next());
                }
            };
        }

        /**
         * グループエントリをラップします。
         * 
         * @param entry
         *            グループエントリ
         * @return ラップしたグループエントリ
         */
        protected Group.Entry wrappedEntry(Group.Entry entry) {
            return new WrappedGroupEntry(entry);
        }
    }

    /**
     * <code>Group.Entry</code> をラッピングするビューを定義します。
     */
    protected class WrappedGroupEntry implements Group.Entry {

        /** 基となるグループエントリ。 */
        protected final Group.Entry e;

        /**
         * ラップするグループエントリを指定して初期化します。
         * 
         * @param e
         *            ラップするグループエントリ
         */
        protected WrappedGroupEntry(Group.Entry e) {
            this.e = e;
        }

        /**
         * 基となるグループエントリの同メソッドを呼び出します。
         * 
         * @see shohaku.core.collections.Group.Entry#getKey()
         */
        public Object getKey() {
            return e.getKey();
        }

        /**
         * 基となるグループエントリの同メソッドの戻り値を <code>WrappedCollection</code> でラッピングしたコレクションを返します。
         * 
         * @see shohaku.core.collections.Group.Entry#getValues()
         */
        public Collection getValues() {
            return new WrappedCollection(e.getValues());
        }

        /**
         * 基となるグループエントリの同メソッドを呼び出します。
         * 
         * @see java.lang.Object#hashCode()
         */
        public int hashCode() {
            return e.hashCode();
        }

        /**
         * 基となるグループエントリの同メソッドを呼び出します。
         * 
         * @see java.lang.Object#equals(java.lang.Object)
         */
        public boolean equals(Object o) {
            return e.equals(o);
        }

        /**
         * 基となるグループエントリの同メソッドを呼び出します。
         * 
         * @see java.lang.Object#toString()
         */
        public String toString() {
            return e.toString();
        }
    }
}