/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * 指定されたマップの変更不可能なビューを定義します。 <br>
 * <br>
 * このクラスは <code>java.util.Collections#unmodifiableMap(Map)</code> で返されるクラスとほぼ同等の仕様です。 <br>
 */
public class UnmodifiableMap extends WrappedMap {

    /**
     * ラップするマップを指定して初期化します。
     * 
     * @param m
     *            ラップするマップ
     */
    public UnmodifiableMap(Map m) {
        super(m);
    }

    /*
     * UnsupportedOperation
     */

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    public Object put(Object key, Object value) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Map#remove(java.lang.Object)
     */
    public Object remove(Object key) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Map#putAll(java.util.Map)
     */
    public void putAll(Map t) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Map#clear()
     */
    public void clear() {
        throw new UnsupportedOperationException();
    }

    /*
     * Unmodifiable
     */

    private transient Set keySet = null;

    private transient Set entrySet = null;

    private transient Collection values = null;

    /**
     * 基となるマップの同メソッドの戻り値を <code>UnmodifiableSet</code> でラッピングしたセットを返します。
     * 
     * @see java.util.Map#keySet()
     */
    public Set keySet() {
        if (keySet == null) {
            keySet = new UnmodifiableSet(map.keySet());
        }
        return keySet;
    }

    /**
     * 基となるマップの同メソッドの戻り値を <code>UnmodifiableEntrySet</code> でラッピングしたセットを返します。
     * 
     * @see java.util.Map#entrySet()
     */
    public Set entrySet() {
        if (entrySet == null) {
            entrySet = new UnmodifiableEntrySet(map.entrySet());
        }
        return entrySet;
    }

    /**
     * 基となるマップの同メソッドの戻り値を <code>UnmodifiableCollection</code> でラッピングしたコレクションを返します。
     * 
     * @see java.util.Map#values()
     */
    public Collection values() {
        if (values == null) {
            values = new UnmodifiableCollection(map.values());
        }
        return values;
    }

    /*
     * classes
     */

    /**
     * <code>Map.Entry</code> をラッピングする変更不可能なセットビューを定義します。
     */
    protected class UnmodifiableEntrySet extends UnmodifiableSet {

        /**
         * ラップするセットを指定して初期化します。
         * 
         * @param s
         *            ラップするセット
         */
        protected UnmodifiableEntrySet(Set s) {
            super(s);
        }

        /**
         * <code>Map.Entry</code> をラップする反復子を返します。
         * 
         * @see java.util.Collection#iterator()
         */
        public Iterator iterator() {
            return wrappedIterator(coll.iterator());
        }

        /**
         * <code>Map.Entry</code> をラップする <code>WrappedMap.WrappedMapEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray()
         */
        public Object[] toArray() {
            Object[] a = coll.toArray();
            for (int i = 0; i < a.length; i++) {
                a[i] = wrappedEntry((Map.Entry) a[i]);
            }
            return a;
        }

        /**
         * <code>Map.Entry</code> をラップする <code>WrappedMap.WrappedMapEntry</code> の配列に変換して返します。
         * 
         * @see java.util.Collection#toArray(java.lang.Object[])
         */
        public Object[] toArray(Object a[]) {
            Object[] arr = coll.toArray(a.length == 0 ? a : (Object[]) java.lang.reflect.Array.newInstance(a.getClass()
                    .getComponentType(), 0));
            for (int i = 0; i < arr.length; i++) {
                arr[i] = wrappedEntry((Map.Entry) arr[i]);
            }
            if (arr.length > a.length) {
                return arr;
            }

            System.arraycopy(arr, 0, a, 0, arr.length);
            if (a.length > arr.length) {
                a[arr.length] = null;
            }
            return a;
        }

        /**
         * 
         * <code>Map.Entry</code> をラップする <code>WrappedMap.WrappedMapEntry</code> に変換して検索します。
         * 
         * @see java.util.Collection#contains(java.lang.Object)
         */
        public boolean contains(Object o) {
            if (!(o instanceof Map.Entry)) {
                return false;
            }
            return coll.contains(wrappedEntry((Map.Entry) o));
        }

        /**
         * <code>WrappedMap.WrappedMapEntry</code> で検索するため <code>Set#contains(Object o)</code> を呼び出して実装します。
         * 
         * @see java.util.Collection#containsAll(java.util.Collection)
         */
        public boolean containsAll(Collection coll) {
            Iterator e = coll.iterator();
            while (e.hasNext()) {
                if (!contains(e.next())) { // Invokes safe contains() above
                    return false;
                }
            }
            return true;
        }

        /**
         * <code>WrappedMap.WrappedMapEntry</code> で検索するため <code>Set#containsAll(Set s)</code> を呼び出して実装します。
         * 
         * @see java.lang.Object#equals(java.lang.Object)
         */
        public boolean equals(Object o) {
            if (o == this) {
                return true;
            }

            if (!(o instanceof Set)) {
                return false;
            }
            Set s = (Set) o;
            if (s.size() != coll.size()) {
                return false;
            }
            return containsAll(s); // Invokes safe containsAll() above
        }

        /**
         * 反復子を変更不可能なビューでラップします。
         * 
         * @param iterator
         *            反復子
         * @return ラップした反復子
         */
        protected Iterator wrappedIterator(Iterator iterator) {
            return new UnmodifiableIterator(iterator) {
                public Object next() {
                    return wrappedEntry((Map.Entry) super.next());
                }
            };
        }

        /**
         * マップエントリを変更不可能なビューでラップします。
         * 
         * @param entry
         *            マップエントリ
         * @return ラップしたマップエントリ
         */
        protected Map.Entry wrappedEntry(Map.Entry entry) {
            return new UnmodifiableEntry(entry);
        }

    }

    /**
     * <code>Map.Entry</code> をラッピングする変更不可能なビューを定義します。
     */
    protected class UnmodifiableEntry extends WrappedMapEntry {

        /**
         * ラップするマップエントリを指定して初期化します。
         * 
         * @param e
         *            ラップするマップエントリ
         */
        protected UnmodifiableEntry(Map.Entry e) {
            super(e);
        }

        /**
         * <code>UnsupportedOperationException</code> を発生させます。
         * 
         * @see java.util.Map.Entry#setValue(java.lang.Object)
         */
        public Object setValue(Object value) {
            throw new UnsupportedOperationException();
        }
    }
}