/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.wrapper;

import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;

/**
 * 指定されたコレクションの変更不可能なビューを定義します。 <br>
 * <br>
 * このクラスは <code>java.util.Collections#unmodifiableCollection(Collection)</code> で返されるクラスとほぼ同等の仕様です。 <br>
 * <br>
 * コレクションのオペレーションの規約を守るため、 <code>hashCode</code> オペレーションおよび <code>equals</code> オペレーションを基となるコレクションに渡すことはなく、
 * <code>Object</code> の <code>equals</code> メソッドおよび <code>hashCode</code> メソッドに依存します。
 */
public class UnmodifiableCollection extends WrappedCollection implements Collection, Serializable {

    /**
     * ラップするコレクションを指定して初期化します。
     * 
     * @param c
     *            ラップするコレクション
     */
    public UnmodifiableCollection(Collection c) {
        super(c);
    }

    /*
     * Unmodifiable
     */

    /**
     * 変更不可の反復子を返します。
     * 
     * @return 変更不可の反復子
     * @see java.util.Collection#iterator()
     */
    public Iterator iterator() {
        return new UnmodifiableIterator(coll.iterator());
    }

    /*
     * UnsupportedOperation
     */

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Collection#add(java.lang.Object)
     */
    public boolean add(Object o) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Collection#remove(java.lang.Object)
     */
    public boolean remove(Object o) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Collection#addAll(java.util.Collection)
     */
    public boolean addAll(Collection c) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Collection#removeAll(java.util.Collection)
     */
    public boolean removeAll(Collection c) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Collection#retainAll(java.util.Collection)
     */
    public boolean retainAll(Collection c) {
        throw new UnsupportedOperationException();
    }

    /**
     * <code>UnsupportedOperationException</code> を発生させます。
     * 
     * @see java.util.Collection#clear()
     */
    public void clear() {
        throw new UnsupportedOperationException();
    }
}