/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.collections.group;

import java.util.AbstractSet;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import shohaku.core.collections.Group;
import shohaku.core.collections.IteratorUtils;

/**
 * このクラスは、Map を基にした Group インタフェースのスケルトン実装を提供し、このインタフェースを実装するのに必要な作業量を最小限に抑えます。
 */
public abstract class AbstractMapGroup implements Group {

    /**
     * グループエントリのシンプル実装です。
     */
    protected static class SimpleEntry implements Entry {

        /** このエントリのキー。 */
        protected final Object key;

        /** このエントリのコレクション。 */
        protected final Collection values;

        /**
         * キーとコレクションを格納して初期化します。
         * 
         * @param k
         *            キー
         * @param v
         *            コレクション
         */
        protected SimpleEntry(Object k, Collection v) {
            this.key = k;
            this.values = v;
        }

        /**
         * このグループエントリのキーを返却します。
         * 
         * @return このエントリのキー
         */
        public Object getKey() {
            return key;
        }

        /**
         * このグループエントリされるコレクションを返却します。
         * 
         * @return このエントリのマッピングされるコレクション
         */
        public Collection getValues() {
            return values;
        }

        /**
         * 同値として認識出来るか検証します。 <br>
         * Group.Entry 型で同一のキーとコレクションを持つ場合に同一と認識されます。
         * <p>
         * グループエントリ e1 と e2 の同値性比較は、次のように定義されます。 <br>
         * <code>
         *      (e1.getKey().equals(e2) && e1.getValue().equals(e2))
         * </code>
         * </p>
         * 
         * @param o
         *            同値性を検証するオブジェクト
         * @return 同値として認識出来る場合True
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            return (key.equals(e.getKey()) && values.equals(e.getValues()));
        }

        /**
         * このグループエントリのハッシュコード値を返します。
         * <p>
         * グループエントリ e のハッシュコードは、次のように定義されます。 <br>
         * <code>
         *      (e.getKey().hashCode() + e.getValue().hashCode())
         * </code> これにより、Object.hashCode の一般規約によって要求される、 任意の 2
         * つのエントリ e1 と e2 で、e1.equals(e2) であれば e1.hashCode()==e2.hashCode() となることが保証されます。
         * </p>
         * 
         * @return グループエントリのハッシュコード値
         */
        public int hashCode() {
            return key.hashCode() + values.hashCode();
        }

        /**
         * グループエントリの文字列表現を返却します。 <br>
         * <code>
         *      e.getKey() + e.getValue() の書式で出力します。
         * </code>
         * 
         * @return グループエントリの文字列表現
         */
        public String toString() {
            return key + "=" + values;
        }

    }

    /** 実装の基となる Map。 */
    protected final Map groupMap;

    /**
     * 基にする Map を登録して初期化します。
     * 
     * @param groupMap
     *            基にする Map
     */
    protected AbstractMapGroup(Map groupMap) {
        this.groupMap = groupMap;
    }

    /**
     * グループのマッピングをすべて削除します。
     * 
     * @see shohaku.core.collections.Group#clear()
     */
    public void clear() {
        groupMap.clear();
    }

    /**
     * グループがキーとコレクションのマッピングを保持しない場合に true を返します。
     * 
     * @return グループがキーとコレクションのマッピングを保持しない場合は true
     * @see shohaku.core.collections.Group#isEmpty()
     */
    public boolean isEmpty() {
        return groupMap.isEmpty();
    }

    /**
     * 指定されたキーに対するコレクションを追加します (任意のオペレーション)。 <br>
     * グループに既にこのキーに対するマッピングがある場合何も行われません。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定されるコレクションが関連付けられるキー
     * @return 既にこのキーに対するマッピングがある場合 true
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#add(java.lang.Object)
     */
    public boolean add(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        return addEntry(key);
    }

    /**
     * 指定された値を指定されたキーに対応するコレクションに追加します (任意のオペレーション)。 <br>
     * グループにこのキーに対するマッピングが存在しない場合は新規に追加されます。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定される値が関連付けられるキー
     * @param value
     *            指定されるキーに関連付けられる値
     * @return 既にこのキーに対するマッピングがある場合 true
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#add(java.lang.Object, java.lang.Object)
     */
    public boolean add(Object key, Object value) {
        if (key == null) {
            throw new NullPointerException();
        }
        return addEntryValue(key, value);
    }

    /**
     * 指定されたコレクションの全要素をキーに対応するコレクションに追加します (任意のオペレーション)。 <br>
     * グループにこのキーに対するマッピングが存在しない場合は新規に追加されます。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定されるコレクションの全要素が関連付けられるキー
     * @param c
     *            指定されるキーに関連付けられるコレクション
     * @return 既にこのキーに対するマッピングがある場合 true
     * @throws NullPointerException
     *             指定されたキーまたはコレクションがNullの場合
     * @see shohaku.core.collections.Group#addAll(java.lang.Object, java.util.Collection)
     */
    public boolean addAll(Object key, Collection c) {
        if (key == null || c == null) {
            throw new NullPointerException();
        }
        return addEntryValues(key, c);
    }

    /**
     * 指定されたグループのすべてのキーと対応する全値ををこのグループに追加します (任意のオペレーション)。
     * <p>
     * 指定されたグループのキー k から値 v までの各マッピングに関して、 この呼び出しの効果は、putAll(k, v) を呼び出した場合と同じです。
     * 指定されたグループがこのオペレーションの処理中に変更された場合、そのオペレーションの動作は指定外となります。
     * </p>
     * 
     * @param g
     *            グループに追加されるグループ
     * @throws NullPointerException
     *             指定されたグループがNullの場合
     * @see shohaku.core.collections.Group#addAll(shohaku.core.collections.Group)
     */
    public void addAll(Group g) {
        if (g == null) {
            throw new NullPointerException();
        }
        addAllEntrys(g);
    }

    /**
     * グループに含まれるグループエントリの反復子を返します。 <br>
     * 反復子の要素は Group.Entry 型に為ります。 エントリが存在しない場合空の反復子が返ります。
     * 
     * @return グループエントリの反復子
     * @see shohaku.core.collections.Group#iterator()
     */
    public Iterator iterator() {
        return newEntryIterator();
    }

    /**
     * 指定されたキーに対応するコレクションの要素の反復子を返します。 <br>
     * 指定されたキー対応するコレクションが存在しない場合空の反復子が返ります。
     * 
     * @param key
     *            関連づけられている識別キー
     * @return キーに対応するコレクションの反復子
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#iterator(java.lang.Object)
     */
    public Iterator iterator(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        Collection values = getEntryValues(key);
        return (values != null) ? values.iterator() : IteratorUtils.EMPTY_ITERATOR;
    }

    /**
     * 指定されたキーのマッピングするコレクションを返します。 <br>
     * グループがこのキーのマッピングを保持していない場合はnull を返します。
     * 
     * @param key
     *            関連づけられている識別キー
     * @return 指定のキーをマッピングするコレクション
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#get(java.lang.Object)
     */
    public Collection get(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        return getEntryValues(key);
    }

    /**
     * このキーにマッピングがある場合に、そのマッピングをグループから削除します (任意のオペレーション)。
     * <p>
     * つまり、 (key==null ? k==null : key.equals(k)) という条件で、 キー k からコレクション c までマッピングがグループに含まれる場合、このマッピングは削除されます。
     * グループはこのようなマッピングを 1 つだけ含みます。
     * </p>
     * <p>
     * キーに以前対応付けられていたコレクションを返します。 このキーのマッピングがグループにない場合は、null を返します 1 度呼び出しが返れば、グループは指定されたキーのマッピングを含みません。
     * </p>
     * 
     * @param key
     *            対応付けられているキー
     * @return 指定されたキーと対応付けられていた以前のコレクション。キーのマッピングがなかった場合は null
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#remove(java.lang.Object)
     */
    public Collection remove(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        return removeEntryByKey(key);
    }

    /**
     * このキーにマッピングがある場合に、そのマッピングの要素から指定の値を削除します (任意のオペレーション)。
     * <p>
     * つまり、 (key==null ? k==null : key.equals(k)) という条件で、キー k からコレクション c までマッピングがグループに含まれる場合、 (value==null ? e==null :
     * value.equals(e)) である要素 e がコレクションに 1 つ以上格納されている場合に、これを削除します。
     * </p>
     * <p>
     * 指定された要素がマッピングされたコレクション内に格納されていた場合には true を返します。 <br>
     * すなわち、この呼び出しの結果、グループが変更された場合に true を返します。
     * </p>
     * 
     * @param key
     *            関連づけられている識別キー
     * @param value
     * @return 指定されたキーと関連付けられていた以前のコレクション。キーのマッピングがなかった場合は null
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#remove(java.lang.Object, java.lang.Object)
     */
    public boolean remove(Object key, Object value) {
        if (key == null) {
            throw new NullPointerException();
        }
        return removeValue(key, value);
    }

    /**
     * 指定されたキーのマッピングがグループに含まれている場合に true を返します。
     * <p>
     * つまり、グループに、(key==null ? k==null : key.equals(k)) となるキーk のマッピングが含まれている場合にだけ true を返します。 グループはこのようなマッピングを 1
     * つだけ含むことができます。
     * </p>
     * 
     * @param key
     *            マッピングされているか判定されるキー
     * @return グループが指定のキーのマッピングを保持する場合は true
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#containsKey(java.lang.Object)
     */
    public boolean containsKey(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        return groupMap.containsKey(key);
    }

    /**
     * グループが、キーのマッピングしているコレクションに指定された値を 1 つ以上含む場合に true を返します。
     * <p>
     * つまり、キーのマッピングしているコレクションに、(value==null ? v==null : value.equals(v)) となる値 v が 1 つ以上ある場合にだけ true を返します。
     * </p>
     * 
     * @param key
     *            マッピングされているか判定されるキー
     * @param value
     *            キーをマッピングしているコレクションの内に存在するか判定される値
     * @return グループが指定のキーにマッピングされるコレクションの内に値が存在する場合は true
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#containsValue(java.lang.Object, java.lang.Object)
     */
    public boolean containsValue(Object key, Object value) {
        if (key == null) {
            throw new NullPointerException();
        }
        Collection c = getEntryValues(key);
        return (c != null) ? c.contains(value) : false;
    }

    /**
     * グループに含まれているキーのセットビューを返します。
     * <p>
     * セットはグループと連動しているので、グループに対する変更はセットに反映され、 また、セットに対する変更はグループに反映されます。 セットに対する反復の処理中にグループが変更された場合は、反復の結果は保証されません。
     * セットは、Iterator.remove、Set.remove、removeAll、retainAll、および clear の各オペレーションを使って グループから対応するマッピングを削除する要素削除処理をサポートします。
     * add オペレーションと addAll オペレーションは、セットではサポートされていません。
     * </p>
     * 
     * @return グループに含まれているキーのセットビュー
     * @see shohaku.core.collections.Group#keySet()
     */
    public Set keySet() {
        return groupMap.keySet();
    }

    /** グループに含まれているグループエントリのセットビュー。 */
    private transient Set entrySet = null;

    /**
     * グループに含まれているグループエントリのセットビューを返します。
     * <p>
     * セットはグループと連動しているので、グループに対する変更はセットに反映され、 また、セットに対する変更はグループに反映されます。 セットに対する反復の処理中にグループが変更された場合は、反復の結果は保証されません。
     * セットは、Iterator.remove、Set.remove、removeAll、retainAll、および clear の各オペレーションを使って グループから対応するマッピングを削除する要素削除処理をサポートします。
     * add オペレーションと addAll オペレーションは、セットではサポートされていません。
     * </p>
     * 
     * @return グループに含まれているグループエントリのセットビュー
     * @see shohaku.core.collections.Group#entrySet()
     */
    public Set entrySet() {
        if (entrySet == null) {
            entrySet = new EntrySet(this);
        }
        return entrySet;
    }

    /*
     * グループに含まれているグループエントリのセットビュー。
     */
    private static class EntrySet extends AbstractSet {

        /** セットビューの基となるグループ。 */
        private final AbstractMapGroup group;

        /**
         * セットビューの基となるグループを格納して初期化します。
         * 
         * @param g
         *            セットビューの基となるグループ
         */
        public EntrySet(AbstractMapGroup g) {
            this.group = g;
        }

        /**
         * 反復子を返却します。
         * 
         * @return 反復子
         */
        public Iterator iterator() {
            return group.iterator();
        }

        /**
         * 同値のエントリが含まれるか検証します。
         * 
         * @param o
         *            検証するエントリ
         * @return 含まれる場合True
         */
        public boolean contains(Object o) {
            if (!(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            return group.containsEntry(e);
        }

        /**
         * 同値のエントリを削除します。
         * 
         * @param o
         *            削除するエントリ
         * @return 削除された場合True
         */
        public boolean remove(Object o) {
            if (!(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            return group.removeEntry(e);
        }

        /**
         * エントリ数を返します。
         * 
         * @return エントリ数
         */
        public int size() {
            return group.size();
        }

        /**
         * 全てのエントリをクリアします。
         */
        public void clear() {
            group.clear();
        }
    }

    /**
     * グループ内のキーと値の集合の数を返します。 グループに Integer.MAX_VALUE より多くの要素がある場合は、Integer.MAX_VALUE を返します。
     * 
     * @return マッピングの数
     * @see shohaku.core.collections.Group#size()
     */
    public int size() {
        return groupMap.size();
    }

    /**
     * グループ内のキーとマッピングする値のの要素数を返します。 この値に Integer.MAX_VALUE より多くの要素がある場合は、Integer.MAX_VALUE を返します。
     * 
     * @param key
     *            キー
     * @return キーとマッピングする値のの要素数
     * @throws NullPointerException
     *             指定されたキーがNullの場合
     * @see shohaku.core.collections.Group#size(java.lang.Object)
     */
    public int size(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        Collection c = getEntryValues(key);
        return (c != null) ? c.size() : 0;
    }

    /**
     * グループのハッシュコード値を返します。
     * <p>
     * グループのハッシュコードは、グループが保有する全てのグループエントリの hashCode の合計になるように定義されます。 これにより、Object.hashCode の一般規約によって要求されるように、 任意の 2
     * つのグループ t1 と t2 で t1.equals(t2) であれば、t1.hashCode()==t2.hashCode() となることが保証されます。
     * </p>
     * 
     * @return グループのハッシュコード値
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        int h = 0;
        Iterator i = iterator();
        while (i.hasNext()) {
            h += i.next().hashCode();
        }
        return h;
    }

    /**
     * 定されたオブジェクトがこのグループと等しいかどうかを比較します。
     * <p>
     * 指定されたオブジェクトもグループであり、2 つの Group が同じグループエントリを表している場合は true を返します。 つまり、t1.entrySet().equals(t2.entrySet()) である場合、2
     * つのグループ t1 と t2 は同じグループエントリを表します。 これにより、Group インタフェースの実装が異なる場合でも、equals メソッドが正しく動作することが保証されます。
     * </p>
     * 
     * @param o
     *            グループと等しいかどうかを比較するオブジェクト
     * @return 指定されたオブジェクトがグループと等しい場合は true
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }

        if (!(o instanceof Group)) {
            return false;
        }
        Group t = (Group) o;
        if (t.size() != size()) {
            return false;
        }
        return (entrySet().equals(t.entrySet()));
    }

    /**
     * グループの文字列表現を返却します。
     * 
     * @return グループの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("{");

        Iterator i = iterator();
        boolean hasNext = i.hasNext();
        while (hasNext) {
            Entry e = (Entry) (i.next());
            Object key = e.getKey();
            Object values = e.getValues();
            buf.append((key == this ? "(this Group)" : key));
            buf.append('=');
            buf.append((values == this ? "(this Group)" : values));
            hasNext = i.hasNext();
            if (hasNext) {
                buf.append(", ");
            }
        }

        buf.append("}");
        return buf.toString();
    }

    /*
     * protected
     */

    /**
     * グループの内に指定のエントリと同値のエントリを含む場合に true を返します。
     * 
     * @param entry
     *            存在するか検証するグループエントリ
     * @return グループの内に指定のエントリと同値のエントリが存在する場合は true
     */
    protected boolean containsEntry(Entry entry) {
        if (entry == null) {
            throw new NullPointerException();
        }
        return groupMap.containsValue(entry);
    }

    /**
     * グループの内に指定のエントリと同値のエントリを削除します。
     * 
     * @param entry
     *            削除するグループエントリ
     * @return エントリが実際に削除されたが場合は true
     */
    protected boolean removeEntry(Entry entry) {
        if (entry == null) {
            throw new NullPointerException();
        }
        Iterator i = iterator();
        while (i.hasNext()) {
            Entry e = (Entry) (i.next());
            if (e.equals(entry)) {
                i.remove();
                return true;
            }
        }
        return false;
    }

    /**
     * 指定されたキーのマッピングするグループエントリを返します。 <br>
     * グループがこのキーのマッピングを保持していない場合はnull を返します。
     * 
     * @param key
     *            関連づけられている識別キー
     * @return 指定のキーをマッピングするグループエントリ
     */
    protected Entry getEntry(Object key) {
        if (key == null) {
            throw new NullPointerException();
        }
        return (Entry) groupMap.get(key);
    }

    /**
     * 指定されたキーのマッピングするコレクションを返します。 グループがこのキーのマッピングを保持していない場合は null を返します。
     * 
     * @param key
     *            関連づけられている識別キー
     * @return 指定のキーをマッピングするコレクション
     */
    protected Collection getEntryValues(Object key) {
        Entry e = getEntry(key);
        if (e == null) {
            return null;
        } else {
            return e.getValues();
        }
    }

    /**
     * 指定されたキーに対するコレクションを追加します (任意のオペレーション)。 <br>
     * グループに既にこのキーに対するマッピングがある場合何も行われません。 <br>
     * グループに既にこのキーに対するマッピングがある場合 true を返します。
     * 
     * @param key
     *            指定されるコレクションが関連付けられるキー
     * @return 既にこのキーに対するマッピングがある場合 true
     */
    protected boolean addEntry(Object key) {
        boolean ret = true;
        Entry e = getEntry(key);
        if (e == null) {
            e = newEntry(key);
            groupMap.put(key, e);
            ret = false;
        }
        return ret;
    }

    /**
     * 指定されたキーに対するコレクションを削除します。 <br>
     * 実際に削除された場合削除されたコレクション、削除されなかった場合は <code>null</code> を返します。
     * 
     * @param key
     *            新規登録するキー
     * @return 実際に削除された場合削除されたコレクション、削除されなかった場合は <code>null</code>
     */
    protected Collection removeEntryByKey(Object key) {
        Entry e = (Entry) groupMap.remove(key);
        if (e == null) {
            return null;
        } else {
            return e.getValues();
        }
    }

    /**
     * 指定されたキーに対するコレクションの指定された値を削除します。 <br>
     * 実際に削除された場合 <code>true</code> を返します。
     * 
     * @param key
     *            新規登録するキー
     * @param value
     *            指定されるキーに関連付けられる値
     * @return 既にこのキーに対するマッピングがある場合 <code>true</code>
     */
    protected boolean removeValue(Object key, Object value) {
        Collection c = getEntryValues(key);
        return (c != null) ? c.remove(value) : false;
    }

    /**
     * 指定された値を指定されたキーに対応するコレクションに追加します。 <br>
     * グループにこのキーに対するマッピングが存在しない場合は新規に追加されます。 <br>
     * グループに既にこのキーに対するマッピングがある場合 <code>true</code> を返します。
     * 
     * @param key
     *            指定される値が関連付けられるキー
     * @param value
     *            指定されるキーに関連付けられる値
     * @return 既にこのキーに対するマッピングがある場合 <code>true</code>
     */
    protected boolean addEntryValue(Object key, Object value) {
        boolean ret = true;
        Entry e = getEntry(key);
        if (e == null) {
            e = newEntry(key);
            groupMap.put(key, e);
            ret = false;
        }
        e.getValues().add(value);
        return ret;
    }

    /**
     * 指定されたコレクションの全要素をキーに対応するコレクションに追加します。 <br>
     * グループにこのキーに対するマッピングが存在しない場合は新規に追加されます。 <br>
     * グループに既にこのキーに対するマッピングがある場合 <code>true</code> を返します。
     * 
     * @param key
     *            指定されるコレクションの全要素が関連付けられるキー
     * @param values
     *            指定されるキーに関連付けられるコレクション
     * @return 既にこのキーに対するマッピングがある場合 <code>true</code>
     */
    protected boolean addEntryValues(Object key, Collection values) {
        boolean ret = true;
        Entry e = getEntry(key);
        if (e == null) {
            e = newEntry(key);
            groupMap.put(key, e);
            ret = false;
        }
        e.getValues().addAll(values);
        return ret;
    }

    /**
     * 指定されたグループのすべてのキーと対応する全値ををこのグループに追加します。
     * 
     * @param g
     *            グループに追加されるグループ
     */
    protected void addAllEntrys(Group g) {
        for (Iterator i = g.iterator(); i.hasNext();) {
            Entry entry = (Entry) i.next();
            addEntryValues(entry.getKey(), entry.getValues());
        }
    }

    /**
     * 新規にグループエントリを生成して返します。
     * 
     * @param key
     *            新規登録するキー
     * @return 新規のグループエントリ
     */
    protected Entry newEntry(Object key) {
        return newEntry(key, newEntryCollection());
    }

    /**
     * グループエントリの反復子を生成して返します。
     * 
     * @return グループエントリの反復子
     */
    protected Iterator newEntryIterator() {
        return groupMap.values().iterator();
    }

    /**
     * 実装が提供するグループエントリを生成して返します。 <br>
     * 実装クラスでエントリを実装する場合、オーバライドしてください。
     * 
     * @param key
     *            新規登録するキー
     * @param values
     *            新規登録するコレクション
     * @return 実装が提供する新規のコレクション
     */
    protected Entry newEntry(Object key, Collection values) {
        return new SimpleEntry(key, values);
    }

    /*
     * abstract public
     */

    /**
     * グループのクローンオブジェクトを生成して返却します。
     * 
     * @return グループのクローンオブジェクト
     * @see java.lang.Object#clone()
     */
    abstract public Object clone();

    /*
     * abstract protected
     */

    /**
     * 実装が提供するキーに関連づけられるコレクションを生成して返します。
     * 
     * @return 実装が提供する新規のコレクション
     */
    abstract protected Collection newEntryCollection();

}
