package shohaku.core.collections;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;

/**
 * 媒介変数のマッピングを格納するデータ構造を定義します。 <br>
 * このデータは名前と値の対のデータの双方向リンクリスト構造を持ちます。
 */
public class Parameters implements Cloneable {
    
    /** 空のパラメータリスト。 */
    public static final Parameters EMPTY_PARAMETERS = new Parameters();

    /* エントリを保存するバケット。 */
    private final Entry[] table;

    /* 双方向リンクの末尾エントリ。 */
    private final Entry endEntry;

    /* パラメータリストのエントリ数。 */
    private final int size;

    /* パラメータリストのハッシュ値。 */
    private final int hashCode;

    /**
     * 空のパラメータリストを初期化します。
     */
    public Parameters() {
        this(new String[0]);
    }

    /**
     * 指定された名前と <code>null</code> の値でパラメータリストを初期化します。
     * 
     * @param names
     *            パラメータ名
     */
    public Parameters(final String[] names) {
        this(new KayValueIterator() {
            private int _index = -1;

            public void remove() {
                throw new UnsupportedOperationException();
            }

            public Object next() {
                if (hasNext()) {
                    _index++;
                } else {
                    throw new NoSuchElementException();
                }
                return getKey();
            }

            public boolean hasNext() {
                return ((_index + 1) < names.length);
            }

            public Object getKey() {
                return names[_index];
            }

            public Object getValue() {
                return null;
            }

            public Object setValue(Object value) {
                throw new UnsupportedOperationException();
            }

            public int size() {
                return names.length;
            }
        });
    }

    /**
     * 指定された配列を名前と値を交互に持つデータとしてパラメータリストを初期化します。
     * 
     * @param nameValue
     *            名前と値を交互に持つ配列
     */
    public Parameters(Object[] nameValue) {
        this(KayValueIteratorUtils.segmentKayValueIterator(nameValue));
    }

    /**
     * 指定されたリストを名前と値を交互に持つデータとしてパラメータリストを初期化します。
     * 
     * @param nameValue
     *            名前と値を交互に持つリスト
     */
    public Parameters(List nameValue) {
        this(KayValueIteratorUtils.segmentKayValueIterator(nameValue));
    }

    /**
     * 引数のマップのキーを名前として、そのキーの値を持つパラメータリストを初期化します。
     * 
     * @param prams
     *            基となるマップ
     */
    public Parameters(Map prams) {
        this(KayValueIteratorUtils.asKayValueIterator(prams));
    }

    /**
     * 引数のパラメータリストと同エントリを持つパラメータリストを初期化します。
     * 
     * @param prams
     *            基となるパラメータリスト
     */
    public Parameters(Parameters prams) {
        this(prams.kayValueIterator());
    }

    /**
     * 引数のキーと値の対とする反復子と同エントリを持つパラメータリストを初期化します。
     * 
     * @param i
     *            基となるパラメータリスト
     */
    public Parameters(KayValueIterator i) {

        this.size = i.size();
        this.table = new Entry[capacity(this.size)];

        Entry _entry = null;
        Entry _beginEntry = null;
        while (i.hasNext()) {
            i.next();

            String name = i.getKey().toString();// name is null throw NullException
            Object value = i.getValue();

            int hashCode = hash(name);
            int index = indexFor(hashCode, table.length);
            Entry newEntry = new Entry(name, value);
            Entry oldEntry = putEntry(index, newEntry);
            if (oldEntry != null) {
                if (oldEntry == _entry) {
                    _entry = oldEntry.before;
                }
                if (oldEntry == _beginEntry) {
                    _beginEntry = oldEntry.after;
                }
            }
            if (_entry != null) {
                _entry.after = newEntry;
                newEntry.before = _entry;
            } else {
                _beginEntry = newEntry;
            }
            _entry = newEntry;

        }
        if (null != _entry) {
            this.endEntry = _entry;
            this.endEntry.after = _beginEntry;
            _beginEntry.before = this.endEntry;
        } else {
            this.endEntry = null;
        }
        this.hashCode = _hashCode(this);

    }

    /**
     * パラメータリストのエントリ数を返します。
     * 
     * @return パラメータリストのエントリ数
     */
    public int size() {
        return this.size;
    }

    /**
     * パラメータリストの反復子を生成して返します。 <br>
     * 反復子の要素は <code>Parameters.Entry</code> 型と為ります。
     * 
     * @return パラメータリストの反復子
     */
    public Iterator iterator() {
        return new ParametersIterator();
    }

    /* パラメータリストの反復子。 */
    private class ParametersIterator implements Iterator {

        /* 走査中のインデックス。 */
        private int _index = -1;

        /* 走査中のエントリ。 */
        private Entry e = endEntry;

        /* 反復する次の要素が有る場合 true を返します。 */
        public boolean hasNext() {
            return (e != null && (_index + 1) < size);
        }

        /* 反復する次の要素(Parameters.Entry)を返します。 */
        public Object next() {
            if (hasNext()) {
                _index++;
                e = e.after;
            } else {
                throw new NoSuchElementException();
            }
            return e;
        }

        /* <code> UnsupportedOperationException </code> を発生させます。 */
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * パラメータリストの反復子を生成して返します。 <br>
     * 反復子の要素は <code>Parameters.Entry</code> 型と為ります。
     * 
     * @return パラメータリストの反復子
     */
    public KayValueIterator kayValueIterator() {
        return new ParametersKayValueIterator();
    }

    /* パラメータリストの反復子。 */
    private class ParametersKayValueIterator implements KayValueIterator {

        /* 走査中のエントリ。 */
        private Entry e = endEntry;

        /* 走査中のインデックス。 */
        private int _index = -1;

        /* <code> UnsupportedOperationException </code> を発生させます。 */
        public void remove() {
            throw new UnsupportedOperationException();
        }

        /* 反復する次の要素が有る場合 true を返します。 */
        public boolean hasNext() {
            return (e != null && (_index + 1) < size);
        }

        /* 反復する次の要素を返します。 */
        public Object next() {
            if (hasNext()) {
                _index++;
                e = e.after;
            } else {
                throw new NoSuchElementException();
            }
            return getKey();
        }

        public Object getKey() {
            check();
            return e.getName();
        }

        public Object getValue() {
            check();
            return e.getValue();
        }

        public Object setValue(Object value) {
            check();
            return e.setValue(value);
        }

        public int size() {
            return size;
        }

        private void check() {
            if (_index == -1) {
                throw new IllegalStateException();
            }
            if (e == null) {
                throw new NoSuchElementException();
            }
        }
    }

    /**
     * パラメータが空の場合 <code>true</code> を返します。
     * 
     * @return パラメータが空の場合 <code>true</code>
     */
    public boolean isEmpty() {
        return (endEntry == null);
    }

    /**
     * インデックスのパラメータ名を返します。
     * 
     * @param index
     *            インデックス
     * @return インデックスのパラメータ名
     */
    public String getName(int index) {
        return getEntry(index).getName();
    }

    /**
     * パラメータ名のインデックスを返します。
     * 
     * @param name
     *            パラメータ名
     * @return パラメータ名のインデックス
     */
    public int getIndex(String name) {
        if (name == null) {
            //err
        }
        if (!isEmpty()) {
            Entry e = endEntry.after;
            for (int i = 0; i < size; i++) {
                if (name.equals(e.getName())) {
                    return i;
                }
                e = e.after;
            }
        }
        return -1;
    }

    /**
     * パラメータ名のパラメータ値を返します。
     * 
     * @param name
     *            パラメータ名
     * @return パラメータ値
     */
    public Object getValue(String name) {
        Entry e = getEntry(name);
        if (null != e) {
            return e.getValue();
        }
        return null;
    }
    
    
    /**
     * パラメータ名のパラメータ値を返します。<br>
     * 指定された名前が存在しない場合は、 <code>defaultValue</code> が返されます。
     * 
     * @param name
     *           パラメータ名
     * @param defaultValue
     *            パラメータが存在しない場合に返却される値
     * @return 指定された名前のパラメータ値又は <code>defaultValue</code>
     */
    public Object getValue(String name, String defaultValue) {
        Entry e = getEntry(name);
        if (null != e) {
            return e.getValue();
        }
        return defaultValue;
    }

    /**
     * インデックスのパラメータ値を返します。
     * 
     * @param index
     *            インデックス
     * @return インデックスのパラメータ値
     */
    public Object getValue(int index) {
        return getEntry(index).getValue();
    }

    /**
     * パラメータ名のパラメータ値を格納します。 <br>
     * 既存のパラメータ値が存在する場合その値を返し、存在しない場合 <code>null</code> を返します。
     * 
     * @param name
     *            パラメータ名
     * @param value
     *            新たなパラメータ値
     * @return 既存のパラメータ値が存在する場合その値、存在しない場合 <code>null</code>
     */
    public Object setValue(String name, Object value) {
        if (name == null) {
            //err
        }
        Entry e = getEntry(name);
        if (null != e) {
            return swapValue(e, value);
        }
        return null;
    }

    /**
     * インデックスのパラメータ値を格納します。 <br>
     * 既存のパラメータ値が存在する場合その値を返し、存在しない場合 <code>null</code> を返します。
     * 
     * @param index
     *            インデックス
     * @param value
     *            新たなパラメータ値
     * @return 既存のパラメータ値が存在する場合その値、存在しない場合 <code>null</code>
     */
    public Object setValue(int index, Object value) {
        Entry e = getEntry(index);
        if (null != e) {
            return swapValue(e, value);
        }
        return null;
    }

    /**
     * パラメータ名のエントリを返します。
     * 
     * @param name
     *            パラメータ名
     * @return パラメータ名のエントリ
     */
    public Entry getEntry(String name) {
        if (name == null) {
            //err
        }
        int hashCode = hash(name);
        int index = indexFor(hashCode, table.length);
        for (Entry e = table[index]; e != null; e = e.next) {
            if ((e.hashCode() == hashCode) && name.equals(e.getName())) {
                return e;
            }
        }
        return null;
    }

    /**
     * インデックスのエントリを返します。
     * 
     * @param index
     *            インデックス
     * @return インデックスのエントリ
     */
    public Entry getEntry(int index) {
        if (0 > index || index >= size) {
            throw new IndexOutOfBoundsException("Index:" + index + ", Size:" + size);
        }
        if (index < (size >> 1)) {
            Entry e = endEntry.after;//start entry
            for (int i = 0; i < index; i++) {
                e = e.after;
            }
            return e;
        } else {
            Entry e = endEntry;
            for (int i = (size - 1); i > index; i--) {
                e = e.before;
            }
            return e;
        }
    }

    /**
     * 引数の名前のパラメータエントリが存在するか検証します。 <br>
     * パラメータエントリが存在する場合のみ <code>true</code> を返します。
     * 
     * @param name
     *            検証する名前
     * @return パラメータエントリが存在する場合のみ <code>true</code>
     */
    public boolean containsName(String name) {
        return (-1 < getIndex(name));
    }

    /**
     * このオブジェクトのハッシュ値を返します。 <br>
     * 全てのエントリのハッシュ値の合計です。
     * 
     * @return このオブジェクトのハッシュ値
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        return this.hashCode;
    }

    /**
     * 引数が同値であるか検証します。 <br>
     * <code>Parameters</code> 型でエントリが全て同一の場合のみ <code>true</code> を返します。
     * 
     * @param o
     *            比較先のオブジェクト
     * @return <code>Parameters</code> 型でエントリが全て同一の場合のみ <code>true</code>
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Parameters)) {
            return false;
        }
        Parameters p = (Parameters) o;
        if (size != p.size) {
            return false;
        }
        if (!isEmpty()) {
            Entry e = endEntry.after;
            Entry e2 = p.endEntry.after;
            for (int i = 0; i < size; i++) {
                if (!e.equals(e2)) {
                    return false;
                }
                e = e.after;
                e2 = e2.after;
            }
        }
        return true;
    }

    /**
     * このオブジェクトの文字列表現を返します。 <br>
     * <code>{name=value, name=value, ...}</code>
     * 
     * @return このオブジェクトの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append("{");

        Iterator i = iterator();
        boolean hasNext = i.hasNext();
        while (hasNext) {
            Entry e = (Entry) (i.next());
            buf.append(e.getName());
            buf.append('=');
            Object value = e.getValue();
            buf.append((value == this ? "(this Parameters)" : value));
            hasNext = i.hasNext();
            if (hasNext) {
                buf.append(", ");
            }
        }

        buf.append("}");
        return buf.toString();
    }

    /**
     * クローンを生成して返します。
     * 
     * @return このオブジェクトのクローン
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        return new Parameters(this);
    }

    /* インデックス位置にエントリを追加します。上書きされた場合そのエントリを返します。 */
    private Entry putEntry(int index, Entry newEntry) {
        Entry oldEntry = null;
        Entry e = table[index];
        if (e == null) {
            table[index] = newEntry;
        } else {
            while (e != null) {
                if (null == e.next) {
                    e.next = newEntry;
                    break;
                }
                if (eq(e.next, newEntry)) {
                    oldEntry = e.next;
                    if (null != oldEntry.next) {
                        newEntry.next = oldEntry.next;
                    }
                    e.next = newEntry;
                    removeLinkList(oldEntry);
                    break;
                }
                e = e.next;
            }
        }
        return oldEntry;
    }

    /*
     * static
     */

    /*
     * ハッシュ値とバケット容量に関しては今後に最適化を行う。
     */

    /* */
    private void removeLinkList(Entry oldEntry) {
        if (null != oldEntry.after) {
            oldEntry.after.before = oldEntry.before;
        }
        if (null != oldEntry.before) {
            oldEntry.before.after = oldEntry.after;
        }
    }

    /* */
    private static boolean eq(Entry entry, Entry entry2) {
        return entry.name.equals(entry2.name);
    }

    /* バケットの容量を返します。 */
    private static int capacity(int size) {
        return size * 2 + 1;
    }

    /* ハッシュ値とバケット容量からインデックスを算出して返します。 */
    private static int indexFor(int hashCode, int length) {
        return hashCode & (length - 1);
    }

    /* 名前からハッシュ値を算出して返します。 */
    private static int hash(String name) {
        int h = name.hashCode();
        h += ~(h << 9);
        h ^= (h >>> 14);
        h += (h << 4);
        h ^= (h >>> 10);
        return h;
    }

    /* エントリ値を入れ替えて古い値を返します。 */
    private static Object swapValue(Entry entry, Object value) {
        Object old = entry.getValue();
        entry.setValue(value);
        return old;
    }

    /* パラメータリストのハッシュ値を算出して返します。 */
    private static int _hashCode(Parameters params) {
        int hashCode = 0;
        if (!params.isEmpty()) {
            Entry e = params.endEntry.after;
            for (int i = 0; i < params.size; i++) {
                hashCode += e.hashCode();
                e = e.after;
            }
        }
        return hashCode;
    }

    /*
     * class
     */

    /**
     * パラメータのエントリを定義します。 <br>
     * これはパラメータリストの要素となります。
     */
    public static class Entry {

        /* エントリ名。 */
        private final String name;

        /* このエントリの値。 */
        private Object value;

        /* このエントリのハッシュ値。 */
        private final int hashCode;

        /* バケットの次のエントリ。 */
        private Entry next;

        /* 双方向リンクの後方エントリ。 */
        private Entry before;

        /* 双方向リンクの前方エントリ。 */
        private Entry after;

        /**
         * エントリ名とエントリ値を設定して初期化します。
         * 
         * @param name
         *            エントリ名
         * @param value
         *            エントリ値
         */
        public Entry(String name, Object value) {
            this.name = name;
            this.hashCode = hash(name);
            this.value = value;
        }

        /**
         * エントリ名を返します。
         * 
         * @return エントリ名
         */
        public String getName() {
            return this.name;
        }

        /**
         * エントリの値を返します。
         * 
         * @return エントリの値
         */
        public Object getValue() {
            return this.value;
        }

        /**
         * エントリの値を格納します。 <br>
         * 既存の値が存在する場合その値を返し、存在しない場合 <code>null</code> を返します。
         * 
         * @param value
         *            新たなエントリの値
         * @return 既存の値が存在する場合その値、存在しない場合 <code>null</code>
         */
        public Object setValue(Object value) {
            Object oldValue = value;
            this.value = value;
            return oldValue;
        }

        /**
         * このエントリのハッシュ値を返します。
         * 
         * @return このエントリのハッシュ値
         * @see java.lang.Object#hashCode()
         */
        public int hashCode() {
            return hashCode;
        }

        /**
         * 引数が同値であるか検証します。 <br>
         * <code>Parameters.Entry</code> 型で名前と値が同一の場合のみ <code>true</code> を返します。
         * 
         * @param o
         *            比較先のオブジェクト
         * @return <code>Parameters.Entry</code> 型で名前と値が同一の場合のみ <code>true</code>
         * @see java.lang.Object#equals(java.lang.Object)
         */
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (!(o instanceof Entry)) {
                return false;
            }
            Entry e = (Entry) o;
            return (name.equals(e.name)) && ((value != null) ? value.equals(e.value) : e.value == null);
        }

        /**
         * このエントリの文字列表現を返します。 <br>
         * <code>name=value</code>
         * 
         * @return このエントリの文字列表現
         * @see java.lang.Object#toString()
         */
        public String toString() {
            return this.name + "=" + this.value;
        }
    }
}